/* -*- mode: C; c-file-style: "gnu"; indent-tabs-mode: nil; -*-
 *
 * Copyright 2025 GNOME Foundation, Inc.
 *
 * SPDX-License-Identifier: LGPL-2.1-or-later
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * Authors:
 *  - Philip Withnall <pwithnall@gnome.org>
 */

#pragma once

#include <glib.h>
#include <glib-object.h>

G_BEGIN_DECLS

/**
 * MctOperationCounter:
 *
 * Helper struct used to increment a counter when it’s created and decrement
 * when it’s destroyed.
 *
 * This helps tie a counter to the lifecycles of the operations being counted by
 * it. It is designed to be used inline within another struct.
 *
 * Since: 0.14.0
 */
typedef struct
{
 /*< private >*/
  unsigned int *counter;  /* (not nullable) */
  GObject *object;  /* (owned) (nullable) */
  GParamSpec *pspec;  /* (not owned) (nullable) */
} MctOperationCounter;

/**
 * mct_operation_counter_init_and_hold:
 * @self: an operation counter struct, already allocated within another struct
 * @counter: (inout): counter variable to increment
 * @object: (transfer none) (nullable): object to notify when counter changes
 * @pspec: (transfer none) (nullable): param spec within @object to notify when
 *   counter changes
 *
 * Inits the [struct@Mct.OperationCounter] and increments @counter.
 *
 * If @object and @pspec are provided, `g_object_notify_by_pspec()` will be
 * called on them to notify of the change in value to @counter.
 *
 * Since: 0.14.0
 */
static inline void
mct_operation_counter_init_and_hold (MctOperationCounter *self,
                                     unsigned int        *counter,
                                     GObject             *object,
                                     GParamSpec          *pspec)
{
  self->counter = counter;
  self->object = (object != NULL && pspec != NULL) ? g_object_ref (object) : NULL;
  self->pspec = (object != NULL && pspec != NULL) ? pspec : NULL;

  g_assert (*(self->counter) < UINT_MAX);
  *(self->counter) = *(self->counter) + 1;
  if (self->object != NULL)
    g_object_notify_by_pspec (self->object, self->pspec);
}

/**
 * mct_operation_counter_init_and_hold:
 * @self: an operation counter struct, already allocated within another struct
 *
 * Decrements the counter and clears the [struct@Mct.OperationCounter].
 *
 * The counter variable is the one provided when @self was initialised.
 *
 * If @object and @pspec were provided when @self was initialised,
 * `g_object_notify_by_pspec()` will be called on them to notify of the change
 * in value to @counter.
 *
 * Since: 0.14.0
 */
static inline void
mct_operation_counter_release_and_clear (MctOperationCounter *self)
{
  g_assert (*(self->counter) > 0);
  *(self->counter) = *(self->counter) - 1;
  if (self->object != NULL)
    g_object_notify_by_pspec (self->object, self->pspec);

  g_clear_object (&self->object);
  self->pspec = NULL;
  self->counter = NULL;
}

G_END_DECLS
