/*
 * Copyright (C) 2014-2026 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#pragma once

#include "src/datovka_shared/isds/message_interface.h"

class QByteArray; /* Forward declaration. */
class QString; /* Forward declaration. */

namespace Isds {

	/*!
	 * @brief ZFO type.
	 */
	enum LoadType {
		LT_ANY, /*!< All types. */
		LT_MESSAGE, /*!< Data message. */
		LT_DELIVERY /*!< Delivery info. */
	};

	/*!
	 * @brief Guess type of data.
	 *
	 * @param[in] rawMsgData Raw ZFO data.
	 * @return ZFO data type, RT_UNKNOWN on any error.
	 */
	enum Type::RawType guessDataRawType(const QByteArray &rawMsgData);

	/*!
	 * @brief Guess type of data in file.
	 *
	 * @param[in] fName File name.
	 * @return ZFO data type, RT_UNKNOWN on any error.
	 */
	enum Type::RawType guessFileRawType(const QString &fName);

	/*!
	 * @brief Guess type of data.
	 *
	 * @param[in] rawMsgData Raw ZFO data.
	 * @param[out] attachTotal Total of attachment sizes.
	 * @return ZFO data type, RT_UNKNOWN on any error.
	 */
	enum Type::RawType guessDataRawTypeAttachTotal(
	    const QByteArray &rawMsgData, qint64 &attachTotal);

	/*!
	 * @brief Guess type of data in file.
	 *
	 * @param[in] fName File name.
	 * @param[out] attachTotal Total of attachment sizes.
	 * @return ZFO data type, RT_UNKNOWN on any error.
	 */
	enum Type::RawType guessFileRawTypeAttachTotal(const QString &fName,
	    qint64 &attachTotal);

	/*!
	 * @brief Creates a message from supplied raw CMS data.
	 *
	 * @param[in] rawMsgData Raw message data.
	 * @param[in] zfoType Message or delivery info.
	 * @return Return null message on error.
	 */
	Message messageFromData(const QByteArray &rawMsgData,
	    enum LoadType zfoType);

	/*!
	 * @brief Try loading ZFO data.
	 *
	 * @param[in] rawMsgData Raw ZFO data.
	 * @param[out] zfoType Detected ZFO file type.
	 * @param[out] ok Set to true on success.
	 * @return Return null message on error.
	 */
	Message parseZfoData(const QByteArray &rawMsgData,
	    enum LoadType &zfoType, bool *ok = Q_NULLPTR);

	/*!
	 * @brief Create a message from supplied ZFO file.
	 *
	 * @param[in] fName File name.
	 * @param[in] zfoType Message or delivery info.
	 * @return Return null message on error.
	 */
	Message messageFromFile(const QString &fName, enum LoadType zfoType);

	/*!
	 * @brief Try loading ZFO file.
	 *
	 * @param[in] fName File name.
	 * @param[out] zfoType Detected ZFO file type.
	 * @param[out] ok Set to true on success.
	 * @return Return null message on error.
	 */
	Message parseZfoFile(const QString &fName,
	    enum LoadType &zfoType, bool *ok = Q_NULLPTR);

	/*!
	 * @brief Compute message hash.
	 *
	 * @param[in] message Message.
	 * @param[in] hashAlg Hash algorithm to be used.
	 * @retunr Computed hash.
	 */
	Hash computeDmHash(const Message &message,
	    enum Type::HashAlg hashAlg);

}
