/*
 * Copyright (C) 2014 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include <math.h>
#include "util/rounding.h" /* for _mesa_roundeven */
#include "util/half_float.h"
#include "util/double.h"
#include "util/softfloat.h"
#include "util/bfloat.h"
#include "util/float8.h"
#include "util/bigmath.h"
#include "util/format/format_utils.h"
#include "util/format_r11g11b10f.h"
#include "util/u_math.h"
#include "util/u_overflow.h"
#include "nir_constant_expressions.h"
#include "nir.h"

/**
 * Checks if the provided value is a denorm and flushes it to zero.
 */
static void
constant_denorm_flush_to_zero(nir_const_value *value, unsigned bit_size)
{
    switch(bit_size) {
    case 64:
        if (0 == (value->u64 & 0x7ff0000000000000))
            value->u64 &= 0x8000000000000000;
        break;
    case 32:
        if (0 == (value->u32 & 0x7f800000))
            value->u32 &= 0x80000000;
        break;
    case 16:
        if (0 == (value->u16 & 0x7c00))
            value->u16 &= 0x8000;
    }
}

static double
get_float_source(nir_const_value value, unsigned execution_mode, unsigned bit_size)
{
    if (nir_is_denorm_flush_to_zero(execution_mode, bit_size))
        constant_denorm_flush_to_zero(&value, bit_size);

    return nir_const_value_as_float(value, bit_size);
}

/**
 * Evaluate one component of packSnorm4x8.
 */
static uint8_t
pack_snorm_1x8(float x)
{
    /* From section 8.4 of the GLSL 4.30 spec:
     *
     *    packSnorm4x8
     *    ------------
     *    The conversion for component c of v to fixed point is done as
     *    follows:
     *
     *      packSnorm4x8: round(clamp(c, -1, +1) * 127.0)
     *
     * We must first cast the float to an int, because casting a negative
     * float to a uint is undefined.
     */
   return (uint8_t) (int)
          _mesa_roundevenf(CLAMP(x, -1.0f, +1.0f) * 127.0f);
}

/**
 * Evaluate one component of packSnorm2x16.
 */
static uint16_t
pack_snorm_1x16(float x)
{
    /* From section 8.4 of the GLSL ES 3.00 spec:
     *
     *    packSnorm2x16
     *    -------------
     *    The conversion for component c of v to fixed point is done as
     *    follows:
     *
     *      packSnorm2x16: round(clamp(c, -1, +1) * 32767.0)
     *
     * We must first cast the float to an int, because casting a negative
     * float to a uint is undefined.
     */
   return (uint16_t) (int)
          _mesa_roundevenf(CLAMP(x, -1.0f, +1.0f) * 32767.0f);
}

/**
 * Evaluate one component of unpackSnorm4x8.
 */
static float
unpack_snorm_1x8(uint8_t u)
{
    /* From section 8.4 of the GLSL 4.30 spec:
     *
     *    unpackSnorm4x8
     *    --------------
     *    The conversion for unpacked fixed-point value f to floating point is
     *    done as follows:
     *
     *       unpackSnorm4x8: clamp(f / 127.0, -1, +1)
     */
   return CLAMP((int8_t) u / 127.0f, -1.0f, +1.0f);
}

/**
 * Evaluate one component of unpackSnorm2x16.
 */
static float
unpack_snorm_1x16(uint16_t u)
{
    /* From section 8.4 of the GLSL ES 3.00 spec:
     *
     *    unpackSnorm2x16
     *    ---------------
     *    The conversion for unpacked fixed-point value f to floating point is
     *    done as follows:
     *
     *       unpackSnorm2x16: clamp(f / 32767.0, -1, +1)
     */
   return CLAMP((int16_t) u / 32767.0f, -1.0f, +1.0f);
}

/**
 * Evaluate one component packUnorm4x8.
 */
static uint8_t
pack_unorm_1x8(float x)
{
    /* From section 8.4 of the GLSL 4.30 spec:
     *
     *    packUnorm4x8
     *    ------------
     *    The conversion for component c of v to fixed point is done as
     *    follows:
     *
     *       packUnorm4x8: round(clamp(c, 0, +1) * 255.0)
     */
   return (uint8_t) (int)
          _mesa_roundevenf(CLAMP(x, 0.0f, 1.0f) * 255.0f);
}

/**
 * Evaluate one component packUnorm2x16.
 */
static uint16_t
pack_unorm_1x16(float x)
{
    /* From section 8.4 of the GLSL ES 3.00 spec:
     *
     *    packUnorm2x16
     *    -------------
     *    The conversion for component c of v to fixed point is done as
     *    follows:
     *
     *       packUnorm2x16: round(clamp(c, 0, +1) * 65535.0)
     */
   return (uint16_t) (int)
          _mesa_roundevenf(CLAMP(x, 0.0f, 1.0f) * 65535.0f);
}

/**
 * Evaluate one component of unpackUnorm4x8.
 */
static float
unpack_unorm_1x8(uint8_t u)
{
    /* From section 8.4 of the GLSL 4.30 spec:
     *
     *    unpackUnorm4x8
     *    --------------
     *    The conversion for unpacked fixed-point value f to floating point is
     *    done as follows:
     *
     *       unpackUnorm4x8: f / 255.0
     */
   return (float) u / 255.0f;
}

/**
 * Evaluate one component of unpackUnorm2x16.
 */
static float
unpack_unorm_1x16(uint16_t u)
{
    /* From section 8.4 of the GLSL ES 3.00 spec:
     *
     *    unpackUnorm2x16
     *    ---------------
     *    The conversion for unpacked fixed-point value f to floating point is
     *    done as follows:
     *
     *       unpackUnorm2x16: f / 65535.0
     */
   return (float) u / 65535.0f;
}

/**
 * Evaluate one component of packHalf2x16.
 */
static uint16_t
pack_half_1x16(float x)
{
   return _mesa_float_to_half(x);
}

/**
 * Evaluate one component of packHalf2x16, RTZ mode.
 */
static uint16_t
pack_half_1x16_rtz(float x)
{
   return _mesa_float_to_float16_rtz(x);
}

/**
 * Evaluate one component of unpackHalf2x16.
 */
static float
unpack_half_1x16(uint16_t u, bool ftz)
{
   if (0 == (u & 0x7c00) && ftz)
      u &= 0x8000;
   return _mesa_half_to_float(u);
}

/* Broadcom v3d specific instructions */
/**
 * Packs 2 2x16 floating split into a r11g11b10f:
 *
 * dst[10:0]  = float16_to_float11 (src0[15:0])
 * dst[21:11] = float16_to_float11 (src0[31:16])
 * dst[31:22] = float16_to_float10 (src1[15:0])
 */
static uint32_t pack_32_to_r11g11b10_v3d(const uint32_t src0,
                                         const uint32_t src1)
{
   float rgb[3] = {
      unpack_half_1x16((src0 & 0xffff), false),
      unpack_half_1x16((src0 >> 16), false),
      unpack_half_1x16((src1 & 0xffff), false),
   };

   return float3_to_r11g11b10f(rgb);
}

/**
  * The three methods below are basically wrappers over pack_s/unorm_1x8/1x16,
  * as they receives a uint16_t val instead of a float
  */
static inline uint8_t _mesa_half_to_snorm8(uint16_t val)
{
   return pack_snorm_1x8(_mesa_half_to_float(val));
}

static uint16_t _mesa_float_to_snorm16(uint32_t val)
{
   union fi aux;
   aux.ui = val;
   return pack_snorm_1x16(aux.f);
}

static uint16_t _mesa_float_to_unorm16(uint32_t val)
{
   union fi aux;
   aux.ui = val;
   return pack_unorm_1x16(aux.f);
}

static inline uint32_t float_pack16_v3d(uint32_t f32)
{
   return _mesa_float_to_half(uif(f32));
}

static inline uint32_t float_unpack16_v3d(uint32_t f16)
{
   return fui(_mesa_half_to_float(f16));
}

static inline uint32_t vfpack_v3d(uint32_t a, uint32_t b)
{
   return float_pack16_v3d(b) << 16 | float_pack16_v3d(a);
}

static inline uint32_t vfsat_v3d(uint32_t a)
{
   const uint32_t low = fui(SATURATE(_mesa_half_to_float(a & 0xffff)));
   const uint32_t high = fui(SATURATE(_mesa_half_to_float(a >> 16)));

   return vfpack_v3d(low, high);
}

static inline uint32_t fmul_v3d(uint32_t a, uint32_t b)
{
   return fui(uif(a) * uif(b));
}

static uint32_t vfmul_v3d(uint32_t a, uint32_t b)
{
   const uint32_t low = fmul_v3d(float_unpack16_v3d(a & 0xffff),
                                 float_unpack16_v3d(b & 0xffff));
   const uint32_t high = fmul_v3d(float_unpack16_v3d(a >> 16),
                                  float_unpack16_v3d(b >> 16));

   return vfpack_v3d(low, high);
}

/* Convert 2x16-bit floating point to 2x10-bit unorm */
static uint32_t pack_2x16_to_unorm_2x10(uint32_t src0)
{
   return vfmul_v3d(vfsat_v3d(src0), 0x03ff03ff);
}

/*
 * Convert 2x16-bit floating point to one 2-bit and one
 * 10-bit unorm
 */
static uint32_t pack_2x16_to_unorm_10_2(uint32_t src0)
{
   return vfmul_v3d(vfsat_v3d(src0), 0x000303ff);
}

static uint32_t
msad(uint32_t src0, uint32_t src1, uint32_t src2) {
   uint32_t res = src2;
   for (unsigned i = 0; i < 4; i++) {
      const uint8_t ref = src0 >> (i * 8);
      const uint8_t src = src1 >> (i * 8);
      if (ref != 0)
         res += MAX2(ref, src) - MIN2(ref, src);
   }
   return res;
}

/* Some typed vector structures to make things like src0.y work */
typedef int8_t int1_t;
typedef uint8_t uint1_t;
typedef float float16_t;
typedef float float32_t;
typedef double float64_t;
typedef bool bool1_t;
typedef bool bool8_t;
typedef bool bool16_t;
typedef bool bool32_t;
typedef bool bool64_t;

static inline bool
util_add_check_overflow_int1_t(int1_t a, int1_t b)
{
   return (a & 1 && b & 1);
}
static inline bool
util_add_check_overflow_uint1_t(uint1_t a, int1_t b)
{
   return (a & 1 && b & 1);
}
static inline bool
util_sub_check_overflow_int1_t(int1_t a, int1_t b)
{
   /* int1_t uses 0/-1 convention, so the only
    * overflow case is "0 - (-1)".
    */
   return a == 0 && b != 0;
}

struct float16_vec {
   float16_t x;
   float16_t y;
   float16_t z;
   float16_t w;
   float16_t e;
   float16_t f;
   float16_t g;
   float16_t h;
   float16_t i;
   float16_t j;
   float16_t k;
   float16_t l;
   float16_t m;
   float16_t n;
   float16_t o;
   float16_t p;
};
struct float32_vec {
   float32_t x;
   float32_t y;
   float32_t z;
   float32_t w;
   float32_t e;
   float32_t f;
   float32_t g;
   float32_t h;
   float32_t i;
   float32_t j;
   float32_t k;
   float32_t l;
   float32_t m;
   float32_t n;
   float32_t o;
   float32_t p;
};
struct float64_vec {
   float64_t x;
   float64_t y;
   float64_t z;
   float64_t w;
   float64_t e;
   float64_t f;
   float64_t g;
   float64_t h;
   float64_t i;
   float64_t j;
   float64_t k;
   float64_t l;
   float64_t m;
   float64_t n;
   float64_t o;
   float64_t p;
};
struct int1_vec {
   int1_t x;
   int1_t y;
   int1_t z;
   int1_t w;
   int1_t e;
   int1_t f;
   int1_t g;
   int1_t h;
   int1_t i;
   int1_t j;
   int1_t k;
   int1_t l;
   int1_t m;
   int1_t n;
   int1_t o;
   int1_t p;
};
struct int8_vec {
   int8_t x;
   int8_t y;
   int8_t z;
   int8_t w;
   int8_t e;
   int8_t f;
   int8_t g;
   int8_t h;
   int8_t i;
   int8_t j;
   int8_t k;
   int8_t l;
   int8_t m;
   int8_t n;
   int8_t o;
   int8_t p;
};
struct int16_vec {
   int16_t x;
   int16_t y;
   int16_t z;
   int16_t w;
   int16_t e;
   int16_t f;
   int16_t g;
   int16_t h;
   int16_t i;
   int16_t j;
   int16_t k;
   int16_t l;
   int16_t m;
   int16_t n;
   int16_t o;
   int16_t p;
};
struct int32_vec {
   int32_t x;
   int32_t y;
   int32_t z;
   int32_t w;
   int32_t e;
   int32_t f;
   int32_t g;
   int32_t h;
   int32_t i;
   int32_t j;
   int32_t k;
   int32_t l;
   int32_t m;
   int32_t n;
   int32_t o;
   int32_t p;
};
struct int64_vec {
   int64_t x;
   int64_t y;
   int64_t z;
   int64_t w;
   int64_t e;
   int64_t f;
   int64_t g;
   int64_t h;
   int64_t i;
   int64_t j;
   int64_t k;
   int64_t l;
   int64_t m;
   int64_t n;
   int64_t o;
   int64_t p;
};
struct uint1_vec {
   uint1_t x;
   uint1_t y;
   uint1_t z;
   uint1_t w;
   uint1_t e;
   uint1_t f;
   uint1_t g;
   uint1_t h;
   uint1_t i;
   uint1_t j;
   uint1_t k;
   uint1_t l;
   uint1_t m;
   uint1_t n;
   uint1_t o;
   uint1_t p;
};
struct uint8_vec {
   uint8_t x;
   uint8_t y;
   uint8_t z;
   uint8_t w;
   uint8_t e;
   uint8_t f;
   uint8_t g;
   uint8_t h;
   uint8_t i;
   uint8_t j;
   uint8_t k;
   uint8_t l;
   uint8_t m;
   uint8_t n;
   uint8_t o;
   uint8_t p;
};
struct uint16_vec {
   uint16_t x;
   uint16_t y;
   uint16_t z;
   uint16_t w;
   uint16_t e;
   uint16_t f;
   uint16_t g;
   uint16_t h;
   uint16_t i;
   uint16_t j;
   uint16_t k;
   uint16_t l;
   uint16_t m;
   uint16_t n;
   uint16_t o;
   uint16_t p;
};
struct uint32_vec {
   uint32_t x;
   uint32_t y;
   uint32_t z;
   uint32_t w;
   uint32_t e;
   uint32_t f;
   uint32_t g;
   uint32_t h;
   uint32_t i;
   uint32_t j;
   uint32_t k;
   uint32_t l;
   uint32_t m;
   uint32_t n;
   uint32_t o;
   uint32_t p;
};
struct uint64_vec {
   uint64_t x;
   uint64_t y;
   uint64_t z;
   uint64_t w;
   uint64_t e;
   uint64_t f;
   uint64_t g;
   uint64_t h;
   uint64_t i;
   uint64_t j;
   uint64_t k;
   uint64_t l;
   uint64_t m;
   uint64_t n;
   uint64_t o;
   uint64_t p;
};
struct bool1_vec {
   bool1_t x;
   bool1_t y;
   bool1_t z;
   bool1_t w;
   bool1_t e;
   bool1_t f;
   bool1_t g;
   bool1_t h;
   bool1_t i;
   bool1_t j;
   bool1_t k;
   bool1_t l;
   bool1_t m;
   bool1_t n;
   bool1_t o;
   bool1_t p;
};
struct bool8_vec {
   bool8_t x;
   bool8_t y;
   bool8_t z;
   bool8_t w;
   bool8_t e;
   bool8_t f;
   bool8_t g;
   bool8_t h;
   bool8_t i;
   bool8_t j;
   bool8_t k;
   bool8_t l;
   bool8_t m;
   bool8_t n;
   bool8_t o;
   bool8_t p;
};
struct bool16_vec {
   bool16_t x;
   bool16_t y;
   bool16_t z;
   bool16_t w;
   bool16_t e;
   bool16_t f;
   bool16_t g;
   bool16_t h;
   bool16_t i;
   bool16_t j;
   bool16_t k;
   bool16_t l;
   bool16_t m;
   bool16_t n;
   bool16_t o;
   bool16_t p;
};
struct bool32_vec {
   bool32_t x;
   bool32_t y;
   bool32_t z;
   bool32_t w;
   bool32_t e;
   bool32_t f;
   bool32_t g;
   bool32_t h;
   bool32_t i;
   bool32_t j;
   bool32_t k;
   bool32_t l;
   bool32_t m;
   bool32_t n;
   bool32_t o;
   bool32_t p;
};



static nir_component_mask_t
evaluate_alignbyte_amd(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;
               const uint32_t src1 =
                  _src[1][_i].u32;
               const uint32_t src2 =
                  _src[2][_i].u32;

         
            uint32_t dst;

            
   uint64_t src = src1 | ((uint64_t)src0 << 32);
   dst = src >> ((src2 & 0x3) * 8);


            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_amul(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               /* 1-bit integers use a 0/-1 convention */
               const int1_t src0 = -(int1_t)_src[0][_i].b;
               /* 1-bit integers use a 0/-1 convention */
               const int1_t src1 = -(int1_t)_src[1][_i].b;

         
            int1_t dst = src0 * src1;

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int8_t src0 =
                  _src[0][_i].i8;
               const int8_t src1 =
                  _src[1][_i].i8;

         
            int8_t dst = src0 * src1;

            _dst_val[_i].i8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int16_t src0 =
                  _src[0][_i].i16;
               const int16_t src1 =
                  _src[1][_i].i16;

         
            int16_t dst = src0 * src1;

            _dst_val[_i].i16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int32_t src0 =
                  _src[0][_i].i32;
               const int32_t src1 =
                  _src[1][_i].i32;

         
            int32_t dst = src0 * src1;

            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int64_t src0 =
                  _src[0][_i].i64;
               const int64_t src1 =
                  _src[1][_i].i64;

         
            int64_t dst = src0 * src1;

            _dst_val[_i].i64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_andg_ir3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint1_t src0 =
                  _src[0][_i].b;
               const uint1_t src1 =
                  _src[1][_i].b;
               const uint1_t src2 =
                  _src[2][_i].b;

         
            uint1_t dst = (src0 & src1) | src2;

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint8_t src0 =
                  _src[0][_i].u8;
               const uint8_t src1 =
                  _src[1][_i].u8;
               const uint8_t src2 =
                  _src[2][_i].u8;

         
            uint8_t dst = (src0 & src1) | src2;

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint16_t src0 =
                  _src[0][_i].u16;
               const uint16_t src1 =
                  _src[1][_i].u16;
               const uint16_t src2 =
                  _src[2][_i].u16;

         
            uint16_t dst = (src0 & src1) | src2;

            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;
               const uint32_t src1 =
                  _src[1][_i].u32;
               const uint32_t src2 =
                  _src[2][_i].u32;

         
            uint32_t dst = (src0 & src1) | src2;

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint64_t src0 =
                  _src[0][_i].u64;
               const uint64_t src1 =
                  _src[1][_i].u64;
               const uint64_t src2 =
                  _src[2][_i].u64;

         
            uint64_t dst = (src0 & src1) | src2;

            _dst_val[_i].u64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16all_fequal16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
            get_float_source(_src[0][8], execution_mode, 16),
            get_float_source(_src[0][9], execution_mode, 16),
            get_float_source(_src[0][10], execution_mode, 16),
            get_float_source(_src[0][11], execution_mode, 16),
            get_float_source(_src[0][12], execution_mode, 16),
            get_float_source(_src[0][13], execution_mode, 16),
            get_float_source(_src[0][14], execution_mode, 16),
            get_float_source(_src[0][15], execution_mode, 16),
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
            get_float_source(_src[1][8], execution_mode, 16),
            get_float_source(_src[1][9], execution_mode, 16),
            get_float_source(_src[1][10], execution_mode, 16),
            get_float_source(_src[1][11], execution_mode, 16),
            get_float_source(_src[1][12], execution_mode, 16),
            get_float_source(_src[1][13], execution_mode, 16),
            get_float_source(_src[1][14], execution_mode, 16),
            get_float_source(_src[1][15], execution_mode, 16),
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
            get_float_source(_src[0][8], execution_mode, 32),
            get_float_source(_src[0][9], execution_mode, 32),
            get_float_source(_src[0][10], execution_mode, 32),
            get_float_source(_src[0][11], execution_mode, 32),
            get_float_source(_src[0][12], execution_mode, 32),
            get_float_source(_src[0][13], execution_mode, 32),
            get_float_source(_src[0][14], execution_mode, 32),
            get_float_source(_src[0][15], execution_mode, 32),
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
            get_float_source(_src[1][8], execution_mode, 32),
            get_float_source(_src[1][9], execution_mode, 32),
            get_float_source(_src[1][10], execution_mode, 32),
            get_float_source(_src[1][11], execution_mode, 32),
            get_float_source(_src[1][12], execution_mode, 32),
            get_float_source(_src[1][13], execution_mode, 32),
            get_float_source(_src[1][14], execution_mode, 32),
            get_float_source(_src[1][15], execution_mode, 32),
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
            get_float_source(_src[0][8], execution_mode, 64),
            get_float_source(_src[0][9], execution_mode, 64),
            get_float_source(_src[0][10], execution_mode, 64),
            get_float_source(_src[0][11], execution_mode, 64),
            get_float_source(_src[0][12], execution_mode, 64),
            get_float_source(_src[0][13], execution_mode, 64),
            get_float_source(_src[0][14], execution_mode, 64),
            get_float_source(_src[0][15], execution_mode, 64),
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
            get_float_source(_src[1][8], execution_mode, 64),
            get_float_source(_src[1][9], execution_mode, 64),
            get_float_source(_src[1][10], execution_mode, 64),
            get_float_source(_src[1][11], execution_mode, 64),
            get_float_source(_src[1][12], execution_mode, 64),
            get_float_source(_src[1][13], execution_mode, 64),
            get_float_source(_src[1][14], execution_mode, 64),
            get_float_source(_src[1][15], execution_mode, 64),
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16all_fequal2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16all_fequal3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16all_fequal4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16all_fequal5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16all_fequal8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16all_iequal16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][7].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][8].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][9].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][10].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][11].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][12].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][13].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][14].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][15].b,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][7].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][8].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][9].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][10].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][11].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][12].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][13].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][14].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][15].b,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
            _src[0][5].i8,
            _src[0][6].i8,
            _src[0][7].i8,
            _src[0][8].i8,
            _src[0][9].i8,
            _src[0][10].i8,
            _src[0][11].i8,
            _src[0][12].i8,
            _src[0][13].i8,
            _src[0][14].i8,
            _src[0][15].i8,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
            _src[1][5].i8,
            _src[1][6].i8,
            _src[1][7].i8,
            _src[1][8].i8,
            _src[1][9].i8,
            _src[1][10].i8,
            _src[1][11].i8,
            _src[1][12].i8,
            _src[1][13].i8,
            _src[1][14].i8,
            _src[1][15].i8,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
            _src[0][5].i16,
            _src[0][6].i16,
            _src[0][7].i16,
            _src[0][8].i16,
            _src[0][9].i16,
            _src[0][10].i16,
            _src[0][11].i16,
            _src[0][12].i16,
            _src[0][13].i16,
            _src[0][14].i16,
            _src[0][15].i16,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
            _src[1][5].i16,
            _src[1][6].i16,
            _src[1][7].i16,
            _src[1][8].i16,
            _src[1][9].i16,
            _src[1][10].i16,
            _src[1][11].i16,
            _src[1][12].i16,
            _src[1][13].i16,
            _src[1][14].i16,
            _src[1][15].i16,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
            _src[0][5].i32,
            _src[0][6].i32,
            _src[0][7].i32,
            _src[0][8].i32,
            _src[0][9].i32,
            _src[0][10].i32,
            _src[0][11].i32,
            _src[0][12].i32,
            _src[0][13].i32,
            _src[0][14].i32,
            _src[0][15].i32,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
            _src[1][5].i32,
            _src[1][6].i32,
            _src[1][7].i32,
            _src[1][8].i32,
            _src[1][9].i32,
            _src[1][10].i32,
            _src[1][11].i32,
            _src[1][12].i32,
            _src[1][13].i32,
            _src[1][14].i32,
            _src[1][15].i32,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
            _src[0][5].i64,
            _src[0][6].i64,
            _src[0][7].i64,
            _src[0][8].i64,
            _src[0][9].i64,
            _src[0][10].i64,
            _src[0][11].i64,
            _src[0][12].i64,
            _src[0][13].i64,
            _src[0][14].i64,
            _src[0][15].i64,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
            _src[1][5].i64,
            _src[1][6].i64,
            _src[1][7].i64,
            _src[1][8].i64,
            _src[1][9].i64,
            _src[1][10].i64,
            _src[1][11].i64,
            _src[1][12].i64,
            _src[1][13].i64,
            _src[1][14].i64,
            _src[1][15].i64,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16all_iequal2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16all_iequal3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16all_iequal4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16all_iequal5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16all_iequal8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][7].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][7].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
            _src[0][5].i8,
            _src[0][6].i8,
            _src[0][7].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
            _src[1][5].i8,
            _src[1][6].i8,
            _src[1][7].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
            _src[0][5].i16,
            _src[0][6].i16,
            _src[0][7].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
            _src[1][5].i16,
            _src[1][6].i16,
            _src[1][7].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
            _src[0][5].i32,
            _src[0][6].i32,
            _src[0][7].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
            _src[1][5].i32,
            _src[1][6].i32,
            _src[1][7].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
            _src[0][5].i64,
            _src[0][6].i64,
            _src[0][7].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
            _src[1][5].i64,
            _src[1][6].i64,
            _src[1][7].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16any_fnequal16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
            get_float_source(_src[0][8], execution_mode, 16),
            get_float_source(_src[0][9], execution_mode, 16),
            get_float_source(_src[0][10], execution_mode, 16),
            get_float_source(_src[0][11], execution_mode, 16),
            get_float_source(_src[0][12], execution_mode, 16),
            get_float_source(_src[0][13], execution_mode, 16),
            get_float_source(_src[0][14], execution_mode, 16),
            get_float_source(_src[0][15], execution_mode, 16),
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
            get_float_source(_src[1][8], execution_mode, 16),
            get_float_source(_src[1][9], execution_mode, 16),
            get_float_source(_src[1][10], execution_mode, 16),
            get_float_source(_src[1][11], execution_mode, 16),
            get_float_source(_src[1][12], execution_mode, 16),
            get_float_source(_src[1][13], execution_mode, 16),
            get_float_source(_src[1][14], execution_mode, 16),
            get_float_source(_src[1][15], execution_mode, 16),
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
            get_float_source(_src[0][8], execution_mode, 32),
            get_float_source(_src[0][9], execution_mode, 32),
            get_float_source(_src[0][10], execution_mode, 32),
            get_float_source(_src[0][11], execution_mode, 32),
            get_float_source(_src[0][12], execution_mode, 32),
            get_float_source(_src[0][13], execution_mode, 32),
            get_float_source(_src[0][14], execution_mode, 32),
            get_float_source(_src[0][15], execution_mode, 32),
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
            get_float_source(_src[1][8], execution_mode, 32),
            get_float_source(_src[1][9], execution_mode, 32),
            get_float_source(_src[1][10], execution_mode, 32),
            get_float_source(_src[1][11], execution_mode, 32),
            get_float_source(_src[1][12], execution_mode, 32),
            get_float_source(_src[1][13], execution_mode, 32),
            get_float_source(_src[1][14], execution_mode, 32),
            get_float_source(_src[1][15], execution_mode, 32),
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
            get_float_source(_src[0][8], execution_mode, 64),
            get_float_source(_src[0][9], execution_mode, 64),
            get_float_source(_src[0][10], execution_mode, 64),
            get_float_source(_src[0][11], execution_mode, 64),
            get_float_source(_src[0][12], execution_mode, 64),
            get_float_source(_src[0][13], execution_mode, 64),
            get_float_source(_src[0][14], execution_mode, 64),
            get_float_source(_src[0][15], execution_mode, 64),
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
            get_float_source(_src[1][8], execution_mode, 64),
            get_float_source(_src[1][9], execution_mode, 64),
            get_float_source(_src[1][10], execution_mode, 64),
            get_float_source(_src[1][11], execution_mode, 64),
            get_float_source(_src[1][12], execution_mode, 64),
            get_float_source(_src[1][13], execution_mode, 64),
            get_float_source(_src[1][14], execution_mode, 64),
            get_float_source(_src[1][15], execution_mode, 64),
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16any_fnequal2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16any_fnequal3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16any_fnequal4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16any_fnequal5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16any_fnequal8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16any_inequal16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][7].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][8].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][9].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][10].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][11].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][12].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][13].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][14].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][15].b,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][7].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][8].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][9].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][10].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][11].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][12].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][13].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][14].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][15].b,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
            _src[0][5].i8,
            _src[0][6].i8,
            _src[0][7].i8,
            _src[0][8].i8,
            _src[0][9].i8,
            _src[0][10].i8,
            _src[0][11].i8,
            _src[0][12].i8,
            _src[0][13].i8,
            _src[0][14].i8,
            _src[0][15].i8,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
            _src[1][5].i8,
            _src[1][6].i8,
            _src[1][7].i8,
            _src[1][8].i8,
            _src[1][9].i8,
            _src[1][10].i8,
            _src[1][11].i8,
            _src[1][12].i8,
            _src[1][13].i8,
            _src[1][14].i8,
            _src[1][15].i8,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
            _src[0][5].i16,
            _src[0][6].i16,
            _src[0][7].i16,
            _src[0][8].i16,
            _src[0][9].i16,
            _src[0][10].i16,
            _src[0][11].i16,
            _src[0][12].i16,
            _src[0][13].i16,
            _src[0][14].i16,
            _src[0][15].i16,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
            _src[1][5].i16,
            _src[1][6].i16,
            _src[1][7].i16,
            _src[1][8].i16,
            _src[1][9].i16,
            _src[1][10].i16,
            _src[1][11].i16,
            _src[1][12].i16,
            _src[1][13].i16,
            _src[1][14].i16,
            _src[1][15].i16,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
            _src[0][5].i32,
            _src[0][6].i32,
            _src[0][7].i32,
            _src[0][8].i32,
            _src[0][9].i32,
            _src[0][10].i32,
            _src[0][11].i32,
            _src[0][12].i32,
            _src[0][13].i32,
            _src[0][14].i32,
            _src[0][15].i32,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
            _src[1][5].i32,
            _src[1][6].i32,
            _src[1][7].i32,
            _src[1][8].i32,
            _src[1][9].i32,
            _src[1][10].i32,
            _src[1][11].i32,
            _src[1][12].i32,
            _src[1][13].i32,
            _src[1][14].i32,
            _src[1][15].i32,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
            _src[0][5].i64,
            _src[0][6].i64,
            _src[0][7].i64,
            _src[0][8].i64,
            _src[0][9].i64,
            _src[0][10].i64,
            _src[0][11].i64,
            _src[0][12].i64,
            _src[0][13].i64,
            _src[0][14].i64,
            _src[0][15].i64,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
            _src[1][5].i64,
            _src[1][6].i64,
            _src[1][7].i64,
            _src[1][8].i64,
            _src[1][9].i64,
            _src[1][10].i64,
            _src[1][11].i64,
            _src[1][12].i64,
            _src[1][13].i64,
            _src[1][14].i64,
            _src[1][15].i64,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16any_inequal2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16any_inequal3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16any_inequal4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16any_inequal5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16any_inequal8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][7].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][7].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
            _src[0][5].i8,
            _src[0][6].i8,
            _src[0][7].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
            _src[1][5].i8,
            _src[1][6].i8,
            _src[1][7].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
            _src[0][5].i16,
            _src[0][6].i16,
            _src[0][7].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
            _src[1][5].i16,
            _src[1][6].i16,
            _src[1][7].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
            _src[0][5].i32,
            _src[0][6].i32,
            _src[0][7].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
            _src[1][5].i32,
            _src[1][6].i32,
            _src[1][7].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
            _src[0][5].i64,
            _src[0][6].i64,
            _src[0][7].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
            _src[1][5].i64,
            _src[1][6].i64,
            _src[1][7].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i16 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b16csel(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool16_t src0 =
                  _src[0][_i].i16;
               const uint1_t src1 =
                  _src[1][_i].b;
               const uint1_t src2 =
                  _src[2][_i].b;

         
            uint1_t dst = src0 ? src1 : src2;

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool16_t src0 =
                  _src[0][_i].i16;
               const uint8_t src1 =
                  _src[1][_i].u8;
               const uint8_t src2 =
                  _src[2][_i].u8;

         
            uint8_t dst = src0 ? src1 : src2;

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool16_t src0 =
                  _src[0][_i].i16;
               const uint16_t src1 =
                  _src[1][_i].u16;
               const uint16_t src2 =
                  _src[2][_i].u16;

         
            uint16_t dst = src0 ? src1 : src2;

            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool16_t src0 =
                  _src[0][_i].i16;
               const uint32_t src1 =
                  _src[1][_i].u32;
               const uint32_t src2 =
                  _src[2][_i].u32;

         
            uint32_t dst = src0 ? src1 : src2;

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool16_t src0 =
                  _src[0][_i].i16;
               const uint64_t src1 =
                  _src[1][_i].u64;
               const uint64_t src2 =
                  _src[2][_i].u64;

         
            uint64_t dst = src0 ? src1 : src2;

            _dst_val[_i].u64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b2b1(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool1_t src0 =
                  _src[0][_i].b;

         
            bool1_t dst = src0 != 0;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool8_t src0 =
                  _src[0][_i].i8;

         
            bool1_t dst = src0 != 0;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool16_t src0 =
                  _src[0][_i].i16;

         
            bool1_t dst = src0 != 0;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;

         
            bool1_t dst = src0 != 0;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b2b16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool1_t src0 =
                  _src[0][_i].b;

         
            bool16_t dst = src0 != 0;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool8_t src0 =
                  _src[0][_i].i8;

         
            bool16_t dst = src0 != 0;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool16_t src0 =
                  _src[0][_i].i16;

         
            bool16_t dst = src0 != 0;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;

         
            bool16_t dst = src0 != 0;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b2b32(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool1_t src0 =
                  _src[0][_i].b;

         
            bool32_t dst = src0 != 0;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool8_t src0 =
                  _src[0][_i].i8;

         
            bool32_t dst = src0 != 0;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool16_t src0 =
                  _src[0][_i].i16;

         
            bool32_t dst = src0 != 0;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;

         
            bool32_t dst = src0 != 0;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b2b8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool1_t src0 =
                  _src[0][_i].b;

         
            bool8_t dst = src0 != 0;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool8_t src0 =
                  _src[0][_i].i8;

         
            bool8_t dst = src0 != 0;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool16_t src0 =
                  _src[0][_i].i16;

         
            bool8_t dst = src0 != 0;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;

         
            bool8_t dst = src0 != 0;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b2f16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool1_t src0 =
                  _src[0][_i].b;

         
            float16_t dst = src0;

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool8_t src0 =
                  _src[0][_i].i8;

         
            float16_t dst = src0;

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool16_t src0 =
                  _src[0][_i].i16;

         
            float16_t dst = src0;

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;

         
            float16_t dst = src0;

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b2f32(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool1_t src0 =
                  _src[0][_i].b;

         
            float32_t dst = src0;

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool8_t src0 =
                  _src[0][_i].i8;

         
            float32_t dst = src0;

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool16_t src0 =
                  _src[0][_i].i16;

         
            float32_t dst = src0;

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;

         
            float32_t dst = src0;

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b2f64(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool1_t src0 =
                  _src[0][_i].b;

         
            float64_t dst = src0;

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool8_t src0 =
                  _src[0][_i].i8;

         
            float64_t dst = src0;

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool16_t src0 =
                  _src[0][_i].i16;

         
            float64_t dst = src0;

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;

         
            float64_t dst = src0;

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b2i1(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool1_t src0 =
                  _src[0][_i].b;

         
            int1_t dst = src0;

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool8_t src0 =
                  _src[0][_i].i8;

         
            int1_t dst = src0;

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool16_t src0 =
                  _src[0][_i].i16;

         
            int1_t dst = src0;

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;

         
            int1_t dst = src0;

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b2i16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool1_t src0 =
                  _src[0][_i].b;

         
            int16_t dst = src0;

            _dst_val[_i].i16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool8_t src0 =
                  _src[0][_i].i8;

         
            int16_t dst = src0;

            _dst_val[_i].i16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool16_t src0 =
                  _src[0][_i].i16;

         
            int16_t dst = src0;

            _dst_val[_i].i16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;

         
            int16_t dst = src0;

            _dst_val[_i].i16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b2i32(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool1_t src0 =
                  _src[0][_i].b;

         
            int32_t dst = src0;

            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool8_t src0 =
                  _src[0][_i].i8;

         
            int32_t dst = src0;

            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool16_t src0 =
                  _src[0][_i].i16;

         
            int32_t dst = src0;

            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;

         
            int32_t dst = src0;

            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b2i64(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool1_t src0 =
                  _src[0][_i].b;

         
            int64_t dst = src0;

            _dst_val[_i].i64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool8_t src0 =
                  _src[0][_i].i8;

         
            int64_t dst = src0;

            _dst_val[_i].i64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool16_t src0 =
                  _src[0][_i].i16;

         
            int64_t dst = src0;

            _dst_val[_i].i64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;

         
            int64_t dst = src0;

            _dst_val[_i].i64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b2i8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool1_t src0 =
                  _src[0][_i].b;

         
            int8_t dst = src0;

            _dst_val[_i].i8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool8_t src0 =
                  _src[0][_i].i8;

         
            int8_t dst = src0;

            _dst_val[_i].i8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool16_t src0 =
                  _src[0][_i].i16;

         
            int8_t dst = src0;

            _dst_val[_i].i8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;

         
            int8_t dst = src0;

            _dst_val[_i].i8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32all_fequal16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
            get_float_source(_src[0][8], execution_mode, 16),
            get_float_source(_src[0][9], execution_mode, 16),
            get_float_source(_src[0][10], execution_mode, 16),
            get_float_source(_src[0][11], execution_mode, 16),
            get_float_source(_src[0][12], execution_mode, 16),
            get_float_source(_src[0][13], execution_mode, 16),
            get_float_source(_src[0][14], execution_mode, 16),
            get_float_source(_src[0][15], execution_mode, 16),
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
            get_float_source(_src[1][8], execution_mode, 16),
            get_float_source(_src[1][9], execution_mode, 16),
            get_float_source(_src[1][10], execution_mode, 16),
            get_float_source(_src[1][11], execution_mode, 16),
            get_float_source(_src[1][12], execution_mode, 16),
            get_float_source(_src[1][13], execution_mode, 16),
            get_float_source(_src[1][14], execution_mode, 16),
            get_float_source(_src[1][15], execution_mode, 16),
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
            get_float_source(_src[0][8], execution_mode, 32),
            get_float_source(_src[0][9], execution_mode, 32),
            get_float_source(_src[0][10], execution_mode, 32),
            get_float_source(_src[0][11], execution_mode, 32),
            get_float_source(_src[0][12], execution_mode, 32),
            get_float_source(_src[0][13], execution_mode, 32),
            get_float_source(_src[0][14], execution_mode, 32),
            get_float_source(_src[0][15], execution_mode, 32),
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
            get_float_source(_src[1][8], execution_mode, 32),
            get_float_source(_src[1][9], execution_mode, 32),
            get_float_source(_src[1][10], execution_mode, 32),
            get_float_source(_src[1][11], execution_mode, 32),
            get_float_source(_src[1][12], execution_mode, 32),
            get_float_source(_src[1][13], execution_mode, 32),
            get_float_source(_src[1][14], execution_mode, 32),
            get_float_source(_src[1][15], execution_mode, 32),
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
            get_float_source(_src[0][8], execution_mode, 64),
            get_float_source(_src[0][9], execution_mode, 64),
            get_float_source(_src[0][10], execution_mode, 64),
            get_float_source(_src[0][11], execution_mode, 64),
            get_float_source(_src[0][12], execution_mode, 64),
            get_float_source(_src[0][13], execution_mode, 64),
            get_float_source(_src[0][14], execution_mode, 64),
            get_float_source(_src[0][15], execution_mode, 64),
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
            get_float_source(_src[1][8], execution_mode, 64),
            get_float_source(_src[1][9], execution_mode, 64),
            get_float_source(_src[1][10], execution_mode, 64),
            get_float_source(_src[1][11], execution_mode, 64),
            get_float_source(_src[1][12], execution_mode, 64),
            get_float_source(_src[1][13], execution_mode, 64),
            get_float_source(_src[1][14], execution_mode, 64),
            get_float_source(_src[1][15], execution_mode, 64),
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32all_fequal2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32all_fequal3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32all_fequal4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32all_fequal5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32all_fequal8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32all_iequal16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][7].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][8].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][9].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][10].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][11].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][12].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][13].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][14].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][15].b,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][7].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][8].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][9].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][10].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][11].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][12].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][13].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][14].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][15].b,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
            _src[0][5].i8,
            _src[0][6].i8,
            _src[0][7].i8,
            _src[0][8].i8,
            _src[0][9].i8,
            _src[0][10].i8,
            _src[0][11].i8,
            _src[0][12].i8,
            _src[0][13].i8,
            _src[0][14].i8,
            _src[0][15].i8,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
            _src[1][5].i8,
            _src[1][6].i8,
            _src[1][7].i8,
            _src[1][8].i8,
            _src[1][9].i8,
            _src[1][10].i8,
            _src[1][11].i8,
            _src[1][12].i8,
            _src[1][13].i8,
            _src[1][14].i8,
            _src[1][15].i8,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
            _src[0][5].i16,
            _src[0][6].i16,
            _src[0][7].i16,
            _src[0][8].i16,
            _src[0][9].i16,
            _src[0][10].i16,
            _src[0][11].i16,
            _src[0][12].i16,
            _src[0][13].i16,
            _src[0][14].i16,
            _src[0][15].i16,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
            _src[1][5].i16,
            _src[1][6].i16,
            _src[1][7].i16,
            _src[1][8].i16,
            _src[1][9].i16,
            _src[1][10].i16,
            _src[1][11].i16,
            _src[1][12].i16,
            _src[1][13].i16,
            _src[1][14].i16,
            _src[1][15].i16,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
            _src[0][5].i32,
            _src[0][6].i32,
            _src[0][7].i32,
            _src[0][8].i32,
            _src[0][9].i32,
            _src[0][10].i32,
            _src[0][11].i32,
            _src[0][12].i32,
            _src[0][13].i32,
            _src[0][14].i32,
            _src[0][15].i32,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
            _src[1][5].i32,
            _src[1][6].i32,
            _src[1][7].i32,
            _src[1][8].i32,
            _src[1][9].i32,
            _src[1][10].i32,
            _src[1][11].i32,
            _src[1][12].i32,
            _src[1][13].i32,
            _src[1][14].i32,
            _src[1][15].i32,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
            _src[0][5].i64,
            _src[0][6].i64,
            _src[0][7].i64,
            _src[0][8].i64,
            _src[0][9].i64,
            _src[0][10].i64,
            _src[0][11].i64,
            _src[0][12].i64,
            _src[0][13].i64,
            _src[0][14].i64,
            _src[0][15].i64,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
            _src[1][5].i64,
            _src[1][6].i64,
            _src[1][7].i64,
            _src[1][8].i64,
            _src[1][9].i64,
            _src[1][10].i64,
            _src[1][11].i64,
            _src[1][12].i64,
            _src[1][13].i64,
            _src[1][14].i64,
            _src[1][15].i64,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32all_iequal2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32all_iequal3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32all_iequal4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32all_iequal5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32all_iequal8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][7].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][7].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
            _src[0][5].i8,
            _src[0][6].i8,
            _src[0][7].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
            _src[1][5].i8,
            _src[1][6].i8,
            _src[1][7].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
            _src[0][5].i16,
            _src[0][6].i16,
            _src[0][7].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
            _src[1][5].i16,
            _src[1][6].i16,
            _src[1][7].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
            _src[0][5].i32,
            _src[0][6].i32,
            _src[0][7].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
            _src[1][5].i32,
            _src[1][6].i32,
            _src[1][7].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
            _src[0][5].i64,
            _src[0][6].i64,
            _src[0][7].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
            _src[1][5].i64,
            _src[1][6].i64,
            _src[1][7].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32any_fnequal16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
            get_float_source(_src[0][8], execution_mode, 16),
            get_float_source(_src[0][9], execution_mode, 16),
            get_float_source(_src[0][10], execution_mode, 16),
            get_float_source(_src[0][11], execution_mode, 16),
            get_float_source(_src[0][12], execution_mode, 16),
            get_float_source(_src[0][13], execution_mode, 16),
            get_float_source(_src[0][14], execution_mode, 16),
            get_float_source(_src[0][15], execution_mode, 16),
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
            get_float_source(_src[1][8], execution_mode, 16),
            get_float_source(_src[1][9], execution_mode, 16),
            get_float_source(_src[1][10], execution_mode, 16),
            get_float_source(_src[1][11], execution_mode, 16),
            get_float_source(_src[1][12], execution_mode, 16),
            get_float_source(_src[1][13], execution_mode, 16),
            get_float_source(_src[1][14], execution_mode, 16),
            get_float_source(_src[1][15], execution_mode, 16),
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
            get_float_source(_src[0][8], execution_mode, 32),
            get_float_source(_src[0][9], execution_mode, 32),
            get_float_source(_src[0][10], execution_mode, 32),
            get_float_source(_src[0][11], execution_mode, 32),
            get_float_source(_src[0][12], execution_mode, 32),
            get_float_source(_src[0][13], execution_mode, 32),
            get_float_source(_src[0][14], execution_mode, 32),
            get_float_source(_src[0][15], execution_mode, 32),
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
            get_float_source(_src[1][8], execution_mode, 32),
            get_float_source(_src[1][9], execution_mode, 32),
            get_float_source(_src[1][10], execution_mode, 32),
            get_float_source(_src[1][11], execution_mode, 32),
            get_float_source(_src[1][12], execution_mode, 32),
            get_float_source(_src[1][13], execution_mode, 32),
            get_float_source(_src[1][14], execution_mode, 32),
            get_float_source(_src[1][15], execution_mode, 32),
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
            get_float_source(_src[0][8], execution_mode, 64),
            get_float_source(_src[0][9], execution_mode, 64),
            get_float_source(_src[0][10], execution_mode, 64),
            get_float_source(_src[0][11], execution_mode, 64),
            get_float_source(_src[0][12], execution_mode, 64),
            get_float_source(_src[0][13], execution_mode, 64),
            get_float_source(_src[0][14], execution_mode, 64),
            get_float_source(_src[0][15], execution_mode, 64),
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
            get_float_source(_src[1][8], execution_mode, 64),
            get_float_source(_src[1][9], execution_mode, 64),
            get_float_source(_src[1][10], execution_mode, 64),
            get_float_source(_src[1][11], execution_mode, 64),
            get_float_source(_src[1][12], execution_mode, 64),
            get_float_source(_src[1][13], execution_mode, 64),
            get_float_source(_src[1][14], execution_mode, 64),
            get_float_source(_src[1][15], execution_mode, 64),
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32any_fnequal2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32any_fnequal3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32any_fnequal4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32any_fnequal5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32any_fnequal8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32any_inequal16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][7].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][8].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][9].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][10].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][11].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][12].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][13].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][14].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][15].b,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][7].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][8].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][9].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][10].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][11].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][12].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][13].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][14].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][15].b,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
            _src[0][5].i8,
            _src[0][6].i8,
            _src[0][7].i8,
            _src[0][8].i8,
            _src[0][9].i8,
            _src[0][10].i8,
            _src[0][11].i8,
            _src[0][12].i8,
            _src[0][13].i8,
            _src[0][14].i8,
            _src[0][15].i8,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
            _src[1][5].i8,
            _src[1][6].i8,
            _src[1][7].i8,
            _src[1][8].i8,
            _src[1][9].i8,
            _src[1][10].i8,
            _src[1][11].i8,
            _src[1][12].i8,
            _src[1][13].i8,
            _src[1][14].i8,
            _src[1][15].i8,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
            _src[0][5].i16,
            _src[0][6].i16,
            _src[0][7].i16,
            _src[0][8].i16,
            _src[0][9].i16,
            _src[0][10].i16,
            _src[0][11].i16,
            _src[0][12].i16,
            _src[0][13].i16,
            _src[0][14].i16,
            _src[0][15].i16,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
            _src[1][5].i16,
            _src[1][6].i16,
            _src[1][7].i16,
            _src[1][8].i16,
            _src[1][9].i16,
            _src[1][10].i16,
            _src[1][11].i16,
            _src[1][12].i16,
            _src[1][13].i16,
            _src[1][14].i16,
            _src[1][15].i16,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
            _src[0][5].i32,
            _src[0][6].i32,
            _src[0][7].i32,
            _src[0][8].i32,
            _src[0][9].i32,
            _src[0][10].i32,
            _src[0][11].i32,
            _src[0][12].i32,
            _src[0][13].i32,
            _src[0][14].i32,
            _src[0][15].i32,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
            _src[1][5].i32,
            _src[1][6].i32,
            _src[1][7].i32,
            _src[1][8].i32,
            _src[1][9].i32,
            _src[1][10].i32,
            _src[1][11].i32,
            _src[1][12].i32,
            _src[1][13].i32,
            _src[1][14].i32,
            _src[1][15].i32,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
            _src[0][5].i64,
            _src[0][6].i64,
            _src[0][7].i64,
            _src[0][8].i64,
            _src[0][9].i64,
            _src[0][10].i64,
            _src[0][11].i64,
            _src[0][12].i64,
            _src[0][13].i64,
            _src[0][14].i64,
            _src[0][15].i64,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
            _src[1][5].i64,
            _src[1][6].i64,
            _src[1][7].i64,
            _src[1][8].i64,
            _src[1][9].i64,
            _src[1][10].i64,
            _src[1][11].i64,
            _src[1][12].i64,
            _src[1][13].i64,
            _src[1][14].i64,
            _src[1][15].i64,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32any_inequal2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32any_inequal3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32any_inequal4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32any_inequal5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32any_inequal8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][7].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][7].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
            _src[0][5].i8,
            _src[0][6].i8,
            _src[0][7].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
            _src[1][5].i8,
            _src[1][6].i8,
            _src[1][7].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
            _src[0][5].i16,
            _src[0][6].i16,
            _src[0][7].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
            _src[1][5].i16,
            _src[1][6].i16,
            _src[1][7].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
            _src[0][5].i32,
            _src[0][6].i32,
            _src[0][7].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
            _src[1][5].i32,
            _src[1][6].i32,
            _src[1][7].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
            _src[0][5].i64,
            _src[0][6].i64,
            _src[0][7].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
            _src[1][5].i64,
            _src[1][6].i64,
            _src[1][7].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i32 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32csel(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;
               const uint1_t src1 =
                  _src[1][_i].b;
               const uint1_t src2 =
                  _src[2][_i].b;

         
            uint1_t dst = src0 ? src1 : src2;

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;
               const uint8_t src1 =
                  _src[1][_i].u8;
               const uint8_t src2 =
                  _src[2][_i].u8;

         
            uint8_t dst = src0 ? src1 : src2;

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;
               const uint16_t src1 =
                  _src[1][_i].u16;
               const uint16_t src2 =
                  _src[2][_i].u16;

         
            uint16_t dst = src0 ? src1 : src2;

            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;
               const uint32_t src1 =
                  _src[1][_i].u32;
               const uint32_t src2 =
                  _src[2][_i].u32;

         
            uint32_t dst = src0 ? src1 : src2;

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;
               const uint64_t src1 =
                  _src[1][_i].u64;
               const uint64_t src2 =
                  _src[2][_i].u64;

         
            uint64_t dst = src0 ? src1 : src2;

            _dst_val[_i].u64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b32fcsel_mdg(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);
               const float16_t src2 =
                  get_float_source(_src[2][_i], execution_mode, 16);

         
            uint16_t dst = src0 ? src1 : src2;

            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);
               const float32_t src2 =
                  get_float_source(_src[2][_i], execution_mode, 32);

         
            uint32_t dst = src0 ? src1 : src2;

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool32_t src0 =
                  _src[0][_i].i32;
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);
               const float64_t src2 =
                  get_float_source(_src[2][_i], execution_mode, 64);

         
            uint64_t dst = src0 ? src1 : src2;

            _dst_val[_i].u64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8all_fequal16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
            get_float_source(_src[0][8], execution_mode, 16),
            get_float_source(_src[0][9], execution_mode, 16),
            get_float_source(_src[0][10], execution_mode, 16),
            get_float_source(_src[0][11], execution_mode, 16),
            get_float_source(_src[0][12], execution_mode, 16),
            get_float_source(_src[0][13], execution_mode, 16),
            get_float_source(_src[0][14], execution_mode, 16),
            get_float_source(_src[0][15], execution_mode, 16),
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
            get_float_source(_src[1][8], execution_mode, 16),
            get_float_source(_src[1][9], execution_mode, 16),
            get_float_source(_src[1][10], execution_mode, 16),
            get_float_source(_src[1][11], execution_mode, 16),
            get_float_source(_src[1][12], execution_mode, 16),
            get_float_source(_src[1][13], execution_mode, 16),
            get_float_source(_src[1][14], execution_mode, 16),
            get_float_source(_src[1][15], execution_mode, 16),
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
            get_float_source(_src[0][8], execution_mode, 32),
            get_float_source(_src[0][9], execution_mode, 32),
            get_float_source(_src[0][10], execution_mode, 32),
            get_float_source(_src[0][11], execution_mode, 32),
            get_float_source(_src[0][12], execution_mode, 32),
            get_float_source(_src[0][13], execution_mode, 32),
            get_float_source(_src[0][14], execution_mode, 32),
            get_float_source(_src[0][15], execution_mode, 32),
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
            get_float_source(_src[1][8], execution_mode, 32),
            get_float_source(_src[1][9], execution_mode, 32),
            get_float_source(_src[1][10], execution_mode, 32),
            get_float_source(_src[1][11], execution_mode, 32),
            get_float_source(_src[1][12], execution_mode, 32),
            get_float_source(_src[1][13], execution_mode, 32),
            get_float_source(_src[1][14], execution_mode, 32),
            get_float_source(_src[1][15], execution_mode, 32),
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
            get_float_source(_src[0][8], execution_mode, 64),
            get_float_source(_src[0][9], execution_mode, 64),
            get_float_source(_src[0][10], execution_mode, 64),
            get_float_source(_src[0][11], execution_mode, 64),
            get_float_source(_src[0][12], execution_mode, 64),
            get_float_source(_src[0][13], execution_mode, 64),
            get_float_source(_src[0][14], execution_mode, 64),
            get_float_source(_src[0][15], execution_mode, 64),
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
            get_float_source(_src[1][8], execution_mode, 64),
            get_float_source(_src[1][9], execution_mode, 64),
            get_float_source(_src[1][10], execution_mode, 64),
            get_float_source(_src[1][11], execution_mode, 64),
            get_float_source(_src[1][12], execution_mode, 64),
            get_float_source(_src[1][13], execution_mode, 64),
            get_float_source(_src[1][14], execution_mode, 64),
            get_float_source(_src[1][15], execution_mode, 64),
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8all_fequal2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8all_fequal3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8all_fequal4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8all_fequal5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8all_fequal8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8all_iequal16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][7].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][8].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][9].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][10].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][11].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][12].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][13].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][14].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][15].b,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][7].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][8].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][9].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][10].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][11].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][12].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][13].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][14].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][15].b,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
            _src[0][5].i8,
            _src[0][6].i8,
            _src[0][7].i8,
            _src[0][8].i8,
            _src[0][9].i8,
            _src[0][10].i8,
            _src[0][11].i8,
            _src[0][12].i8,
            _src[0][13].i8,
            _src[0][14].i8,
            _src[0][15].i8,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
            _src[1][5].i8,
            _src[1][6].i8,
            _src[1][7].i8,
            _src[1][8].i8,
            _src[1][9].i8,
            _src[1][10].i8,
            _src[1][11].i8,
            _src[1][12].i8,
            _src[1][13].i8,
            _src[1][14].i8,
            _src[1][15].i8,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
            _src[0][5].i16,
            _src[0][6].i16,
            _src[0][7].i16,
            _src[0][8].i16,
            _src[0][9].i16,
            _src[0][10].i16,
            _src[0][11].i16,
            _src[0][12].i16,
            _src[0][13].i16,
            _src[0][14].i16,
            _src[0][15].i16,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
            _src[1][5].i16,
            _src[1][6].i16,
            _src[1][7].i16,
            _src[1][8].i16,
            _src[1][9].i16,
            _src[1][10].i16,
            _src[1][11].i16,
            _src[1][12].i16,
            _src[1][13].i16,
            _src[1][14].i16,
            _src[1][15].i16,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
            _src[0][5].i32,
            _src[0][6].i32,
            _src[0][7].i32,
            _src[0][8].i32,
            _src[0][9].i32,
            _src[0][10].i32,
            _src[0][11].i32,
            _src[0][12].i32,
            _src[0][13].i32,
            _src[0][14].i32,
            _src[0][15].i32,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
            _src[1][5].i32,
            _src[1][6].i32,
            _src[1][7].i32,
            _src[1][8].i32,
            _src[1][9].i32,
            _src[1][10].i32,
            _src[1][11].i32,
            _src[1][12].i32,
            _src[1][13].i32,
            _src[1][14].i32,
            _src[1][15].i32,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
            _src[0][5].i64,
            _src[0][6].i64,
            _src[0][7].i64,
            _src[0][8].i64,
            _src[0][9].i64,
            _src[0][10].i64,
            _src[0][11].i64,
            _src[0][12].i64,
            _src[0][13].i64,
            _src[0][14].i64,
            _src[0][15].i64,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
            _src[1][5].i64,
            _src[1][6].i64,
            _src[1][7].i64,
            _src[1][8].i64,
            _src[1][9].i64,
            _src[1][10].i64,
            _src[1][11].i64,
            _src[1][12].i64,
            _src[1][13].i64,
            _src[1][14].i64,
            _src[1][15].i64,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8all_iequal2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8all_iequal3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8all_iequal4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8all_iequal5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8all_iequal8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][7].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][7].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
            _src[0][5].i8,
            _src[0][6].i8,
            _src[0][7].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
            _src[1][5].i8,
            _src[1][6].i8,
            _src[1][7].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
            _src[0][5].i16,
            _src[0][6].i16,
            _src[0][7].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
            _src[1][5].i16,
            _src[1][6].i16,
            _src[1][7].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
            _src[0][5].i32,
            _src[0][6].i32,
            _src[0][7].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
            _src[1][5].i32,
            _src[1][6].i32,
            _src[1][7].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
            _src[0][5].i64,
            _src[0][6].i64,
            _src[0][7].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
            _src[1][5].i64,
            _src[1][6].i64,
            _src[1][7].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8any_fnequal16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
            get_float_source(_src[0][8], execution_mode, 16),
            get_float_source(_src[0][9], execution_mode, 16),
            get_float_source(_src[0][10], execution_mode, 16),
            get_float_source(_src[0][11], execution_mode, 16),
            get_float_source(_src[0][12], execution_mode, 16),
            get_float_source(_src[0][13], execution_mode, 16),
            get_float_source(_src[0][14], execution_mode, 16),
            get_float_source(_src[0][15], execution_mode, 16),
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
            get_float_source(_src[1][8], execution_mode, 16),
            get_float_source(_src[1][9], execution_mode, 16),
            get_float_source(_src[1][10], execution_mode, 16),
            get_float_source(_src[1][11], execution_mode, 16),
            get_float_source(_src[1][12], execution_mode, 16),
            get_float_source(_src[1][13], execution_mode, 16),
            get_float_source(_src[1][14], execution_mode, 16),
            get_float_source(_src[1][15], execution_mode, 16),
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
            get_float_source(_src[0][8], execution_mode, 32),
            get_float_source(_src[0][9], execution_mode, 32),
            get_float_source(_src[0][10], execution_mode, 32),
            get_float_source(_src[0][11], execution_mode, 32),
            get_float_source(_src[0][12], execution_mode, 32),
            get_float_source(_src[0][13], execution_mode, 32),
            get_float_source(_src[0][14], execution_mode, 32),
            get_float_source(_src[0][15], execution_mode, 32),
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
            get_float_source(_src[1][8], execution_mode, 32),
            get_float_source(_src[1][9], execution_mode, 32),
            get_float_source(_src[1][10], execution_mode, 32),
            get_float_source(_src[1][11], execution_mode, 32),
            get_float_source(_src[1][12], execution_mode, 32),
            get_float_source(_src[1][13], execution_mode, 32),
            get_float_source(_src[1][14], execution_mode, 32),
            get_float_source(_src[1][15], execution_mode, 32),
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
            get_float_source(_src[0][8], execution_mode, 64),
            get_float_source(_src[0][9], execution_mode, 64),
            get_float_source(_src[0][10], execution_mode, 64),
            get_float_source(_src[0][11], execution_mode, 64),
            get_float_source(_src[0][12], execution_mode, 64),
            get_float_source(_src[0][13], execution_mode, 64),
            get_float_source(_src[0][14], execution_mode, 64),
            get_float_source(_src[0][15], execution_mode, 64),
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
            get_float_source(_src[1][8], execution_mode, 64),
            get_float_source(_src[1][9], execution_mode, 64),
            get_float_source(_src[1][10], execution_mode, 64),
            get_float_source(_src[1][11], execution_mode, 64),
            get_float_source(_src[1][12], execution_mode, 64),
            get_float_source(_src[1][13], execution_mode, 64),
            get_float_source(_src[1][14], execution_mode, 64),
            get_float_source(_src[1][15], execution_mode, 64),
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8any_fnequal2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8any_fnequal3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8any_fnequal4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8any_fnequal5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8any_fnequal8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8any_inequal16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][7].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][8].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][9].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][10].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][11].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][12].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][13].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][14].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][15].b,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][7].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][8].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][9].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][10].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][11].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][12].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][13].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][14].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][15].b,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
            _src[0][5].i8,
            _src[0][6].i8,
            _src[0][7].i8,
            _src[0][8].i8,
            _src[0][9].i8,
            _src[0][10].i8,
            _src[0][11].i8,
            _src[0][12].i8,
            _src[0][13].i8,
            _src[0][14].i8,
            _src[0][15].i8,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
            _src[1][5].i8,
            _src[1][6].i8,
            _src[1][7].i8,
            _src[1][8].i8,
            _src[1][9].i8,
            _src[1][10].i8,
            _src[1][11].i8,
            _src[1][12].i8,
            _src[1][13].i8,
            _src[1][14].i8,
            _src[1][15].i8,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
            _src[0][5].i16,
            _src[0][6].i16,
            _src[0][7].i16,
            _src[0][8].i16,
            _src[0][9].i16,
            _src[0][10].i16,
            _src[0][11].i16,
            _src[0][12].i16,
            _src[0][13].i16,
            _src[0][14].i16,
            _src[0][15].i16,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
            _src[1][5].i16,
            _src[1][6].i16,
            _src[1][7].i16,
            _src[1][8].i16,
            _src[1][9].i16,
            _src[1][10].i16,
            _src[1][11].i16,
            _src[1][12].i16,
            _src[1][13].i16,
            _src[1][14].i16,
            _src[1][15].i16,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
            _src[0][5].i32,
            _src[0][6].i32,
            _src[0][7].i32,
            _src[0][8].i32,
            _src[0][9].i32,
            _src[0][10].i32,
            _src[0][11].i32,
            _src[0][12].i32,
            _src[0][13].i32,
            _src[0][14].i32,
            _src[0][15].i32,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
            _src[1][5].i32,
            _src[1][6].i32,
            _src[1][7].i32,
            _src[1][8].i32,
            _src[1][9].i32,
            _src[1][10].i32,
            _src[1][11].i32,
            _src[1][12].i32,
            _src[1][13].i32,
            _src[1][14].i32,
            _src[1][15].i32,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
            _src[0][5].i64,
            _src[0][6].i64,
            _src[0][7].i64,
            _src[0][8].i64,
            _src[0][9].i64,
            _src[0][10].i64,
            _src[0][11].i64,
            _src[0][12].i64,
            _src[0][13].i64,
            _src[0][14].i64,
            _src[0][15].i64,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
            _src[1][5].i64,
            _src[1][6].i64,
            _src[1][7].i64,
            _src[1][8].i64,
            _src[1][9].i64,
            _src[1][10].i64,
            _src[1][11].i64,
            _src[1][12].i64,
            _src[1][13].i64,
            _src[1][14].i64,
            _src[1][15].i64,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8any_inequal2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8any_inequal3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8any_inequal4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8any_inequal5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8any_inequal8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][7].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][7].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
            _src[0][5].i8,
            _src[0][6].i8,
            _src[0][7].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
            _src[1][5].i8,
            _src[1][6].i8,
            _src[1][7].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
            _src[0][5].i16,
            _src[0][6].i16,
            _src[0][7].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
            _src[1][5].i16,
            _src[1][6].i16,
            _src[1][7].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
            _src[0][5].i32,
            _src[0][6].i32,
            _src[0][7].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
            _src[1][5].i32,
            _src[1][6].i32,
            _src[1][7].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
            _src[0][5].i64,
            _src[0][6].i64,
            _src[0][7].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
            _src[1][5].i64,
            _src[1][6].i64,
            _src[1][7].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool8_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].i8 = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_b8csel(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool8_t src0 =
                  _src[0][_i].i8;
               const uint1_t src1 =
                  _src[1][_i].b;
               const uint1_t src2 =
                  _src[2][_i].b;

         
            uint1_t dst = src0 ? src1 : src2;

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool8_t src0 =
                  _src[0][_i].i8;
               const uint8_t src1 =
                  _src[1][_i].u8;
               const uint8_t src2 =
                  _src[2][_i].u8;

         
            uint8_t dst = src0 ? src1 : src2;

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool8_t src0 =
                  _src[0][_i].i8;
               const uint16_t src1 =
                  _src[1][_i].u16;
               const uint16_t src2 =
                  _src[2][_i].u16;

         
            uint16_t dst = src0 ? src1 : src2;

            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool8_t src0 =
                  _src[0][_i].i8;
               const uint32_t src1 =
                  _src[1][_i].u32;
               const uint32_t src2 =
                  _src[2][_i].u32;

         
            uint32_t dst = src0 ? src1 : src2;

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool8_t src0 =
                  _src[0][_i].i8;
               const uint64_t src1 =
                  _src[1][_i].u64;
               const uint64_t src2 =
                  _src[2][_i].u64;

         
            uint64_t dst = src0 ? src1 : src2;

            _dst_val[_i].u64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ball_fequal16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
            get_float_source(_src[0][8], execution_mode, 16),
            get_float_source(_src[0][9], execution_mode, 16),
            get_float_source(_src[0][10], execution_mode, 16),
            get_float_source(_src[0][11], execution_mode, 16),
            get_float_source(_src[0][12], execution_mode, 16),
            get_float_source(_src[0][13], execution_mode, 16),
            get_float_source(_src[0][14], execution_mode, 16),
            get_float_source(_src[0][15], execution_mode, 16),
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
            get_float_source(_src[1][8], execution_mode, 16),
            get_float_source(_src[1][9], execution_mode, 16),
            get_float_source(_src[1][10], execution_mode, 16),
            get_float_source(_src[1][11], execution_mode, 16),
            get_float_source(_src[1][12], execution_mode, 16),
            get_float_source(_src[1][13], execution_mode, 16),
            get_float_source(_src[1][14], execution_mode, 16),
            get_float_source(_src[1][15], execution_mode, 16),
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
            get_float_source(_src[0][8], execution_mode, 32),
            get_float_source(_src[0][9], execution_mode, 32),
            get_float_source(_src[0][10], execution_mode, 32),
            get_float_source(_src[0][11], execution_mode, 32),
            get_float_source(_src[0][12], execution_mode, 32),
            get_float_source(_src[0][13], execution_mode, 32),
            get_float_source(_src[0][14], execution_mode, 32),
            get_float_source(_src[0][15], execution_mode, 32),
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
            get_float_source(_src[1][8], execution_mode, 32),
            get_float_source(_src[1][9], execution_mode, 32),
            get_float_source(_src[1][10], execution_mode, 32),
            get_float_source(_src[1][11], execution_mode, 32),
            get_float_source(_src[1][12], execution_mode, 32),
            get_float_source(_src[1][13], execution_mode, 32),
            get_float_source(_src[1][14], execution_mode, 32),
            get_float_source(_src[1][15], execution_mode, 32),
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
            get_float_source(_src[0][8], execution_mode, 64),
            get_float_source(_src[0][9], execution_mode, 64),
            get_float_source(_src[0][10], execution_mode, 64),
            get_float_source(_src[0][11], execution_mode, 64),
            get_float_source(_src[0][12], execution_mode, 64),
            get_float_source(_src[0][13], execution_mode, 64),
            get_float_source(_src[0][14], execution_mode, 64),
            get_float_source(_src[0][15], execution_mode, 64),
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
            get_float_source(_src[1][8], execution_mode, 64),
            get_float_source(_src[1][9], execution_mode, 64),
            get_float_source(_src[1][10], execution_mode, 64),
            get_float_source(_src[1][11], execution_mode, 64),
            get_float_source(_src[1][12], execution_mode, 64),
            get_float_source(_src[1][13], execution_mode, 64),
            get_float_source(_src[1][14], execution_mode, 64),
            get_float_source(_src[1][15], execution_mode, 64),
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ball_fequal2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ball_fequal3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ball_fequal4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ball_fequal5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ball_fequal8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ball_iequal16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][7].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][8].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][9].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][10].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][11].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][12].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][13].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][14].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][15].b,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][7].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][8].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][9].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][10].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][11].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][12].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][13].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][14].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][15].b,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
            _src[0][5].i8,
            _src[0][6].i8,
            _src[0][7].i8,
            _src[0][8].i8,
            _src[0][9].i8,
            _src[0][10].i8,
            _src[0][11].i8,
            _src[0][12].i8,
            _src[0][13].i8,
            _src[0][14].i8,
            _src[0][15].i8,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
            _src[1][5].i8,
            _src[1][6].i8,
            _src[1][7].i8,
            _src[1][8].i8,
            _src[1][9].i8,
            _src[1][10].i8,
            _src[1][11].i8,
            _src[1][12].i8,
            _src[1][13].i8,
            _src[1][14].i8,
            _src[1][15].i8,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
            _src[0][5].i16,
            _src[0][6].i16,
            _src[0][7].i16,
            _src[0][8].i16,
            _src[0][9].i16,
            _src[0][10].i16,
            _src[0][11].i16,
            _src[0][12].i16,
            _src[0][13].i16,
            _src[0][14].i16,
            _src[0][15].i16,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
            _src[1][5].i16,
            _src[1][6].i16,
            _src[1][7].i16,
            _src[1][8].i16,
            _src[1][9].i16,
            _src[1][10].i16,
            _src[1][11].i16,
            _src[1][12].i16,
            _src[1][13].i16,
            _src[1][14].i16,
            _src[1][15].i16,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
            _src[0][5].i32,
            _src[0][6].i32,
            _src[0][7].i32,
            _src[0][8].i32,
            _src[0][9].i32,
            _src[0][10].i32,
            _src[0][11].i32,
            _src[0][12].i32,
            _src[0][13].i32,
            _src[0][14].i32,
            _src[0][15].i32,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
            _src[1][5].i32,
            _src[1][6].i32,
            _src[1][7].i32,
            _src[1][8].i32,
            _src[1][9].i32,
            _src[1][10].i32,
            _src[1][11].i32,
            _src[1][12].i32,
            _src[1][13].i32,
            _src[1][14].i32,
            _src[1][15].i32,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
            _src[0][5].i64,
            _src[0][6].i64,
            _src[0][7].i64,
            _src[0][8].i64,
            _src[0][9].i64,
            _src[0][10].i64,
            _src[0][11].i64,
            _src[0][12].i64,
            _src[0][13].i64,
            _src[0][14].i64,
            _src[0][15].i64,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
            _src[1][5].i64,
            _src[1][6].i64,
            _src[1][7].i64,
            _src[1][8].i64,
            _src[1][9].i64,
            _src[1][10].i64,
            _src[1][11].i64,
            _src[1][12].i64,
            _src[1][13].i64,
            _src[1][14].i64,
            _src[1][15].i64,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ball_iequal2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ball_iequal3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ball_iequal4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ball_iequal5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ball_iequal8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][7].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][7].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
            _src[0][5].i8,
            _src[0][6].i8,
            _src[0][7].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
            _src[1][5].i8,
            _src[1][6].i8,
            _src[1][7].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
            _src[0][5].i16,
            _src[0][6].i16,
            _src[0][7].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
            _src[1][5].i16,
            _src[1][6].i16,
            _src[1][7].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
            _src[0][5].i32,
            _src[0][6].i32,
            _src[0][7].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
            _src[1][5].i32,
            _src[1][6].i32,
            _src[1][7].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
            _src[0][5].i64,
            _src[0][6].i64,
            _src[0][7].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
            _src[1][5].i64,
            _src[1][6].i64,
            _src[1][7].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bany_fnequal16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
            get_float_source(_src[0][8], execution_mode, 16),
            get_float_source(_src[0][9], execution_mode, 16),
            get_float_source(_src[0][10], execution_mode, 16),
            get_float_source(_src[0][11], execution_mode, 16),
            get_float_source(_src[0][12], execution_mode, 16),
            get_float_source(_src[0][13], execution_mode, 16),
            get_float_source(_src[0][14], execution_mode, 16),
            get_float_source(_src[0][15], execution_mode, 16),
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
            get_float_source(_src[1][8], execution_mode, 16),
            get_float_source(_src[1][9], execution_mode, 16),
            get_float_source(_src[1][10], execution_mode, 16),
            get_float_source(_src[1][11], execution_mode, 16),
            get_float_source(_src[1][12], execution_mode, 16),
            get_float_source(_src[1][13], execution_mode, 16),
            get_float_source(_src[1][14], execution_mode, 16),
            get_float_source(_src[1][15], execution_mode, 16),
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
            get_float_source(_src[0][8], execution_mode, 32),
            get_float_source(_src[0][9], execution_mode, 32),
            get_float_source(_src[0][10], execution_mode, 32),
            get_float_source(_src[0][11], execution_mode, 32),
            get_float_source(_src[0][12], execution_mode, 32),
            get_float_source(_src[0][13], execution_mode, 32),
            get_float_source(_src[0][14], execution_mode, 32),
            get_float_source(_src[0][15], execution_mode, 32),
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
            get_float_source(_src[1][8], execution_mode, 32),
            get_float_source(_src[1][9], execution_mode, 32),
            get_float_source(_src[1][10], execution_mode, 32),
            get_float_source(_src[1][11], execution_mode, 32),
            get_float_source(_src[1][12], execution_mode, 32),
            get_float_source(_src[1][13], execution_mode, 32),
            get_float_source(_src[1][14], execution_mode, 32),
            get_float_source(_src[1][15], execution_mode, 32),
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
            get_float_source(_src[0][8], execution_mode, 64),
            get_float_source(_src[0][9], execution_mode, 64),
            get_float_source(_src[0][10], execution_mode, 64),
            get_float_source(_src[0][11], execution_mode, 64),
            get_float_source(_src[0][12], execution_mode, 64),
            get_float_source(_src[0][13], execution_mode, 64),
            get_float_source(_src[0][14], execution_mode, 64),
            get_float_source(_src[0][15], execution_mode, 64),
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
            get_float_source(_src[1][8], execution_mode, 64),
            get_float_source(_src[1][9], execution_mode, 64),
            get_float_source(_src[1][10], execution_mode, 64),
            get_float_source(_src[1][11], execution_mode, 64),
            get_float_source(_src[1][12], execution_mode, 64),
            get_float_source(_src[1][13], execution_mode, 64),
            get_float_source(_src[1][14], execution_mode, 64),
            get_float_source(_src[1][15], execution_mode, 64),
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bany_fnequal2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bany_fnequal3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bany_fnequal4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bany_fnequal5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bany_fnequal8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bany_inequal16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][7].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][8].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][9].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][10].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][11].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][12].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][13].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][14].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][15].b,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][7].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][8].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][9].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][10].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][11].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][12].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][13].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][14].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][15].b,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
            _src[0][5].i8,
            _src[0][6].i8,
            _src[0][7].i8,
            _src[0][8].i8,
            _src[0][9].i8,
            _src[0][10].i8,
            _src[0][11].i8,
            _src[0][12].i8,
            _src[0][13].i8,
            _src[0][14].i8,
            _src[0][15].i8,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
            _src[1][5].i8,
            _src[1][6].i8,
            _src[1][7].i8,
            _src[1][8].i8,
            _src[1][9].i8,
            _src[1][10].i8,
            _src[1][11].i8,
            _src[1][12].i8,
            _src[1][13].i8,
            _src[1][14].i8,
            _src[1][15].i8,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
            _src[0][5].i16,
            _src[0][6].i16,
            _src[0][7].i16,
            _src[0][8].i16,
            _src[0][9].i16,
            _src[0][10].i16,
            _src[0][11].i16,
            _src[0][12].i16,
            _src[0][13].i16,
            _src[0][14].i16,
            _src[0][15].i16,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
            _src[1][5].i16,
            _src[1][6].i16,
            _src[1][7].i16,
            _src[1][8].i16,
            _src[1][9].i16,
            _src[1][10].i16,
            _src[1][11].i16,
            _src[1][12].i16,
            _src[1][13].i16,
            _src[1][14].i16,
            _src[1][15].i16,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
            _src[0][5].i32,
            _src[0][6].i32,
            _src[0][7].i32,
            _src[0][8].i32,
            _src[0][9].i32,
            _src[0][10].i32,
            _src[0][11].i32,
            _src[0][12].i32,
            _src[0][13].i32,
            _src[0][14].i32,
            _src[0][15].i32,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
            _src[1][5].i32,
            _src[1][6].i32,
            _src[1][7].i32,
            _src[1][8].i32,
            _src[1][9].i32,
            _src[1][10].i32,
            _src[1][11].i32,
            _src[1][12].i32,
            _src[1][13].i32,
            _src[1][14].i32,
            _src[1][15].i32,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
            _src[0][5].i64,
            _src[0][6].i64,
            _src[0][7].i64,
            _src[0][8].i64,
            _src[0][9].i64,
            _src[0][10].i64,
            _src[0][11].i64,
            _src[0][12].i64,
            _src[0][13].i64,
            _src[0][14].i64,
            _src[0][15].i64,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
            _src[1][5].i64,
            _src[1][6].i64,
            _src[1][7].i64,
            _src[1][8].i64,
            _src[1][9].i64,
            _src[1][10].i64,
            _src[1][11].i64,
            _src[1][12].i64,
            _src[1][13].i64,
            _src[1][14].i64,
            _src[1][15].i64,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bany_inequal2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bany_inequal3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bany_inequal4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bany_inequal5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bany_inequal8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   


      const struct int1_vec src0 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[0][7].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int1_vec src1 = {
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][0].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][1].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][2].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][3].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][4].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][5].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][6].b,
             /* 1-bit integers use a 0/-1 convention */
             -(int1_t)_src[1][7].b,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 8: {
         
   


      const struct int8_vec src0 = {
            _src[0][0].i8,
            _src[0][1].i8,
            _src[0][2].i8,
            _src[0][3].i8,
            _src[0][4].i8,
            _src[0][5].i8,
            _src[0][6].i8,
            _src[0][7].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int8_vec src1 = {
            _src[1][0].i8,
            _src[1][1].i8,
            _src[1][2].i8,
            _src[1][3].i8,
            _src[1][4].i8,
            _src[1][5].i8,
            _src[1][6].i8,
            _src[1][7].i8,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 16: {
         
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
            _src[0][2].i16,
            _src[0][3].i16,
            _src[0][4].i16,
            _src[0][5].i16,
            _src[0][6].i16,
            _src[0][7].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
            _src[1][2].i16,
            _src[1][3].i16,
            _src[1][4].i16,
            _src[1][5].i16,
            _src[1][6].i16,
            _src[1][7].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 32: {
         
   


      const struct int32_vec src0 = {
            _src[0][0].i32,
            _src[0][1].i32,
            _src[0][2].i32,
            _src[0][3].i32,
            _src[0][4].i32,
            _src[0][5].i32,
            _src[0][6].i32,
            _src[0][7].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int32_vec src1 = {
            _src[1][0].i32,
            _src[1][1].i32,
            _src[1][2].i32,
            _src[1][3].i32,
            _src[1][4].i32,
            _src[1][5].i32,
            _src[1][6].i32,
            _src[1][7].i32,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }
      case 64: {
         
   


      const struct int64_vec src0 = {
            _src[0][0].i64,
            _src[0][1].i64,
            _src[0][2].i64,
            _src[0][3].i64,
            _src[0][4].i64,
            _src[0][5].i64,
            _src[0][6].i64,
            _src[0][7].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int64_vec src1 = {
            _src[1][0].i64,
            _src[1][1].i64,
            _src[1][2].i64,
            _src[1][3].i64,
            _src[1][4].i64,
            _src[1][5].i64,
            _src[1][6].i64,
            _src[1][7].i64,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct bool1_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x));

            _dst_val[0].b = -(int)dst.x;


         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bcsel(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool1_t src0 =
                  _src[0][_i].b;
               const uint1_t src1 =
                  _src[1][_i].b;
               const uint1_t src2 =
                  _src[2][_i].b;

         
            uint1_t dst = src0 ? src1 : src2;

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool1_t src0 =
                  _src[0][_i].b;
               const uint8_t src1 =
                  _src[1][_i].u8;
               const uint8_t src2 =
                  _src[2][_i].u8;

         
            uint8_t dst = src0 ? src1 : src2;

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool1_t src0 =
                  _src[0][_i].b;
               const uint16_t src1 =
                  _src[1][_i].u16;
               const uint16_t src2 =
                  _src[2][_i].u16;

         
            uint16_t dst = src0 ? src1 : src2;

            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool1_t src0 =
                  _src[0][_i].b;
               const uint32_t src1 =
                  _src[1][_i].u32;
               const uint32_t src2 =
                  _src[2][_i].u32;

         
            uint32_t dst = src0 ? src1 : src2;

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const bool1_t src0 =
                  _src[0][_i].b;
               const uint64_t src1 =
                  _src[1][_i].u64;
               const uint64_t src2 =
                  _src[2][_i].u64;

         
            uint64_t dst = src0 ? src1 : src2;

            _dst_val[_i].u64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bf2f(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint16_t src0 =
                  _src[0][_i].u16;

         
            float32_t dst = _mesa_bfloat16_bits_to_float(src0);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_bfdot16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct uint16_vec src0 = {
            _src[0][0].u16,
            _src[0][1].u16,
            _src[0][2].u16,
            _src[0][3].u16,
            _src[0][4].u16,
            _src[0][5].u16,
            _src[0][6].u16,
            _src[0][7].u16,
            _src[0][8].u16,
            _src[0][9].u16,
            _src[0][10].u16,
            _src[0][11].u16,
            _src[0][12].u16,
            _src[0][13].u16,
            _src[0][14].u16,
            _src[0][15].u16,
      };

      const struct uint16_vec src1 = {
            _src[1][0].u16,
            _src[1][1].u16,
            _src[1][2].u16,
            _src[1][3].u16,
            _src[1][4].u16,
            _src[1][5].u16,
            _src[1][6].u16,
            _src[1][7].u16,
            _src[1][8].u16,
            _src[1][9].u16,
            _src[1][10].u16,
            _src[1][11].u16,
            _src[1][12].u16,
            _src[1][13].u16,
            _src[1][14].u16,
            _src[1][15].u16,
      };

      struct uint16_vec dst;

         dst.x = dst.y = dst.z = dst.w = _mesa_float_to_bfloat16_bits_rte((_mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.p) * _mesa_bfloat16_bits_to_float(src1.p))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.o) * _mesa_bfloat16_bits_to_float(src1.o)))) + _mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.n) * _mesa_bfloat16_bits_to_float(src1.n))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.m) * _mesa_bfloat16_bits_to_float(src1.m))))) + _mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.l) * _mesa_bfloat16_bits_to_float(src1.l))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.k) * _mesa_bfloat16_bits_to_float(src1.k)))) + _mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.j) * _mesa_bfloat16_bits_to_float(src1.j))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.i) * _mesa_bfloat16_bits_to_float(src1.i)))))) + _mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.h) * _mesa_bfloat16_bits_to_float(src1.h))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.g) * _mesa_bfloat16_bits_to_float(src1.g)))) + _mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.f) * _mesa_bfloat16_bits_to_float(src1.f))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.e) * _mesa_bfloat16_bits_to_float(src1.e))))) + _mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.w) * _mesa_bfloat16_bits_to_float(src1.w))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.z) * _mesa_bfloat16_bits_to_float(src1.z)))) + _mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.y) * _mesa_bfloat16_bits_to_float(src1.y))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.x) * _mesa_bfloat16_bits_to_float(src1.x))))))));

            _dst_val[0].u16 = dst.x;



   return poison_mask;
}
static nir_component_mask_t
evaluate_bfdot2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct uint16_vec src0 = {
            _src[0][0].u16,
            _src[0][1].u16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct uint16_vec src1 = {
            _src[1][0].u16,
            _src[1][1].u16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct uint16_vec dst;

         dst.x = dst.y = dst.z = dst.w = _mesa_float_to_bfloat16_bits_rte((_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.y) * _mesa_bfloat16_bits_to_float(src1.y))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.x) * _mesa_bfloat16_bits_to_float(src1.x)))));

            _dst_val[0].u16 = dst.x;



   return poison_mask;
}
static nir_component_mask_t
evaluate_bfdot2_bfadd(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct int16_vec src0 = {
            _src[0][0].i16,
            _src[0][1].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src1 = {
            _src[1][0].i16,
            _src[1][1].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct int16_vec src2 = {
            _src[2][0].i16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct int16_vec dst;

         
   const float a0 = _mesa_bfloat16_bits_to_float(src0.x);
   const float a1 = _mesa_bfloat16_bits_to_float(src0.y);
   const float b0 = _mesa_bfloat16_bits_to_float(src1.x);
   const float b1 = _mesa_bfloat16_bits_to_float(src1.y);

   float acc = _mesa_bfloat16_bits_to_float(src2.x);
   acc = fmaf(a0, b0, acc);
   acc = fmaf(a1, b1, acc);

   dst.x = _mesa_float_to_bfloat16_bits_rte(acc);


            _dst_val[0].i16 = dst.x;



   return poison_mask;
}
static nir_component_mask_t
evaluate_bfdot3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct uint16_vec src0 = {
            _src[0][0].u16,
            _src[0][1].u16,
            _src[0][2].u16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct uint16_vec src1 = {
            _src[1][0].u16,
            _src[1][1].u16,
            _src[1][2].u16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct uint16_vec dst;

         dst.x = dst.y = dst.z = dst.w = _mesa_float_to_bfloat16_bits_rte((_mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.z) * _mesa_bfloat16_bits_to_float(src1.z))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.y) * _mesa_bfloat16_bits_to_float(src1.y)))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.x) * _mesa_bfloat16_bits_to_float(src1.x)))));

            _dst_val[0].u16 = dst.x;



   return poison_mask;
}
static nir_component_mask_t
evaluate_bfdot4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct uint16_vec src0 = {
            _src[0][0].u16,
            _src[0][1].u16,
            _src[0][2].u16,
            _src[0][3].u16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct uint16_vec src1 = {
            _src[1][0].u16,
            _src[1][1].u16,
            _src[1][2].u16,
            _src[1][3].u16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct uint16_vec dst;

         dst.x = dst.y = dst.z = dst.w = _mesa_float_to_bfloat16_bits_rte((_mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.w) * _mesa_bfloat16_bits_to_float(src1.w))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.z) * _mesa_bfloat16_bits_to_float(src1.z)))) + _mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.y) * _mesa_bfloat16_bits_to_float(src1.y))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.x) * _mesa_bfloat16_bits_to_float(src1.x))))));

            _dst_val[0].u16 = dst.x;



   return poison_mask;
}
static nir_component_mask_t
evaluate_bfdot5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct uint16_vec src0 = {
            _src[0][0].u16,
            _src[0][1].u16,
            _src[0][2].u16,
            _src[0][3].u16,
            _src[0][4].u16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct uint16_vec src1 = {
            _src[1][0].u16,
            _src[1][1].u16,
            _src[1][2].u16,
            _src[1][3].u16,
            _src[1][4].u16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct uint16_vec dst;

         dst.x = dst.y = dst.z = dst.w = _mesa_float_to_bfloat16_bits_rte((_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.e) * _mesa_bfloat16_bits_to_float(src1.e))) + _mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.w) * _mesa_bfloat16_bits_to_float(src1.w))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.z) * _mesa_bfloat16_bits_to_float(src1.z)))) + _mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.y) * _mesa_bfloat16_bits_to_float(src1.y))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.x) * _mesa_bfloat16_bits_to_float(src1.x)))))));

            _dst_val[0].u16 = dst.x;



   return poison_mask;
}
static nir_component_mask_t
evaluate_bfdot8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct uint16_vec src0 = {
            _src[0][0].u16,
            _src[0][1].u16,
            _src[0][2].u16,
            _src[0][3].u16,
            _src[0][4].u16,
            _src[0][5].u16,
            _src[0][6].u16,
            _src[0][7].u16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct uint16_vec src1 = {
            _src[1][0].u16,
            _src[1][1].u16,
            _src[1][2].u16,
            _src[1][3].u16,
            _src[1][4].u16,
            _src[1][5].u16,
            _src[1][6].u16,
            _src[1][7].u16,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct uint16_vec dst;

         dst.x = dst.y = dst.z = dst.w = _mesa_float_to_bfloat16_bits_rte((_mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.h) * _mesa_bfloat16_bits_to_float(src1.h))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.g) * _mesa_bfloat16_bits_to_float(src1.g)))) + _mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.f) * _mesa_bfloat16_bits_to_float(src1.f))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.e) * _mesa_bfloat16_bits_to_float(src1.e))))) + _mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.w) * _mesa_bfloat16_bits_to_float(src1.w))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.z) * _mesa_bfloat16_bits_to_float(src1.z)))) + _mesa_bfloat16_bits_to_float(_mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.y) * _mesa_bfloat16_bits_to_float(src1.y))) + _mesa_bfloat16_bits_to_float((_mesa_bfloat16_bits_to_float(src0.x) * _mesa_bfloat16_bits_to_float(src1.x)))))));

            _dst_val[0].u16 = dst.x;



   return poison_mask;
}
static nir_component_mask_t
evaluate_bffma(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint16_t src0 =
                  _src[0][_i].u16;
               const uint16_t src1 =
                  _src[1][_i].u16;
               const uint16_t src2 =
                  _src[2][_i].u16;

         
            uint16_t dst;

            
    const float a = _mesa_bfloat16_bits_to_float(src0);
    const float b = _mesa_bfloat16_bits_to_float(src1);
    const float c = _mesa_bfloat16_bits_to_float(src2);
    dst = _mesa_float_to_bfloat16_bits_rte(fmaf(a, b, c));


            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_bfi(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;
               const uint32_t src1 =
                  _src[1][_i].u32;
               const uint32_t src2 =
                  _src[2][_i].u32;

         
            uint32_t dst;

            
unsigned mask = src0, insert = src1, base = src2;
if (mask == 0) {
   dst = base;
} else {
   unsigned tmp = mask;
   while (!(tmp & 1)) {
      tmp >>= 1;
      insert <<= 1;
   }
   dst = (base & ~mask) | (insert & mask);
}


            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_bfm(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int32_t src0 =
                  _src[0][_i].i32;
               const int32_t src1 =
                  _src[1][_i].i32;

         
            uint32_t dst;

            
int bits = src0 & 0x1F;
int offset = src1 & 0x1F;
dst = ((1u << bits) - 1) << offset;


            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_bfmul(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint16_t src0 =
                  _src[0][_i].u16;
               const uint16_t src1 =
                  _src[1][_i].u16;

         
            uint16_t dst;

            
   const float a = _mesa_bfloat16_bits_to_float(src0);
   const float b = _mesa_bfloat16_bits_to_float(src1);
   dst = _mesa_float_to_bfloat16_bits_rte(a * b);


            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_bit_count(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint1_t src0 =
                  _src[0][_i].b;

         
            uint32_t dst;

            
dst = 0;
for (unsigned bit = 0; bit < bit_size; bit++) {
   if ((src0 >> bit) & 1)
      dst++;
}


            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint8_t src0 =
                  _src[0][_i].u8;

         
            uint32_t dst;

            
dst = 0;
for (unsigned bit = 0; bit < bit_size; bit++) {
   if ((src0 >> bit) & 1)
      dst++;
}


            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint16_t src0 =
                  _src[0][_i].u16;

         
            uint32_t dst;

            
dst = 0;
for (unsigned bit = 0; bit < bit_size; bit++) {
   if ((src0 >> bit) & 1)
      dst++;
}


            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;

         
            uint32_t dst;

            
dst = 0;
for (unsigned bit = 0; bit < bit_size; bit++) {
   if ((src0 >> bit) & 1)
      dst++;
}


            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint64_t src0 =
                  _src[0][_i].u64;

         
            uint32_t dst;

            
dst = 0;
for (unsigned bit = 0; bit < bit_size; bit++) {
   if ((src0 >> bit) & 1)
      dst++;
}


            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bitfield_insert(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                                    
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint1_t src0 =
                  _src[0][_i].b;
               const uint1_t src1 =
                  _src[1][_i].b;
               const int32_t src2 =
                  _src[2][_i].i32;
               const int32_t src3 =
                  _src[3][_i].i32;

         
            uint1_t dst;

            
unsigned base = src0, insert = src1;
int offset = src2, bits = src3;
if (bits == 0) {
   dst = base;
} else if (offset < 0 || bits < 0 || bits + offset > 32) {
   dst = 0;
   poison = true;
} else {
   unsigned mask = ((1ull << bits) - 1) << offset;
   dst = (base & ~mask) | ((insert << offset) & mask);
}


            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                                    
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint8_t src0 =
                  _src[0][_i].u8;
               const uint8_t src1 =
                  _src[1][_i].u8;
               const int32_t src2 =
                  _src[2][_i].i32;
               const int32_t src3 =
                  _src[3][_i].i32;

         
            uint8_t dst;

            
unsigned base = src0, insert = src1;
int offset = src2, bits = src3;
if (bits == 0) {
   dst = base;
} else if (offset < 0 || bits < 0 || bits + offset > 32) {
   dst = 0;
   poison = true;
} else {
   unsigned mask = ((1ull << bits) - 1) << offset;
   dst = (base & ~mask) | ((insert << offset) & mask);
}


            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                                    
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint16_t src0 =
                  _src[0][_i].u16;
               const uint16_t src1 =
                  _src[1][_i].u16;
               const int32_t src2 =
                  _src[2][_i].i32;
               const int32_t src3 =
                  _src[3][_i].i32;

         
            uint16_t dst;

            
unsigned base = src0, insert = src1;
int offset = src2, bits = src3;
if (bits == 0) {
   dst = base;
} else if (offset < 0 || bits < 0 || bits + offset > 32) {
   dst = 0;
   poison = true;
} else {
   unsigned mask = ((1ull << bits) - 1) << offset;
   dst = (base & ~mask) | ((insert << offset) & mask);
}


            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                                    
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;
               const uint32_t src1 =
                  _src[1][_i].u32;
               const int32_t src2 =
                  _src[2][_i].i32;
               const int32_t src3 =
                  _src[3][_i].i32;

         
            uint32_t dst;

            
unsigned base = src0, insert = src1;
int offset = src2, bits = src3;
if (bits == 0) {
   dst = base;
} else if (offset < 0 || bits < 0 || bits + offset > 32) {
   dst = 0;
   poison = true;
} else {
   unsigned mask = ((1ull << bits) - 1) << offset;
   dst = (base & ~mask) | ((insert << offset) & mask);
}


            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                                    
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint64_t src0 =
                  _src[0][_i].u64;
               const uint64_t src1 =
                  _src[1][_i].u64;
               const int32_t src2 =
                  _src[2][_i].i32;
               const int32_t src3 =
                  _src[3][_i].i32;

         
            uint64_t dst;

            
unsigned base = src0, insert = src1;
int offset = src2, bits = src3;
if (bits == 0) {
   dst = base;
} else if (offset < 0 || bits < 0 || bits + offset > 32) {
   dst = 0;
   poison = true;
} else {
   unsigned mask = ((1ull << bits) - 1) << offset;
   dst = (base & ~mask) | ((insert << offset) & mask);
}


            _dst_val[_i].u64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bitfield_reverse(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint1_t src0 =
                  _src[0][_i].b;

         
            uint1_t dst;

            
/* we're not winning any awards for speed here, but that's ok */
dst = 0;
for (unsigned bit = 0; bit < bit_size; bit++)
   dst |= ((src0 >> bit) & 1) << ((bit_size - 1) - bit);


            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint8_t src0 =
                  _src[0][_i].u8;

         
            uint8_t dst;

            
/* we're not winning any awards for speed here, but that's ok */
dst = 0;
for (unsigned bit = 0; bit < bit_size; bit++)
   dst |= ((src0 >> bit) & 1) << ((bit_size - 1) - bit);


            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint16_t src0 =
                  _src[0][_i].u16;

         
            uint16_t dst;

            
/* we're not winning any awards for speed here, but that's ok */
dst = 0;
for (unsigned bit = 0; bit < bit_size; bit++)
   dst |= ((src0 >> bit) & 1) << ((bit_size - 1) - bit);


            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;

         
            uint32_t dst;

            
/* we're not winning any awards for speed here, but that's ok */
dst = 0;
for (unsigned bit = 0; bit < bit_size; bit++)
   dst |= ((src0 >> bit) & 1) << ((bit_size - 1) - bit);


            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint64_t src0 =
                  _src[0][_i].u64;

         
            uint64_t dst;

            
/* we're not winning any awards for speed here, but that's ok */
dst = 0;
for (unsigned bit = 0; bit < bit_size; bit++)
   dst |= ((src0 >> bit) & 1) << ((bit_size - 1) - bit);


            _dst_val[_i].u64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bitfield_select(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint1_t src0 =
                  _src[0][_i].b;
               const uint1_t src1 =
                  _src[1][_i].b;
               const uint1_t src2 =
                  _src[2][_i].b;

         
            uint1_t dst = (src0 & src1) | (~src0 & src2);

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint8_t src0 =
                  _src[0][_i].u8;
               const uint8_t src1 =
                  _src[1][_i].u8;
               const uint8_t src2 =
                  _src[2][_i].u8;

         
            uint8_t dst = (src0 & src1) | (~src0 & src2);

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint16_t src0 =
                  _src[0][_i].u16;
               const uint16_t src1 =
                  _src[1][_i].u16;
               const uint16_t src2 =
                  _src[2][_i].u16;

         
            uint16_t dst = (src0 & src1) | (~src0 & src2);

            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;
               const uint32_t src1 =
                  _src[1][_i].u32;
               const uint32_t src2 =
                  _src[2][_i].u32;

         
            uint32_t dst = (src0 & src1) | (~src0 & src2);

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint64_t src0 =
                  _src[0][_i].u64;
               const uint64_t src1 =
                  _src[1][_i].u64;
               const uint64_t src2 =
                  _src[2][_i].u64;

         
            uint64_t dst = (src0 & src1) | (~src0 & src2);

            _dst_val[_i].u64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bitnz(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint1_t src0 =
                  _src[0][_i].b;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool1_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint8_t src0 =
                  _src[0][_i].u8;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool1_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint16_t src0 =
                  _src[0][_i].u16;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool1_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool1_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint64_t src0 =
                  _src[0][_i].u64;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool1_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bitnz16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint1_t src0 =
                  _src[0][_i].b;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool16_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint8_t src0 =
                  _src[0][_i].u8;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool16_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint16_t src0 =
                  _src[0][_i].u16;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool16_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool16_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint64_t src0 =
                  _src[0][_i].u64;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool16_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bitnz32(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint1_t src0 =
                  _src[0][_i].b;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool32_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint8_t src0 =
                  _src[0][_i].u8;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool32_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint16_t src0 =
                  _src[0][_i].u16;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool32_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool32_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint64_t src0 =
                  _src[0][_i].u64;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool32_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bitnz8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint1_t src0 =
                  _src[0][_i].b;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool8_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint8_t src0 =
                  _src[0][_i].u8;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool8_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint16_t src0 =
                  _src[0][_i].u16;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool8_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool8_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint64_t src0 =
                  _src[0][_i].u64;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool8_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bitz(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint1_t src0 =
                  _src[0][_i].b;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool1_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint8_t src0 =
                  _src[0][_i].u8;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool1_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint16_t src0 =
                  _src[0][_i].u16;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool1_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool1_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint64_t src0 =
                  _src[0][_i].u64;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool1_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bitz16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint1_t src0 =
                  _src[0][_i].b;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool16_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint8_t src0 =
                  _src[0][_i].u8;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool16_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint16_t src0 =
                  _src[0][_i].u16;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool16_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool16_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint64_t src0 =
                  _src[0][_i].u64;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool16_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bitz32(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint1_t src0 =
                  _src[0][_i].b;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool32_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint8_t src0 =
                  _src[0][_i].u8;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool32_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint16_t src0 =
                  _src[0][_i].u16;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool32_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool32_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint64_t src0 =
                  _src[0][_i].u64;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool32_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bitz8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint1_t src0 =
                  _src[0][_i].b;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool8_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint8_t src0 =
                  _src[0][_i].u8;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool8_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint16_t src0 =
                  _src[0][_i].u16;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool8_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool8_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint64_t src0 =
                  _src[0][_i].u64;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            bool8_t dst = ((uint64_t)src0 >> (src1 & (bit_size - 1)) & 0x1) == 0x0;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_bounds_agx(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               /* 1-bit integers use a 0/-1 convention */
               const int1_t src0 = -(int1_t)_src[0][_i].b;
               /* 1-bit integers use a 0/-1 convention */
               const int1_t src1 = -(int1_t)_src[1][_i].b;
               /* 1-bit integers use a 0/-1 convention */
               const int1_t src2 = -(int1_t)_src[2][_i].b;

         
            int1_t dst = src1 <= src2 ? src0 : 0;

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int8_t src0 =
                  _src[0][_i].i8;
               const int8_t src1 =
                  _src[1][_i].i8;
               const int8_t src2 =
                  _src[2][_i].i8;

         
            int8_t dst = src1 <= src2 ? src0 : 0;

            _dst_val[_i].i8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int16_t src0 =
                  _src[0][_i].i16;
               const int16_t src1 =
                  _src[1][_i].i16;
               const int16_t src2 =
                  _src[2][_i].i16;

         
            int16_t dst = src1 <= src2 ? src0 : 0;

            _dst_val[_i].i16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int32_t src0 =
                  _src[0][_i].i32;
               const int32_t src1 =
                  _src[1][_i].i32;
               const int32_t src2 =
                  _src[2][_i].i32;

         
            int32_t dst = src1 <= src2 ? src0 : 0;

            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int64_t src0 =
                  _src[0][_i].i64;
               const int64_t src1 =
                  _src[1][_i].i64;
               const int64_t src2 =
                  _src[2][_i].i64;

         
            int64_t dst = src1 <= src2 ? src0 : 0;

            _dst_val[_i].i64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_byte_perm_amd(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;
               const uint32_t src1 =
                  _src[1][_i].u32;
               const uint32_t src2 =
                  _src[2][_i].u32;

         
            uint32_t dst;

            
   uint64_t src = src1 | ((uint64_t)src0 << 32);
   dst = 0;
   for (unsigned i = 0; i < 4; i++) {
      uint8_t sel = (src2 >> (i * 8)) & 0xff;
      unsigned res;
      if (sel >= 13) {
         res = 0xff;
      } else if (sel == 12) {
         res = 0;
      } else if (sel >= 8) {
         res = ((src >> (((sel - 8) * 2 + 1) * 8 + 7)) & 1) * 0xff;
      } else {
         res = (src >> (sel * 8)) & 0xff;
      }
      dst |= res << (i * 8);
   }


            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_cube_amd(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float32_vec dst;

         
   dst.x = dst.y = dst.z = 0.0;
   float absX = fabsf(src0.x);
   float absY = fabsf(src0.y);
   float absZ = fabsf(src0.z);

   if (absX >= absY && absX >= absZ) { dst.z = 2 * src0.x; }
   if (absY >= absX && absY >= absZ) { dst.z = 2 * src0.y; }
   if (absZ >= absX && absZ >= absY) { dst.z = 2 * src0.z; }

   if (src0.x >= 0 && absX >= absY && absX >= absZ) {
      dst.y = -src0.z; dst.x = -src0.y; dst.w = 0;
   }
   if (src0.x < 0 && absX >= absY && absX >= absZ) {
      dst.y = src0.z; dst.x = -src0.y; dst.w = 1;
   }
   if (src0.y >= 0 && absY >= absX && absY >= absZ) {
      dst.y = src0.x; dst.x = src0.z; dst.w = 2;
   }
   if (src0.y < 0 && absY >= absX && absY >= absZ) {
      dst.y = src0.x; dst.x = -src0.z; dst.w = 3;
   }
   if (src0.z >= 0 && absZ >= absX && absZ >= absY) {
      dst.y = src0.x; dst.x = -src0.y; dst.w = 4;
   }
   if (src0.z < 0 && absZ >= absX && absZ >= absY) {
      dst.y = -src0.x; dst.x = -src0.y; dst.w = 5;
   }


            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }
            _dst_val[1].f32 = dst.y;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[1], 32);
               }
            _dst_val[2].f32 = dst.z;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[2], 32);
               }
            _dst_val[3].f32 = dst.w;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[3], 32);
               }


   return poison_mask;
}
static nir_component_mask_t
evaluate_e4m3fn2f(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint8_t src0 =
                  _src[0][_i].u8;

         
            float32_t dst = _mesa_e4m3fn_to_float(src0);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_e5m22f(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint8_t src0 =
                  _src[0][_i].u8;

         
            float32_t dst = _mesa_e5m2_to_float(src0);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_extr_agx(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

                                    
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;
               const uint32_t src1 =
                  _src[1][_i].u32;
               const uint32_t src2 =
                  _src[2][_i].u32;
               const uint32_t src3 =
                  _src[3][_i].u32;

         
            uint32_t dst;

            
    uint32_t mask = 0xFFFFFFFF;
    uint8_t shift = src2 & 0x7F;
    if (src3 != 0) {
       mask = (1 << src3) - 1;
    }
    if (shift >= 64) {
        dst = 0;
    } else {
        dst = (((((uint64_t) src1) << 32) | (uint64_t) src0) >> shift) & mask;
    }


            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_extract_i16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               /* 1-bit integers use a 0/-1 convention */
               const int1_t src0 = -(int1_t)_src[0][_i].b;
               /* 1-bit integers use a 0/-1 convention */
               const int1_t src1 = -(int1_t)_src[1][_i].b;

         
            int1_t dst;

            
         if ((uint64_t)(src1 * 16) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (int16_t)(src0 >> (src1 * 16));
         }
         

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int8_t src0 =
                  _src[0][_i].i8;
               const int8_t src1 =
                  _src[1][_i].i8;

         
            int8_t dst;

            
         if ((uint64_t)(src1 * 16) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (int16_t)(src0 >> (src1 * 16));
         }
         

            _dst_val[_i].i8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int16_t src0 =
                  _src[0][_i].i16;
               const int16_t src1 =
                  _src[1][_i].i16;

         
            int16_t dst;

            
         if ((uint64_t)(src1 * 16) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (int16_t)(src0 >> (src1 * 16));
         }
         

            _dst_val[_i].i16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int32_t src0 =
                  _src[0][_i].i32;
               const int32_t src1 =
                  _src[1][_i].i32;

         
            int32_t dst;

            
         if ((uint64_t)(src1 * 16) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (int16_t)(src0 >> (src1 * 16));
         }
         

            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int64_t src0 =
                  _src[0][_i].i64;
               const int64_t src1 =
                  _src[1][_i].i64;

         
            int64_t dst;

            
         if ((uint64_t)(src1 * 16) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (int16_t)(src0 >> (src1 * 16));
         }
         

            _dst_val[_i].i64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_extract_i8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               /* 1-bit integers use a 0/-1 convention */
               const int1_t src0 = -(int1_t)_src[0][_i].b;
               /* 1-bit integers use a 0/-1 convention */
               const int1_t src1 = -(int1_t)_src[1][_i].b;

         
            int1_t dst;

            
         if ((uint64_t)(src1 * 8) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (int8_t)(src0 >> (src1 * 8));
         }
         

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int8_t src0 =
                  _src[0][_i].i8;
               const int8_t src1 =
                  _src[1][_i].i8;

         
            int8_t dst;

            
         if ((uint64_t)(src1 * 8) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (int8_t)(src0 >> (src1 * 8));
         }
         

            _dst_val[_i].i8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int16_t src0 =
                  _src[0][_i].i16;
               const int16_t src1 =
                  _src[1][_i].i16;

         
            int16_t dst;

            
         if ((uint64_t)(src1 * 8) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (int8_t)(src0 >> (src1 * 8));
         }
         

            _dst_val[_i].i16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int32_t src0 =
                  _src[0][_i].i32;
               const int32_t src1 =
                  _src[1][_i].i32;

         
            int32_t dst;

            
         if ((uint64_t)(src1 * 8) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (int8_t)(src0 >> (src1 * 8));
         }
         

            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int64_t src0 =
                  _src[0][_i].i64;
               const int64_t src1 =
                  _src[1][_i].i64;

         
            int64_t dst;

            
         if ((uint64_t)(src1 * 8) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (int8_t)(src0 >> (src1 * 8));
         }
         

            _dst_val[_i].i64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_extract_u16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint1_t src0 =
                  _src[0][_i].b;
               const uint1_t src1 =
                  _src[1][_i].b;

         
            uint1_t dst;

            
         if ((uint64_t)(src1 * 16) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (uint16_t)(src0 >> (src1 * 16));
         }
         

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint8_t src0 =
                  _src[0][_i].u8;
               const uint8_t src1 =
                  _src[1][_i].u8;

         
            uint8_t dst;

            
         if ((uint64_t)(src1 * 16) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (uint16_t)(src0 >> (src1 * 16));
         }
         

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint16_t src0 =
                  _src[0][_i].u16;
               const uint16_t src1 =
                  _src[1][_i].u16;

         
            uint16_t dst;

            
         if ((uint64_t)(src1 * 16) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (uint16_t)(src0 >> (src1 * 16));
         }
         

            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            uint32_t dst;

            
         if ((uint64_t)(src1 * 16) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (uint16_t)(src0 >> (src1 * 16));
         }
         

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint64_t src0 =
                  _src[0][_i].u64;
               const uint64_t src1 =
                  _src[1][_i].u64;

         
            uint64_t dst;

            
         if ((uint64_t)(src1 * 16) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (uint16_t)(src0 >> (src1 * 16));
         }
         

            _dst_val[_i].u64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_extract_u8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint1_t src0 =
                  _src[0][_i].b;
               const uint1_t src1 =
                  _src[1][_i].b;

         
            uint1_t dst;

            
         if ((uint64_t)(src1 * 8) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (uint8_t)(src0 >> (src1 * 8));
         }
         

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint8_t src0 =
                  _src[0][_i].u8;
               const uint8_t src1 =
                  _src[1][_i].u8;

         
            uint8_t dst;

            
         if ((uint64_t)(src1 * 8) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (uint8_t)(src0 >> (src1 * 8));
         }
         

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint16_t src0 =
                  _src[0][_i].u16;
               const uint16_t src1 =
                  _src[1][_i].u16;

         
            uint16_t dst;

            
         if ((uint64_t)(src1 * 8) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (uint8_t)(src0 >> (src1 * 8));
         }
         

            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;
               const uint32_t src1 =
                  _src[1][_i].u32;

         
            uint32_t dst;

            
         if ((uint64_t)(src1 * 8) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (uint8_t)(src0 >> (src1 * 8));
         }
         

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint64_t src0 =
                  _src[0][_i].u64;
               const uint64_t src1 =
                  _src[1][_i].u64;

         
            uint64_t dst;

            
         if ((uint64_t)(src1 * 8) >= bit_size) {
            dst = 0;
            poison = true;
         } else {
            dst = (uint8_t)(src0 >> (src1 * 8));
         }
         

            _dst_val[_i].u64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2bf(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            uint16_t dst = _mesa_float_to_bfloat16_bits_rte(src0);

            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_f2e4m3fn(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            uint8_t dst = _mesa_float_to_e4m3fn(src0);

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_f2e4m3fn_sat(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            uint8_t dst = _mesa_float_to_e4m3fn_sat(src0);

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_f2e4m3fn_satfn(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            uint8_t dst = isinf(src0) ? 0x7f : _mesa_float_to_e4m3fn_sat(src0);

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_f2e5m2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            uint8_t dst = _mesa_float_to_e5m2(src0);

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_f2e5m2_sat(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            uint8_t dst = _mesa_float_to_e5m2_sat(src0);

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_f2f16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst;

            
                      if (bit_size > 32) {
                         if (nir_is_rounding_mode_rtz(execution_mode, 16))
                            dst = _mesa_half_to_float(_mesa_double_to_float16_rtz(src0));
                         else
                            dst = _mesa_half_to_float(_mesa_double_to_float16_rtne(src0));
                      } else if (bit_size > 16) {
                         if (nir_is_rounding_mode_rtz(execution_mode, 16))
                            dst = _mesa_half_to_float(_mesa_float_to_float16_rtz(src0));
                         else
                            dst = _mesa_half_to_float(_mesa_float_to_float16_rtne(src0));
                      } else {
                         dst = src0;
                      }
                      

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float16_t dst;

            
                      if (bit_size > 32) {
                         if (nir_is_rounding_mode_rtz(execution_mode, 16))
                            dst = _mesa_half_to_float(_mesa_double_to_float16_rtz(src0));
                         else
                            dst = _mesa_half_to_float(_mesa_double_to_float16_rtne(src0));
                      } else if (bit_size > 16) {
                         if (nir_is_rounding_mode_rtz(execution_mode, 16))
                            dst = _mesa_half_to_float(_mesa_float_to_float16_rtz(src0));
                         else
                            dst = _mesa_half_to_float(_mesa_float_to_float16_rtne(src0));
                      } else {
                         dst = src0;
                      }
                      

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float16_t dst;

            
                      if (bit_size > 32) {
                         if (nir_is_rounding_mode_rtz(execution_mode, 16))
                            dst = _mesa_half_to_float(_mesa_double_to_float16_rtz(src0));
                         else
                            dst = _mesa_half_to_float(_mesa_double_to_float16_rtne(src0));
                      } else if (bit_size > 16) {
                         if (nir_is_rounding_mode_rtz(execution_mode, 16))
                            dst = _mesa_half_to_float(_mesa_float_to_float16_rtz(src0));
                         else
                            dst = _mesa_half_to_float(_mesa_float_to_float16_rtne(src0));
                      } else {
                         dst = src0;
                      }
                      

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2f16_rd(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst;

            
                      if (bit_size > 16) {
                         dst = _mesa_half_to_float(_mesa_float_to_float16_rd(src0));
                      } else {
                         dst = src0;
                      }
                      

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float16_t dst;

            
                      if (bit_size > 16) {
                         dst = _mesa_half_to_float(_mesa_float_to_float16_rd(src0));
                      } else {
                         dst = src0;
                      }
                      

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float16_t dst;

            
                      if (bit_size > 16) {
                         dst = _mesa_half_to_float(_mesa_float_to_float16_rd(src0));
                      } else {
                         dst = src0;
                      }
                      

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2f16_rtne(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst;

            
                      if (bit_size > 32) {
                         dst = _mesa_half_to_float(_mesa_double_to_float16_rtne(src0));
                      } else if (bit_size > 16) {
                         dst = _mesa_half_to_float(_mesa_float_to_float16_rtne(src0));
                      } else {
                         dst = src0;
                      }
                      

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float16_t dst;

            
                      if (bit_size > 32) {
                         dst = _mesa_half_to_float(_mesa_double_to_float16_rtne(src0));
                      } else if (bit_size > 16) {
                         dst = _mesa_half_to_float(_mesa_float_to_float16_rtne(src0));
                      } else {
                         dst = src0;
                      }
                      

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float16_t dst;

            
                      if (bit_size > 32) {
                         dst = _mesa_half_to_float(_mesa_double_to_float16_rtne(src0));
                      } else if (bit_size > 16) {
                         dst = _mesa_half_to_float(_mesa_float_to_float16_rtne(src0));
                      } else {
                         dst = src0;
                      }
                      

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2f16_rtz(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst;

            
                      if (bit_size > 32) {
                         dst = _mesa_half_to_float(_mesa_double_to_float16_rtz(src0));
                      } else if (bit_size > 16) {
                         dst = _mesa_half_to_float(_mesa_float_to_float16_rtz(src0));
                      } else {
                         dst = src0;
                      }
                      

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float16_t dst;

            
                      if (bit_size > 32) {
                         dst = _mesa_half_to_float(_mesa_double_to_float16_rtz(src0));
                      } else if (bit_size > 16) {
                         dst = _mesa_half_to_float(_mesa_float_to_float16_rtz(src0));
                      } else {
                         dst = src0;
                      }
                      

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float16_t dst;

            
                      if (bit_size > 32) {
                         dst = _mesa_half_to_float(_mesa_double_to_float16_rtz(src0));
                      } else if (bit_size > 16) {
                         dst = _mesa_half_to_float(_mesa_float_to_float16_rtz(src0));
                      } else {
                         dst = src0;
                      }
                      

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2f16_ru(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst;

            
                      if (bit_size > 16) {
                         dst = _mesa_half_to_float(_mesa_float_to_float16_ru(src0));
                      } else {
                         dst = src0;
                      }
                      

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float16_t dst;

            
                      if (bit_size > 16) {
                         dst = _mesa_half_to_float(_mesa_float_to_float16_ru(src0));
                      } else {
                         dst = src0;
                      }
                      

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float16_t dst;

            
                      if (bit_size > 16) {
                         dst = _mesa_half_to_float(_mesa_float_to_float16_ru(src0));
                      } else {
                         dst = src0;
                      }
                      

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2f32(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float32_t dst;

            
              if (bit_size > 32 && nir_is_rounding_mode_rtz(execution_mode, 32)) {
                 dst = _mesa_double_to_float_rtz(src0);
              } else {
                 dst = src0;
              }
              

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float32_t dst;

            
              if (bit_size > 32 && nir_is_rounding_mode_rtz(execution_mode, 32)) {
                 dst = _mesa_double_to_float_rtz(src0);
              } else {
                 dst = src0;
              }
              

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float32_t dst;

            
              if (bit_size > 32 && nir_is_rounding_mode_rtz(execution_mode, 32)) {
                 dst = _mesa_double_to_float_rtz(src0);
              } else {
                 dst = src0;
              }
              

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2f64(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float64_t dst = src0;

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float64_t dst = src0;

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float64_t dst = src0;

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2fmp(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float16_t dst;

            
                      if (bit_size > 32) {
                         if (nir_is_rounding_mode_rtz(execution_mode, 16))
                            dst = _mesa_half_to_float(_mesa_double_to_float16_rtz(src0));
                         else
                            dst = _mesa_half_to_float(_mesa_double_to_float16_rtne(src0));
                      } else if (bit_size > 16) {
                         if (nir_is_rounding_mode_rtz(execution_mode, 16))
                            dst = _mesa_half_to_float(_mesa_float_to_float16_rtz(src0));
                         else
                            dst = _mesa_half_to_float(_mesa_float_to_float16_rtne(src0));
                      } else {
                         dst = src0;
                      }
                      

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_f2i1(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            int1_t dst;

            
                dst = src0;
                if (src0 < u_intN_min(1) || src0 > u_intN_max(1)) {
                   poison = true;
                }
              

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            int1_t dst;

            
                dst = src0;
                if (src0 < u_intN_min(1) || src0 > u_intN_max(1)) {
                   poison = true;
                }
              

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            int1_t dst;

            
                dst = src0;
                if (src0 < u_intN_min(1) || src0 > u_intN_max(1)) {
                   poison = true;
                }
              

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2i16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            int16_t dst;

            
                dst = src0;
                if (src0 < u_intN_min(16) || src0 > u_intN_max(16)) {
                   poison = true;
                }
              

            _dst_val[_i].i16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            int16_t dst;

            
                dst = src0;
                if (src0 < u_intN_min(16) || src0 > u_intN_max(16)) {
                   poison = true;
                }
              

            _dst_val[_i].i16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            int16_t dst;

            
                dst = src0;
                if (src0 < u_intN_min(16) || src0 > u_intN_max(16)) {
                   poison = true;
                }
              

            _dst_val[_i].i16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2i16_sat(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            int16_t dst = (int16_t) isnan(src0) ? 0.0 : fmin(fmax(src0, (double)u_intN_min(16)), (double)u_intN_max(16));

            _dst_val[_i].i16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            int16_t dst = (int16_t) isnan(src0) ? 0.0 : fmin(fmax(src0, (double)u_intN_min(16)), (double)u_intN_max(16));

            _dst_val[_i].i16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            int16_t dst = (int16_t) isnan(src0) ? 0.0 : fmin(fmax(src0, (double)u_intN_min(16)), (double)u_intN_max(16));

            _dst_val[_i].i16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2i32(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            int32_t dst;

            
                dst = src0;
                if (src0 < u_intN_min(32) || src0 > u_intN_max(32)) {
                   poison = true;
                }
              

            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            int32_t dst;

            
                dst = src0;
                if (src0 < u_intN_min(32) || src0 > u_intN_max(32)) {
                   poison = true;
                }
              

            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            int32_t dst;

            
                dst = src0;
                if (src0 < u_intN_min(32) || src0 > u_intN_max(32)) {
                   poison = true;
                }
              

            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2i32_rtne(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            int32_t dst = (int32_t)_mesa_roundevenf(src0);

            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_f2i32_sat(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            int32_t dst = (int32_t) isnan(src0) ? 0.0 : fmin(fmax(src0, (double)u_intN_min(32)), (double)u_intN_max(32));

            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            int32_t dst = (int32_t) isnan(src0) ? 0.0 : fmin(fmax(src0, (double)u_intN_min(32)), (double)u_intN_max(32));

            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            int32_t dst = (int32_t) isnan(src0) ? 0.0 : fmin(fmax(src0, (double)u_intN_min(32)), (double)u_intN_max(32));

            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2i64(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            int64_t dst;

            
                dst = src0;
                if (src0 < u_intN_min(64) || src0 > u_intN_max(64)) {
                   poison = true;
                }
              

            _dst_val[_i].i64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            int64_t dst;

            
                dst = src0;
                if (src0 < u_intN_min(64) || src0 > u_intN_max(64)) {
                   poison = true;
                }
              

            _dst_val[_i].i64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            int64_t dst;

            
                dst = src0;
                if (src0 < u_intN_min(64) || src0 > u_intN_max(64)) {
                   poison = true;
                }
              

            _dst_val[_i].i64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2i8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            int8_t dst;

            
                dst = src0;
                if (src0 < u_intN_min(8) || src0 > u_intN_max(8)) {
                   poison = true;
                }
              

            _dst_val[_i].i8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            int8_t dst;

            
                dst = src0;
                if (src0 < u_intN_min(8) || src0 > u_intN_max(8)) {
                   poison = true;
                }
              

            _dst_val[_i].i8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            int8_t dst;

            
                dst = src0;
                if (src0 < u_intN_min(8) || src0 > u_intN_max(8)) {
                   poison = true;
                }
              

            _dst_val[_i].i8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2i8_sat(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            int8_t dst = (int8_t) isnan(src0) ? 0.0 : fmin(fmax(src0, (double)u_intN_min(8)), (double)u_intN_max(8));

            _dst_val[_i].i8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            int8_t dst = (int8_t) isnan(src0) ? 0.0 : fmin(fmax(src0, (double)u_intN_min(8)), (double)u_intN_max(8));

            _dst_val[_i].i8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            int8_t dst = (int8_t) isnan(src0) ? 0.0 : fmin(fmax(src0, (double)u_intN_min(8)), (double)u_intN_max(8));

            _dst_val[_i].i8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2imp(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            int16_t dst;

            
                dst = src0;
                if (src0 < u_intN_min(16) || src0 > u_intN_max(16)) {
                   poison = true;
                }
              

            _dst_val[_i].i16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_f2snorm_16_v3d(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;

         
            uint32_t dst = _mesa_float_to_snorm16(src0);

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_f2u1(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            uint1_t dst;

            
                dst = src0;
                if (src0 < 0.0 || src0 > u_uintN_max(1)) {
                   poison = true;
                }
              

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            uint1_t dst;

            
                dst = src0;
                if (src0 < 0.0 || src0 > u_uintN_max(1)) {
                   poison = true;
                }
              

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            uint1_t dst;

            
                dst = src0;
                if (src0 < 0.0 || src0 > u_uintN_max(1)) {
                   poison = true;
                }
              

            /* 1-bit integers get truncated */
            _dst_val[_i].b = dst & 1;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2u16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            uint16_t dst;

            
                dst = src0;
                if (src0 < 0.0 || src0 > u_uintN_max(16)) {
                   poison = true;
                }
              

            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            uint16_t dst;

            
                dst = src0;
                if (src0 < 0.0 || src0 > u_uintN_max(16)) {
                   poison = true;
                }
              

            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            uint16_t dst;

            
                dst = src0;
                if (src0 < 0.0 || src0 > u_uintN_max(16)) {
                   poison = true;
                }
              

            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2u16_sat(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            uint16_t dst = (uint16_t)fmin(fmax(src0, 0.0), (double)u_uintN_max(16));

            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            uint16_t dst = (uint16_t)fmin(fmax(src0, 0.0), (double)u_uintN_max(16));

            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            uint16_t dst = (uint16_t)fmin(fmax(src0, 0.0), (double)u_uintN_max(16));

            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2u32(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            uint32_t dst;

            
                dst = src0;
                if (src0 < 0.0 || src0 > u_uintN_max(32)) {
                   poison = true;
                }
              

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            uint32_t dst;

            
                dst = src0;
                if (src0 < 0.0 || src0 > u_uintN_max(32)) {
                   poison = true;
                }
              

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            uint32_t dst;

            
                dst = src0;
                if (src0 < 0.0 || src0 > u_uintN_max(32)) {
                   poison = true;
                }
              

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2u32_sat(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            uint32_t dst = (uint32_t)fmin(fmax(src0, 0.0), (double)u_uintN_max(32));

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            uint32_t dst = (uint32_t)fmin(fmax(src0, 0.0), (double)u_uintN_max(32));

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            uint32_t dst = (uint32_t)fmin(fmax(src0, 0.0), (double)u_uintN_max(32));

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2u64(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            uint64_t dst;

            
                dst = src0;
                if (src0 < 0.0 || src0 > u_uintN_max(64)) {
                   poison = true;
                }
              

            _dst_val[_i].u64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            uint64_t dst;

            
                dst = src0;
                if (src0 < 0.0 || src0 > u_uintN_max(64)) {
                   poison = true;
                }
              

            _dst_val[_i].u64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            uint64_t dst;

            
                dst = src0;
                if (src0 < 0.0 || src0 > u_uintN_max(64)) {
                   poison = true;
                }
              

            _dst_val[_i].u64 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2u8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            uint8_t dst;

            
                dst = src0;
                if (src0 < 0.0 || src0 > u_uintN_max(8)) {
                   poison = true;
                }
              

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            uint8_t dst;

            
                dst = src0;
                if (src0 < 0.0 || src0 > u_uintN_max(8)) {
                   poison = true;
                }
              

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            uint8_t dst;

            
                dst = src0;
                if (src0 < 0.0 || src0 > u_uintN_max(8)) {
                   poison = true;
                }
              

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2u8_sat(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            uint8_t dst = (uint8_t)fmin(fmax(src0, 0.0), (double)u_uintN_max(8));

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            uint8_t dst = (uint8_t)fmin(fmax(src0, 0.0), (double)u_uintN_max(8));

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            uint8_t dst = (uint8_t)fmin(fmax(src0, 0.0), (double)u_uintN_max(8));

            _dst_val[_i].u8 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_f2ump(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            uint16_t dst;

            
                dst = src0;
                if (src0 < 0.0 || src0 > u_uintN_max(16)) {
                   poison = true;
                }
              

            _dst_val[_i].u16 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_f2unorm_16_v3d(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const uint32_t src0 =
                  _src[0][_i].u32;

         
            uint32_t dst = _mesa_float_to_unorm16(src0);

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fabs(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst = fabs(src0);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float32_t dst = fabs(src0);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float64_t dst = fabs(src0);

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fadd(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            float16_t dst;

            
if (nir_is_rounding_mode_rtz(execution_mode, bit_size)) {
   if (bit_size == 64)
      dst = _mesa_double_add_rtz(src0, src1);
   else
      dst = _mesa_double_to_float_rtz((double)src0 + (double)src1);
} else {
   dst = src0 + src1;
}


            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            float32_t dst;

            
if (nir_is_rounding_mode_rtz(execution_mode, bit_size)) {
   if (bit_size == 64)
      dst = _mesa_double_add_rtz(src0, src1);
   else
      dst = _mesa_double_to_float_rtz((double)src0 + (double)src1);
} else {
   dst = src0 + src1;
}


            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            float64_t dst;

            
if (nir_is_rounding_mode_rtz(execution_mode, bit_size)) {
   if (bit_size == 64)
      dst = _mesa_double_add_rtz(src0, src1);
   else
      dst = _mesa_double_to_float_rtz((double)src0 + (double)src1);
} else {
   dst = src0 + src1;
}


            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fall_equal16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
            get_float_source(_src[0][8], execution_mode, 32),
            get_float_source(_src[0][9], execution_mode, 32),
            get_float_source(_src[0][10], execution_mode, 32),
            get_float_source(_src[0][11], execution_mode, 32),
            get_float_source(_src[0][12], execution_mode, 32),
            get_float_source(_src[0][13], execution_mode, 32),
            get_float_source(_src[0][14], execution_mode, 32),
            get_float_source(_src[0][15], execution_mode, 32),
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
            get_float_source(_src[1][8], execution_mode, 32),
            get_float_source(_src[1][9], execution_mode, 32),
            get_float_source(_src[1][10], execution_mode, 32),
            get_float_source(_src[1][11], execution_mode, 32),
            get_float_source(_src[1][12], execution_mode, 32),
            get_float_source(_src[1][13], execution_mode, 32),
            get_float_source(_src[1][14], execution_mode, 32),
            get_float_source(_src[1][15], execution_mode, 32),
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p == src1.p) && (src0.o == src1.o) && (src0.n == src1.n) && (src0.m == src1.m) && (src0.l == src1.l) && (src0.k == src1.k) && (src0.j == src1.j) && (src0.i == src1.i) && (src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x)) ? 1.0f : 0.0f;

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fall_equal2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y == src1.y) && (src0.x == src1.x)) ? 1.0f : 0.0f;

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fall_equal3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x)) ? 1.0f : 0.0f;

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fall_equal4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x)) ? 1.0f : 0.0f;

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fall_equal5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x)) ? 1.0f : 0.0f;

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fall_equal8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h == src1.h) && (src0.g == src1.g) && (src0.f == src1.f) && (src0.e == src1.e) && (src0.w == src1.w) && (src0.z == src1.z) && (src0.y == src1.y) && (src0.x == src1.x)) ? 1.0f : 0.0f;

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fany_nequal16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
            get_float_source(_src[0][8], execution_mode, 32),
            get_float_source(_src[0][9], execution_mode, 32),
            get_float_source(_src[0][10], execution_mode, 32),
            get_float_source(_src[0][11], execution_mode, 32),
            get_float_source(_src[0][12], execution_mode, 32),
            get_float_source(_src[0][13], execution_mode, 32),
            get_float_source(_src[0][14], execution_mode, 32),
            get_float_source(_src[0][15], execution_mode, 32),
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
            get_float_source(_src[1][8], execution_mode, 32),
            get_float_source(_src[1][9], execution_mode, 32),
            get_float_source(_src[1][10], execution_mode, 32),
            get_float_source(_src[1][11], execution_mode, 32),
            get_float_source(_src[1][12], execution_mode, 32),
            get_float_source(_src[1][13], execution_mode, 32),
            get_float_source(_src[1][14], execution_mode, 32),
            get_float_source(_src[1][15], execution_mode, 32),
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p != src1.p) || (src0.o != src1.o) || (src0.n != src1.n) || (src0.m != src1.m) || (src0.l != src1.l) || (src0.k != src1.k) || (src0.j != src1.j) || (src0.i != src1.i) || (src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x)) ? 1.0f : 0.0f;

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fany_nequal2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y != src1.y) || (src0.x != src1.x)) ? 1.0f : 0.0f;

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fany_nequal3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x)) ? 1.0f : 0.0f;

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fany_nequal4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x)) ? 1.0f : 0.0f;

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fany_nequal5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x)) ? 1.0f : 0.0f;

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fany_nequal8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h != src1.h) || (src0.g != src1.g) || (src0.f != src1.f) || (src0.e != src1.e) || (src0.w != src1.w) || (src0.z != src1.z) || (src0.y != src1.y) || (src0.x != src1.x)) ? 1.0f : 0.0f;

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fcanonicalize(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst = src0;

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float32_t dst = src0;

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float64_t dst = src0;

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fceil(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst = bit_size == 64 ? ceil(src0) : ceilf(src0);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float32_t dst = bit_size == 64 ? ceil(src0) : ceilf(src0);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float64_t dst = bit_size == 64 ? ceil(src0) : ceilf(src0);

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fclamp_pos(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst = fmax(src0, 0.0);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float32_t dst = fmax(src0, 0.0);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float64_t dst = fmax(src0, 0.0);

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fcopysign_pco(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            float16_t dst = bit_size == 64 ? copysign(src0, src1) : copysignf(src0, src1);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            float32_t dst = bit_size == 64 ? copysign(src0, src1) : copysignf(src0, src1);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            float64_t dst = bit_size == 64 ? copysign(src0, src1) : copysignf(src0, src1);

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fcos(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst = bit_size == 64 ? cos(src0) : cosf(src0);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float32_t dst = bit_size == 64 ? cos(src0) : cosf(src0);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float64_t dst = bit_size == 64 ? cos(src0) : cosf(src0);

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fcos_amd(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst = cosf(6.2831853 * src0);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float32_t dst = cosf(6.2831853 * src0);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float64_t dst = cosf(6.2831853 * src0);

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fcos_mdg(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst = cosf(3.141592653589793 * src0);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float32_t dst = cosf(3.141592653589793 * src0);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float64_t dst = cosf(3.141592653589793 * src0);

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fcsel(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);
               const float32_t src2 =
                  get_float_source(_src[2][_i], execution_mode, 32);

         
            float32_t dst = (src0 != 0.0f) ? src1 : src2;

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fcsel_ge(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);
               const float32_t src2 =
                  get_float_source(_src[2][_i], execution_mode, 32);

         
            float32_t dst = (src0 >= 0.0f) ? src1 : src2;

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fcsel_gt(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);
               const float32_t src2 =
                  get_float_source(_src[2][_i], execution_mode, 32);

         
            float32_t dst = (src0 > 0.0f) ? src1 : src2;

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fdiv(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            float16_t dst = src0 / src1;

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            float32_t dst = src0 / src1;

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            float64_t dst = src0 / src1;

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fdot16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
            get_float_source(_src[0][8], execution_mode, 16),
            get_float_source(_src[0][9], execution_mode, 16),
            get_float_source(_src[0][10], execution_mode, 16),
            get_float_source(_src[0][11], execution_mode, 16),
            get_float_source(_src[0][12], execution_mode, 16),
            get_float_source(_src[0][13], execution_mode, 16),
            get_float_source(_src[0][14], execution_mode, 16),
            get_float_source(_src[0][15], execution_mode, 16),
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
            get_float_source(_src[1][8], execution_mode, 16),
            get_float_source(_src[1][9], execution_mode, 16),
            get_float_source(_src[1][10], execution_mode, 16),
            get_float_source(_src[1][11], execution_mode, 16),
            get_float_source(_src[1][12], execution_mode, 16),
            get_float_source(_src[1][13], execution_mode, 16),
            get_float_source(_src[1][14], execution_mode, 16),
            get_float_source(_src[1][15], execution_mode, 16),
      };

      struct float16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p * src1.p) + (src0.o * src1.o) + (src0.n * src1.n) + (src0.m * src1.m) + (src0.l * src1.l) + (src0.k * src1.k) + (src0.j * src1.j) + (src0.i * src1.i) + (src0.h * src1.h) + (src0.g * src1.g) + (src0.f * src1.f) + (src0.e * src1.e) + (src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[0].u16 = _mesa_float_to_float16_rtz(dst.x);
            } else {
               _dst_val[0].u16 = _mesa_float_to_float16_rtne(dst.x);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 16);
               }

         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
            get_float_source(_src[0][8], execution_mode, 32),
            get_float_source(_src[0][9], execution_mode, 32),
            get_float_source(_src[0][10], execution_mode, 32),
            get_float_source(_src[0][11], execution_mode, 32),
            get_float_source(_src[0][12], execution_mode, 32),
            get_float_source(_src[0][13], execution_mode, 32),
            get_float_source(_src[0][14], execution_mode, 32),
            get_float_source(_src[0][15], execution_mode, 32),
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
            get_float_source(_src[1][8], execution_mode, 32),
            get_float_source(_src[1][9], execution_mode, 32),
            get_float_source(_src[1][10], execution_mode, 32),
            get_float_source(_src[1][11], execution_mode, 32),
            get_float_source(_src[1][12], execution_mode, 32),
            get_float_source(_src[1][13], execution_mode, 32),
            get_float_source(_src[1][14], execution_mode, 32),
            get_float_source(_src[1][15], execution_mode, 32),
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p * src1.p) + (src0.o * src1.o) + (src0.n * src1.n) + (src0.m * src1.m) + (src0.l * src1.l) + (src0.k * src1.k) + (src0.j * src1.j) + (src0.i * src1.i) + (src0.h * src1.h) + (src0.g * src1.g) + (src0.f * src1.f) + (src0.e * src1.e) + (src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }

         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
            get_float_source(_src[0][8], execution_mode, 64),
            get_float_source(_src[0][9], execution_mode, 64),
            get_float_source(_src[0][10], execution_mode, 64),
            get_float_source(_src[0][11], execution_mode, 64),
            get_float_source(_src[0][12], execution_mode, 64),
            get_float_source(_src[0][13], execution_mode, 64),
            get_float_source(_src[0][14], execution_mode, 64),
            get_float_source(_src[0][15], execution_mode, 64),
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
            get_float_source(_src[1][8], execution_mode, 64),
            get_float_source(_src[1][9], execution_mode, 64),
            get_float_source(_src[1][10], execution_mode, 64),
            get_float_source(_src[1][11], execution_mode, 64),
            get_float_source(_src[1][12], execution_mode, 64),
            get_float_source(_src[1][13], execution_mode, 64),
            get_float_source(_src[1][14], execution_mode, 64),
            get_float_source(_src[1][15], execution_mode, 64),
      };

      struct float64_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.p * src1.p) + (src0.o * src1.o) + (src0.n * src1.n) + (src0.m * src1.m) + (src0.l * src1.l) + (src0.k * src1.k) + (src0.j * src1.j) + (src0.i * src1.i) + (src0.h * src1.h) + (src0.g * src1.g) + (src0.f * src1.f) + (src0.e * src1.e) + (src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[0].f64 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 64);
               }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fdot16_replicated(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
            get_float_source(_src[0][8], execution_mode, 16),
            get_float_source(_src[0][9], execution_mode, 16),
            get_float_source(_src[0][10], execution_mode, 16),
            get_float_source(_src[0][11], execution_mode, 16),
            get_float_source(_src[0][12], execution_mode, 16),
            get_float_source(_src[0][13], execution_mode, 16),
            get_float_source(_src[0][14], execution_mode, 16),
            get_float_source(_src[0][15], execution_mode, 16),
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
            get_float_source(_src[1][8], execution_mode, 16),
            get_float_source(_src[1][9], execution_mode, 16),
            get_float_source(_src[1][10], execution_mode, 16),
            get_float_source(_src[1][11], execution_mode, 16),
            get_float_source(_src[1][12], execution_mode, 16),
            get_float_source(_src[1][13], execution_mode, 16),
            get_float_source(_src[1][14], execution_mode, 16),
            get_float_source(_src[1][15], execution_mode, 16),
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float16_t dst = ((src0.p * src1.p) + (src0.o * src1.o) + (src0.n * src1.n) + (src0.m * src1.m) + (src0.l * src1.l) + (src0.k * src1.k) + (src0.j * src1.j) + (src0.i * src1.i) + (src0.h * src1.h) + (src0.g * src1.g) + (src0.f * src1.f) + (src0.e * src1.e) + (src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
            get_float_source(_src[0][8], execution_mode, 32),
            get_float_source(_src[0][9], execution_mode, 32),
            get_float_source(_src[0][10], execution_mode, 32),
            get_float_source(_src[0][11], execution_mode, 32),
            get_float_source(_src[0][12], execution_mode, 32),
            get_float_source(_src[0][13], execution_mode, 32),
            get_float_source(_src[0][14], execution_mode, 32),
            get_float_source(_src[0][15], execution_mode, 32),
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
            get_float_source(_src[1][8], execution_mode, 32),
            get_float_source(_src[1][9], execution_mode, 32),
            get_float_source(_src[1][10], execution_mode, 32),
            get_float_source(_src[1][11], execution_mode, 32),
            get_float_source(_src[1][12], execution_mode, 32),
            get_float_source(_src[1][13], execution_mode, 32),
            get_float_source(_src[1][14], execution_mode, 32),
            get_float_source(_src[1][15], execution_mode, 32),
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float32_t dst = ((src0.p * src1.p) + (src0.o * src1.o) + (src0.n * src1.n) + (src0.m * src1.m) + (src0.l * src1.l) + (src0.k * src1.k) + (src0.j * src1.j) + (src0.i * src1.i) + (src0.h * src1.h) + (src0.g * src1.g) + (src0.f * src1.f) + (src0.e * src1.e) + (src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
            get_float_source(_src[0][8], execution_mode, 64),
            get_float_source(_src[0][9], execution_mode, 64),
            get_float_source(_src[0][10], execution_mode, 64),
            get_float_source(_src[0][11], execution_mode, 64),
            get_float_source(_src[0][12], execution_mode, 64),
            get_float_source(_src[0][13], execution_mode, 64),
            get_float_source(_src[0][14], execution_mode, 64),
            get_float_source(_src[0][15], execution_mode, 64),
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
            get_float_source(_src[1][8], execution_mode, 64),
            get_float_source(_src[1][9], execution_mode, 64),
            get_float_source(_src[1][10], execution_mode, 64),
            get_float_source(_src[1][11], execution_mode, 64),
            get_float_source(_src[1][12], execution_mode, 64),
            get_float_source(_src[1][13], execution_mode, 64),
            get_float_source(_src[1][14], execution_mode, 64),
            get_float_source(_src[1][15], execution_mode, 64),
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float64_t dst = ((src0.p * src1.p) + (src0.o * src1.o) + (src0.n * src1.n) + (src0.m * src1.m) + (src0.l * src1.l) + (src0.k * src1.k) + (src0.j * src1.j) + (src0.i * src1.i) + (src0.h * src1.h) + (src0.g * src1.g) + (src0.f * src1.f) + (src0.e * src1.e) + (src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fdot2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y * src1.y) + (src0.x * src1.x));

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[0].u16 = _mesa_float_to_float16_rtz(dst.x);
            } else {
               _dst_val[0].u16 = _mesa_float_to_float16_rtne(dst.x);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 16);
               }

         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }

         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float64_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[0].f64 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 64);
               }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fdot2_replicated(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float16_t dst = ((src0.y * src1.y) + (src0.x * src1.x));

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float32_t dst = ((src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float64_t dst = ((src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fdot3(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[0].u16 = _mesa_float_to_float16_rtz(dst.x);
            } else {
               _dst_val[0].u16 = _mesa_float_to_float16_rtne(dst.x);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 16);
               }

         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }

         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float64_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[0].f64 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 64);
               }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fdot3_replicated(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float16_t dst = ((src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float32_t dst = ((src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float64_t dst = ((src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fdot4(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[0].u16 = _mesa_float_to_float16_rtz(dst.x);
            } else {
               _dst_val[0].u16 = _mesa_float_to_float16_rtne(dst.x);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 16);
               }

         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }

         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float64_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[0].f64 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 64);
               }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fdot4_replicated(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float16_t dst = ((src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float32_t dst = ((src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float64_t dst = ((src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fdot5(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e * src1.e) + (src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[0].u16 = _mesa_float_to_float16_rtz(dst.x);
            } else {
               _dst_val[0].u16 = _mesa_float_to_float16_rtne(dst.x);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 16);
               }

         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e * src1.e) + (src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }

         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float64_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.e * src1.e) + (src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[0].f64 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 64);
               }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fdot5_replicated(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float16_t dst = ((src0.e * src1.e) + (src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float32_t dst = ((src0.e * src1.e) + (src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float64_t dst = ((src0.e * src1.e) + (src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fdot8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float16_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h * src1.h) + (src0.g * src1.g) + (src0.f * src1.f) + (src0.e * src1.e) + (src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[0].u16 = _mesa_float_to_float16_rtz(dst.x);
            } else {
               _dst_val[0].u16 = _mesa_float_to_float16_rtne(dst.x);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 16);
               }

         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h * src1.h) + (src0.g * src1.g) + (src0.f * src1.f) + (src0.e * src1.e) + (src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }

         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float64_vec dst;

         dst.x = dst.y = dst.z = dst.w = ((src0.h * src1.h) + (src0.g * src1.g) + (src0.f * src1.f) + (src0.e * src1.e) + (src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[0].f64 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 64);
               }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fdot8_replicated(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
            get_float_source(_src[0][3], execution_mode, 16),
            get_float_source(_src[0][4], execution_mode, 16),
            get_float_source(_src[0][5], execution_mode, 16),
            get_float_source(_src[0][6], execution_mode, 16),
            get_float_source(_src[0][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
            get_float_source(_src[1][4], execution_mode, 16),
            get_float_source(_src[1][5], execution_mode, 16),
            get_float_source(_src[1][6], execution_mode, 16),
            get_float_source(_src[1][7], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float16_t dst = ((src0.h * src1.h) + (src0.g * src1.g) + (src0.f * src1.f) + (src0.e * src1.e) + (src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
            get_float_source(_src[0][3], execution_mode, 32),
            get_float_source(_src[0][4], execution_mode, 32),
            get_float_source(_src[0][5], execution_mode, 32),
            get_float_source(_src[0][6], execution_mode, 32),
            get_float_source(_src[0][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
            get_float_source(_src[1][4], execution_mode, 32),
            get_float_source(_src[1][5], execution_mode, 32),
            get_float_source(_src[1][6], execution_mode, 32),
            get_float_source(_src[1][7], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float32_t dst = ((src0.h * src1.h) + (src0.g * src1.g) + (src0.f * src1.f) + (src0.e * src1.e) + (src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
            get_float_source(_src[0][3], execution_mode, 64),
            get_float_source(_src[0][4], execution_mode, 64),
            get_float_source(_src[0][5], execution_mode, 64),
            get_float_source(_src[0][6], execution_mode, 64),
            get_float_source(_src[0][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
            get_float_source(_src[1][4], execution_mode, 64),
            get_float_source(_src[1][5], execution_mode, 64),
            get_float_source(_src[1][6], execution_mode, 64),
            get_float_source(_src[1][7], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float64_t dst = ((src0.h * src1.h) + (src0.g * src1.g) + (src0.f * src1.f) + (src0.e * src1.e) + (src0.w * src1.w) + (src0.z * src1.z) + (src0.y * src1.y) + (src0.x * src1.x));

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fdph(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float16_vec dst;

         dst.x = dst.y = dst.z = dst.w = src0.x * src1.x + src0.y * src1.y + src0.z * src1.z + src1.w;

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[0].u16 = _mesa_float_to_float16_rtz(dst.x);
            } else {
               _dst_val[0].u16 = _mesa_float_to_float16_rtne(dst.x);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 16);
               }

         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float32_vec dst;

         dst.x = dst.y = dst.z = dst.w = src0.x * src1.x + src0.y * src1.y + src0.z * src1.z + src1.w;

            _dst_val[0].f32 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 32);
               }

         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      struct float64_vec dst;

         dst.x = dst.y = dst.z = dst.w = src0.x * src1.x + src0.y * src1.y + src0.z * src1.z + src1.w;

            _dst_val[0].f64 = dst.x;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[0], 64);
               }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fdph_replicated(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   


      const struct float16_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 16),
            get_float_source(_src[0][1], execution_mode, 16),
            get_float_source(_src[0][2], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float16_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 16),
            get_float_source(_src[1][1], execution_mode, 16),
            get_float_source(_src[1][2], execution_mode, 16),
            get_float_source(_src[1][3], execution_mode, 16),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float16_t dst = src0.x * src1.x + src0.y * src1.y + src0.z * src1.z + src1.w;

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   


      const struct float32_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 32),
            get_float_source(_src[0][1], execution_mode, 32),
            get_float_source(_src[0][2], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float32_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 32),
            get_float_source(_src[1][1], execution_mode, 32),
            get_float_source(_src[1][2], execution_mode, 32),
            get_float_source(_src[1][3], execution_mode, 32),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float32_t dst = src0.x * src1.x + src0.y * src1.y + src0.z * src1.z + src1.w;

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   


      const struct float64_vec src0 = {
            get_float_source(_src[0][0], execution_mode, 64),
            get_float_source(_src[0][1], execution_mode, 64),
            get_float_source(_src[0][2], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      const struct float64_vec src1 = {
            get_float_source(_src[1][0], execution_mode, 64),
            get_float_source(_src[1][1], execution_mode, 64),
            get_float_source(_src[1][2], execution_mode, 64),
            get_float_source(_src[1][3], execution_mode, 64),
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
      };

      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
                              
         
            float64_t dst = src0.x * src1.x + src0.y * src1.y + src0.z * src1.z + src1.w;

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_feq(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool1_t dst = src0 == src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool1_t dst = src0 == src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool1_t dst = src0 == src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_feq16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool16_t dst = src0 == src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool16_t dst = src0 == src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool16_t dst = src0 == src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_feq32(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool32_t dst = src0 == src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool32_t dst = src0 == src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool32_t dst = src0 == src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_feq8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool8_t dst = src0 == src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool8_t dst = src0 == src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool8_t dst = src0 == src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fequ(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool1_t dst = isnan(src0) || isnan(src1) || src0 == src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool1_t dst = isnan(src0) || isnan(src1) || src0 == src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool1_t dst = isnan(src0) || isnan(src1) || src0 == src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fequ16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool16_t dst = isnan(src0) || isnan(src1) || src0 == src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool16_t dst = isnan(src0) || isnan(src1) || src0 == src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool16_t dst = isnan(src0) || isnan(src1) || src0 == src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fequ32(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool32_t dst = isnan(src0) || isnan(src1) || src0 == src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool32_t dst = isnan(src0) || isnan(src1) || src0 == src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool32_t dst = isnan(src0) || isnan(src1) || src0 == src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fequ8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool8_t dst = isnan(src0) || isnan(src1) || src0 == src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool8_t dst = isnan(src0) || isnan(src1) || src0 == src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool8_t dst = isnan(src0) || isnan(src1) || src0 == src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fexp2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst = exp2f(src0);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float32_t dst = exp2f(src0);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float64_t dst = exp2f(src0);

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ffloor(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst = bit_size == 64 ? floor(src0) : floorf(src0);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float32_t dst = bit_size == 64 ? floor(src0) : floorf(src0);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float64_t dst = bit_size == 64 ? floor(src0) : floorf(src0);

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ffma(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);
               const float16_t src2 =
                  get_float_source(_src[2][_i], execution_mode, 16);

         
            float16_t dst;

            
if (nir_is_rounding_mode_rtz(execution_mode, bit_size)) {
   if (bit_size == 64)
      dst = _mesa_double_fma_rtz(src0, src1, src2);
   else if (bit_size == 32)
      dst = _mesa_float_fma_rtz(src0, src1, src2);
   else
      dst = _mesa_double_to_float_rtz(_mesa_double_fma_rtz(src0, src1, src2));
} else {
   if (bit_size == 32)
      dst = fmaf(src0, src1, src2);
   else
      dst = fma(src0, src1, src2);
}


            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);
               const float32_t src2 =
                  get_float_source(_src[2][_i], execution_mode, 32);

         
            float32_t dst;

            
if (nir_is_rounding_mode_rtz(execution_mode, bit_size)) {
   if (bit_size == 64)
      dst = _mesa_double_fma_rtz(src0, src1, src2);
   else if (bit_size == 32)
      dst = _mesa_float_fma_rtz(src0, src1, src2);
   else
      dst = _mesa_double_to_float_rtz(_mesa_double_fma_rtz(src0, src1, src2));
} else {
   if (bit_size == 32)
      dst = fmaf(src0, src1, src2);
   else
      dst = fma(src0, src1, src2);
}


            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);
               const float64_t src2 =
                  get_float_source(_src[2][_i], execution_mode, 64);

         
            float64_t dst;

            
if (nir_is_rounding_mode_rtz(execution_mode, bit_size)) {
   if (bit_size == 64)
      dst = _mesa_double_fma_rtz(src0, src1, src2);
   else if (bit_size == 32)
      dst = _mesa_float_fma_rtz(src0, src1, src2);
   else
      dst = _mesa_double_to_float_rtz(_mesa_double_fma_rtz(src0, src1, src2));
} else {
   if (bit_size == 32)
      dst = fmaf(src0, src1, src2);
   else
      dst = fma(src0, src1, src2);
}


            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ffmaz(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);
               const float32_t src2 =
                  get_float_source(_src[2][_i], execution_mode, 32);

         
            float32_t dst;

            
if (src0 == 0.0 || src1 == 0.0)
   dst = 0.0 + src2;
else if (nir_is_rounding_mode_rtz(execution_mode, 32))
   dst = _mesa_float_fma_rtz(src0, src1, src2);
else
   dst = fmaf(src0, src1, src2);


            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_ffract(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst = src0 - (bit_size == 64 ? floor(src0) : floorf(src0));

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float32_t dst = src0 - (bit_size == 64 ? floor(src0) : floorf(src0));

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float64_t dst = src0 - (bit_size == 64 ? floor(src0) : floorf(src0));

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fge(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool1_t dst = src0 >= src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool1_t dst = src0 >= src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool1_t dst = src0 >= src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fge16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool16_t dst = src0 >= src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool16_t dst = src0 >= src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool16_t dst = src0 >= src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fge32(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool32_t dst = src0 >= src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool32_t dst = src0 >= src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool32_t dst = src0 >= src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fge8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool8_t dst = src0 >= src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool8_t dst = src0 >= src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool8_t dst = src0 >= src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fgeu(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool1_t dst = isnan(src0) || isnan(src1) || src0 >= src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool1_t dst = isnan(src0) || isnan(src1) || src0 >= src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool1_t dst = isnan(src0) || isnan(src1) || src0 >= src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fgeu16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool16_t dst = isnan(src0) || isnan(src1) || src0 >= src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool16_t dst = isnan(src0) || isnan(src1) || src0 >= src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool16_t dst = isnan(src0) || isnan(src1) || src0 >= src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fgeu32(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool32_t dst = isnan(src0) || isnan(src1) || src0 >= src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool32_t dst = isnan(src0) || isnan(src1) || src0 >= src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool32_t dst = isnan(src0) || isnan(src1) || src0 >= src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fgeu8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool8_t dst = isnan(src0) || isnan(src1) || src0 >= src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool8_t dst = isnan(src0) || isnan(src1) || src0 >= src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool8_t dst = isnan(src0) || isnan(src1) || src0 >= src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_find_lsb(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 1: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               /* 1-bit integers use a 0/-1 convention */
               const int1_t src0 = -(int1_t)_src[0][_i].b;

         
            int32_t dst;

            
dst = -1;
for (unsigned bit = 0; bit < bit_size; bit++) {
   if ((src0 >> bit) & 1) {
      dst = bit;
      break;
   }
}


            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 8: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int8_t src0 =
                  _src[0][_i].i8;

         
            int32_t dst;

            
dst = -1;
for (unsigned bit = 0; bit < bit_size; bit++) {
   if ((src0 >> bit) & 1) {
      dst = bit;
      break;
   }
}


            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int16_t src0 =
                  _src[0][_i].i16;

         
            int32_t dst;

            
dst = -1;
for (unsigned bit = 0; bit < bit_size; bit++) {
   if ((src0 >> bit) & 1) {
      dst = bit;
      break;
   }
}


            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int32_t src0 =
                  _src[0][_i].i32;

         
            int32_t dst;

            
dst = -1;
for (unsigned bit = 0; bit < bit_size; bit++) {
   if ((src0 >> bit) & 1) {
      dst = bit;
      break;
   }
}


            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const int64_t src0 =
                  _src[0][_i].i64;

         
            int32_t dst;

            
dst = -1;
for (unsigned bit = 0; bit < bit_size; bit++) {
   if ((src0 >> bit) & 1) {
      dst = bit;
      break;
   }
}


            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fisfinite(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            bool1_t dst = isfinite(src0);

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            bool1_t dst = isfinite(src0);

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            bool1_t dst = isfinite(src0);

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fisfinite32(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            bool32_t dst = isfinite(src0);

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            bool32_t dst = isfinite(src0);

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            bool32_t dst = isfinite(src0);

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fisnormal(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            bool1_t dst = isnormal(src0);

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            bool1_t dst = isnormal(src0);

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            bool1_t dst = isnormal(src0);

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_flog2(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst = log2f(src0);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float32_t dst = log2f(src0);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float64_t dst = log2f(src0);

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_flrp(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);
               const float16_t src2 =
                  get_float_source(_src[2][_i], execution_mode, 16);

         
            float16_t dst = src0 * (1 - src2) + src1 * src2;

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);
               const float32_t src2 =
                  get_float_source(_src[2][_i], execution_mode, 32);

         
            float32_t dst = src0 * (1 - src2) + src1 * src2;

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                           
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);
               const float64_t src2 =
                  get_float_source(_src[2][_i], execution_mode, 64);

         
            float64_t dst = src0 * (1 - src2) + src1 * src2;

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_flt(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool1_t dst = src0 < src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool1_t dst = src0 < src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool1_t dst = src0 < src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_flt16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool16_t dst = src0 < src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool16_t dst = src0 < src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool16_t dst = src0 < src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_flt32(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool32_t dst = src0 < src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool32_t dst = src0 < src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool32_t dst = src0 < src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_flt8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool8_t dst = src0 < src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool8_t dst = src0 < src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool8_t dst = src0 < src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fltu(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool1_t dst = isnan(src0) || isnan(src1) || src0 < src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool1_t dst = isnan(src0) || isnan(src1) || src0 < src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool1_t dst = isnan(src0) || isnan(src1) || src0 < src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fltu16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool16_t dst = isnan(src0) || isnan(src1) || src0 < src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool16_t dst = isnan(src0) || isnan(src1) || src0 < src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool16_t dst = isnan(src0) || isnan(src1) || src0 < src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fltu32(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool32_t dst = isnan(src0) || isnan(src1) || src0 < src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool32_t dst = isnan(src0) || isnan(src1) || src0 < src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool32_t dst = isnan(src0) || isnan(src1) || src0 < src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fltu8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool8_t dst = isnan(src0) || isnan(src1) || src0 < src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool8_t dst = isnan(src0) || isnan(src1) || src0 < src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool8_t dst = isnan(src0) || isnan(src1) || src0 < src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fmax(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            float16_t dst = util_max_num(src0, src1);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            float32_t dst = util_max_num(src0, src1);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            float64_t dst = util_max_num(src0, src1);

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fmax_agx(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            uint32_t dst = (src0 > src1 || isnan(src1)) ? src0 : src1;

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fmin(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            float16_t dst = util_min_num(src0, src1);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            float32_t dst = util_min_num(src0, src1);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            float64_t dst = util_min_num(src0, src1);

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fmin_agx(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            uint32_t dst = (src0 < src1 || isnan(src1)) ? src0 : src1;

            _dst_val[_i].u32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fmod(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            float16_t dst = src0 - src1 * floorf(src0 / src1);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            float32_t dst = src0 - src1 * floorf(src0 / src1);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            float64_t dst = src0 - src1 * floorf(src0 / src1);

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fmul(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            float16_t dst;

            
if (nir_is_rounding_mode_rtz(execution_mode, bit_size)) {
   if (bit_size == 64)
      dst = _mesa_double_mul_rtz(src0, src1);
   else
      dst = _mesa_double_to_float_rtz((double)src0 * (double)src1);
} else {
   dst = src0 * src1;
}


            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            float32_t dst;

            
if (nir_is_rounding_mode_rtz(execution_mode, bit_size)) {
   if (bit_size == 64)
      dst = _mesa_double_mul_rtz(src0, src1);
   else
      dst = _mesa_double_to_float_rtz((double)src0 * (double)src1);
} else {
   dst = src0 * src1;
}


            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            float64_t dst;

            
if (nir_is_rounding_mode_rtz(execution_mode, bit_size)) {
   if (bit_size == 64)
      dst = _mesa_double_mul_rtz(src0, src1);
   else
      dst = _mesa_double_to_float_rtz((double)src0 * (double)src1);
} else {
   dst = src0 * src1;
}


            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fmulz(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            float32_t dst;

            
if (src0 == 0.0 || src1 == 0.0)
   dst = 0.0;
else if (nir_is_rounding_mode_rtz(execution_mode, 32))
   dst = _mesa_double_to_float_rtz((double)src0 * (double)src1);
else
   dst = src0 * src1;


            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_fneg(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst = -src0;

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float32_t dst = -src0;

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float64_t dst = -src0;

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fneo(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool1_t dst = !isnan(src0) && !isnan(src1) && src0 != src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool1_t dst = !isnan(src0) && !isnan(src1) && src0 != src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool1_t dst = !isnan(src0) && !isnan(src1) && src0 != src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fneo16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool16_t dst = !isnan(src0) && !isnan(src1) && src0 != src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool16_t dst = !isnan(src0) && !isnan(src1) && src0 != src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool16_t dst = !isnan(src0) && !isnan(src1) && src0 != src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fneo32(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool32_t dst = !isnan(src0) && !isnan(src1) && src0 != src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool32_t dst = !isnan(src0) && !isnan(src1) && src0 != src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool32_t dst = !isnan(src0) && !isnan(src1) && src0 != src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fneo8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool8_t dst = !isnan(src0) && !isnan(src1) && src0 != src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool8_t dst = !isnan(src0) && !isnan(src1) && src0 != src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool8_t dst = !isnan(src0) && !isnan(src1) && src0 != src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fneu(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool1_t dst = src0 != src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool1_t dst = src0 != src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool1_t dst = src0 != src1;

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fneu16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool16_t dst = src0 != src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool16_t dst = src0 != src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool16_t dst = src0 != src1;

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fneu32(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool32_t dst = src0 != src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool32_t dst = src0 != src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool32_t dst = src0 != src1;

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fneu8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool8_t dst = src0 != src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool8_t dst = src0 != src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool8_t dst = src0 != src1;

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ford(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool1_t dst = !isnan(src0) && !isnan(src1);

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool1_t dst = !isnan(src0) && !isnan(src1);

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool1_t dst = !isnan(src0) && !isnan(src1);

            _dst_val[_i].b = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ford16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool16_t dst = !isnan(src0) && !isnan(src1);

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool16_t dst = !isnan(src0) && !isnan(src1);

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool16_t dst = !isnan(src0) && !isnan(src1);

            _dst_val[_i].i16 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ford32(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool32_t dst = !isnan(src0) && !isnan(src1);

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool32_t dst = !isnan(src0) && !isnan(src1);

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool32_t dst = !isnan(src0) && !isnan(src1);

            _dst_val[_i].i32 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_ford8(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            bool8_t dst = !isnan(src0) && !isnan(src1);

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            bool8_t dst = !isnan(src0) && !isnan(src1);

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            bool8_t dst = !isnan(src0) && !isnan(src1);

            _dst_val[_i].i8 = -(int)dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fpow(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            float16_t dst = bit_size == 64 ? pow(src0, src1) : powf(src0, src1);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            float32_t dst = bit_size == 64 ? pow(src0, src1) : powf(src0, src1);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            float64_t dst = bit_size == 64 ? pow(src0, src1) : powf(src0, src1);

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fquantize2f16(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                 UNUSED unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float32_t dst = (fabsf(src0) < ldexpf(1.0, -14)) ? copysignf(0.0f, src0) : _mesa_half_to_float(_mesa_float_to_half(src0));

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }


   return poison_mask;
}
static nir_component_mask_t
evaluate_frcp(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst = bit_size == 64 ? 1.0 / src0 : 1.0f / src0;

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float32_t dst = bit_size == 64 ? 1.0 / src0 : 1.0f / src0;

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float64_t dst = bit_size == 64 ? 1.0 / src0 : 1.0f / src0;

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_frem(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);
               const float16_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 16);

         
            float16_t dst = src0 - src1 * truncf(src0 / src1);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);
               const float32_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 32);

         
            float32_t dst = src0 - src1 * truncf(src0 / src1);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

                  
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);
               const float64_t src1 =
                  get_float_source(_src[1][_i], execution_mode, 64);

         
            float64_t dst = src0 - src1 * truncf(src0 / src1);

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_frexp_exp(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            int32_t dst;

            frexp(src0, &dst);

            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            int32_t dst;

            frexp(src0, &dst);

            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            int32_t dst;

            frexp(src0, &dst);

            _dst_val[_i].i32 = dst;


         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_frexp_sig(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst;

            int n; dst = frexp(src0, &n);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float32_t dst;

            int n; dst = frexp(src0, &n);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float64_t dst;

            int n; dst = frexp(src0, &n);

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fround_even(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst = bit_size == 64 ? _mesa_roundeven(src0) : _mesa_roundevenf(src0);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float32_t dst = bit_size == 64 ? _mesa_roundeven(src0) : _mesa_roundevenf(src0);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float64_t dst = bit_size == 64 ? _mesa_roundeven(src0) : _mesa_roundevenf(src0);

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_frsq(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst = bit_size == 64 ? 1.0 / sqrt(src0) : 1.0f / sqrtf(src0);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 16);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 32: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float32_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 32);

         
            float32_t dst = bit_size == 64 ? 1.0 / sqrt(src0) : 1.0f / sqrtf(src0);

            _dst_val[_i].f32 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 32)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 32);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }
      case 64: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float64_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 64);

         
            float64_t dst = bit_size == 64 ? 1.0 / sqrt(src0) : 1.0f / sqrtf(src0);

            _dst_val[_i].f64 = dst;

               if (nir_is_denorm_flush_to_zero(execution_mode, 64)) {
                  constant_denorm_flush_to_zero(&_dst_val[_i], 64);
               }

         if (poison)
            poison_mask |= (1 << _i);
      }

         break;
      }

      default:
         UNREACHABLE("unknown bit width");
      }

   return poison_mask;
}
static nir_component_mask_t
evaluate_fsat(nir_const_value *_dst_val,
                 UNUSED unsigned num_components,
                  unsigned bit_size,
                 UNUSED nir_const_value **_src,
                 UNUSED unsigned execution_mode)
{
   nir_component_mask_t poison_mask = 0;

      switch (bit_size) {
      case 16: {
         
   

         
      for (unsigned _i = 0; _i < num_components; _i++) {
         bool poison = false;
               const float16_t src0 =
                  get_float_source(_src[0][_i], execution_mode, 16);

         
            float16_t dst = util_min_num(util_max_num(src0, 0.0), 1.0);

            if (nir_is_rounding_mode_rtz(execution_mode, 16)) {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtz(dst);
            } else {
               _dst_val[_i].u16 = _mesa_float_to_float16_rtne(dst);
            }

               if (nir_is_denorm_flush_to_zero(execution_mode, 16)) {
    