// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef UI_GFX_IMAGE_IMAGE_SKIA_REP_DEFAULT_H_
#define UI_GFX_IMAGE_IMAGE_SKIA_REP_DEFAULT_H_

#include "base/component_export.h"
#include "build/build_config.h"
#include "cc/paint/paint_image.h"
#include "cc/paint/paint_op_buffer.h"
#include "third_party/skia/include/core/SkBitmap.h"
#include "ui/gfx/geometry/size.h"

namespace gfx {

// An ImageSkiaRep represents an image and the scale factor it is intended for.
// 0.0f scale is used to indicate that this ImageSkiaRep is used for unscaled
// (ImageSkia does not automatically scale the image).
// TODO(malaykeshav): Support transport of PaintRecord across mojo. This would
// require adding inline serialization support for PaintRecords.
class COMPONENT_EXPORT(GFX) ImageSkiaRep {
 public:
  // Create null bitmap.
  ImageSkiaRep();

  // Note: This is for testing purpose only.
  // Creates a bitmap with kARGB_8888_Config config with given |size| in DIP.
  // This allocates pixels in the bitmap. It is guaranteed that the data in the
  // bitmap are initialized but the actual values are undefined.
  // Specifying 0 scale means the image is for unscaled image (unscaled()
  // returns true, and scale() returns 1.0f).
  ImageSkiaRep(const gfx::Size& size, float scale);

  // Creates an ImageSkiaRep that holds the `src` bitmap, which is created for
  // display at the given device scale factor. Takes ownership of a reference to
  // the SkBitmap's backing store. The `src` bitmap may not be uninitialized,
  // null or empty; in that case the default constructor should be used
  // instead.
  ImageSkiaRep(const SkBitmap& src, float scale);

  // Creates an image rep backed by a paint record of given size and scale. This
  // is used when the image representation is sourced from a drawable such as
  // CanvasImageSource. The `size` must not be empty; in that case the default
  // constructor should be used instead.
  ImageSkiaRep(cc::PaintRecord paint_record,
               const gfx::Size& size,
               float scale);

  ImageSkiaRep(const ImageSkiaRep& other);
  ~ImageSkiaRep();

  // Get width and height of the image in pixels.
  int pixel_width() const { return pixel_size_.width(); }
  int pixel_height() const { return pixel_size_.height(); }
  const Size& pixel_size() const { return pixel_size_; }

  // Get width and height of the image in DIP.
  int GetWidth() const;
  int GetHeight() const;

  // Retrieves the scale for which this image is a representation of.
  float scale() const { return unscaled() ? 1.0f : scale_; }
  bool unscaled() const { return scale_ == 0.0f; }

  bool is_null() const {
    return type_ == ImageRepType::kImageTypeBitmap ? bitmap_.isNull()
                                                   : !paint_record_;
  }

  // Returns the backing bitmap when the image representation is sourced from a
  // bitmap. If this is a |kImageTypeDrawable| then it will generate(and cache)
  // a bitmap.
  const SkBitmap& GetBitmap() const;

  // Returns the backing drawable as a PaintRecord. Use this when the type of
  // ImageRep is |kImageTypeDrawable|.
  cc::PaintRecord GetPaintRecord() const;

  const cc::PaintImage& paint_image() const { return paint_image_; }
  bool has_paint_image() const { return !!paint_image_; }

 private:
  enum class ImageRepType {
    kImageTypeBitmap,   // When the source image is rasterized. (Bitmaps, PNGs)
    kImageTypeDrawable  // When the source image is a drawable generated by a
                        // CanvasImageSource.
  };

  // TODO(malaykeshav): Remove when migration is complete and it is safe.
  cc::PaintImage paint_image_;
  mutable std::optional<cc::PaintRecord> paint_record_;
  ImageRepType type_;

  Size pixel_size_;
  mutable SkBitmap bitmap_;
  float scale_;
};

}  // namespace gfx

#endif  // UI_GFX_IMAGE_IMAGE_SKIA_REP_DEFAULT_H_
