// SPDX-FileCopyrightText: Copyright (c) Kitware Inc.
// SPDX-License-Identifier: BSD-3-Clause
/**
 * @class vtkMarkSelectedRows
 * @brief mark rows that are selected as such.
 *
 * vtkMarkSelectedRows is an internal filter used by vtkSpreadSheetView. It
 * takes two inputs, both of which are generated by vtkDataTabulator. At index
 * 0, we get the raw tabular data, while at input index 1 is the tabular data
 * corresponding to the extracted selection corresponding to the data on input
 * 0. This filter generates an output which is same as the input expect with a
 * new array named "__vtkIsSelected__" which is set to 1 for all rows that are
 * present in the extracted selection input as well. "__vtkIsSelected__" may not
 * be added at all if the extracted selection input is missing or empty.
 *
 * This only works with `vtkPartitionedDataSet` inputs.
 *
 * @sa vtkDataTabulator
 */

#ifndef vtkMarkSelectedRows_h
#define vtkMarkSelectedRows_h

#include "vtkDataObjectAlgorithm.h"
#include "vtkPVVTKExtensionsFiltersRenderingModule.h" // needed for export macro

class vtkTable;

class VTKPVVTKEXTENSIONSFILTERSRENDERING_EXPORT vtkMarkSelectedRows : public vtkDataObjectAlgorithm
{
public:
  static vtkMarkSelectedRows* New();
  vtkTypeMacro(vtkMarkSelectedRows, vtkDataObjectAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  ///@{
  /**
   * Select the field to process. Only selection associated with the chosen
   * field are considered.
   * Accepted values are
   * \li vtkDataObject::FIELD_ASSOCIATION_POINTS,
   * \li vtkDataObject::FIELD_ASSOCIATION_CELLS,
   * \li vtkDataObject::FIELD_ASSOCIATION_NONE,
   * \li vtkDataObject::FIELD_ASSOCIATION_VERTICES,
   * \li vtkDataObject::FIELD_ASSOCIATION_EDGES,
   * \li vtkDataObject::FIELD_ASSOCIATION_ROWS
   * If value is vtkDataObject::FIELD_ASSOCIATION_NONE, then FieldData
   * associated with the input dataobject is extracted.
   */
  vtkSetMacro(FieldAssociation, int);
  vtkGetMacro(FieldAssociation, int);
  ///@}

protected:
  vtkMarkSelectedRows();
  ~vtkMarkSelectedRows() override;

  int FillInputPortInformation(int port, vtkInformation* info) override;
  int FillOutputPortInformation(int port, vtkInformation* info) override;
  int RequestData(vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;
  int FieldAssociation;

private:
  vtkMarkSelectedRows(const vtkMarkSelectedRows&) = delete;
  void operator=(const vtkMarkSelectedRows&) = delete;
};

#endif
