﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/lightsail/LightsailRequest.h>
#include <aws/lightsail/Lightsail_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Lightsail {
namespace Model {

/**
 */
class GetDistributionsRequest : public LightsailRequest {
 public:
  AWS_LIGHTSAIL_API GetDistributionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetDistributions"; }

  AWS_LIGHTSAIL_API Aws::String SerializePayload() const override;

  AWS_LIGHTSAIL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the distribution for which to return information.</p> <p>When
   * omitted, the response includes all of your distributions in the Amazon Web
   * Services Region where the request is made.</p>
   */
  inline const Aws::String& GetDistributionName() const { return m_distributionName; }
  inline bool DistributionNameHasBeenSet() const { return m_distributionNameHasBeenSet; }
  template <typename DistributionNameT = Aws::String>
  void SetDistributionName(DistributionNameT&& value) {
    m_distributionNameHasBeenSet = true;
    m_distributionName = std::forward<DistributionNameT>(value);
  }
  template <typename DistributionNameT = Aws::String>
  GetDistributionsRequest& WithDistributionName(DistributionNameT&& value) {
    SetDistributionName(std::forward<DistributionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token to advance to the next page of results from your request.</p> <p>To
   * get a page token, perform an initial <code>GetDistributions</code> request. If
   * your results are paginated, the response will return a next page token that you
   * can specify as the page token in a subsequent request.</p>
   */
  inline const Aws::String& GetPageToken() const { return m_pageToken; }
  inline bool PageTokenHasBeenSet() const { return m_pageTokenHasBeenSet; }
  template <typename PageTokenT = Aws::String>
  void SetPageToken(PageTokenT&& value) {
    m_pageTokenHasBeenSet = true;
    m_pageToken = std::forward<PageTokenT>(value);
  }
  template <typename PageTokenT = Aws::String>
  GetDistributionsRequest& WithPageToken(PageTokenT&& value) {
    SetPageToken(std::forward<PageTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_distributionName;

  Aws::String m_pageToken;
  bool m_distributionNameHasBeenSet = false;
  bool m_pageTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace Lightsail
}  // namespace Aws
