{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- Provides a mechanism to share global settings between applications.
-- 
-- On the X window system, this sharing is realized by an
-- <http://www.freedesktop.org/wiki/Specifications/xsettings-spec XSettings>
-- manager that is usually part of the desktop environment, along with
-- utilities that let the user change these settings.
-- 
-- On Wayland, the settings are obtained either via a settings portal,
-- or by reading desktop settings from t'GI.Gio.Objects.Settings.Settings'.
-- 
-- On macOS, the settings are obtained from @NSUserDefaults@.
-- 
-- In the absence of these sharing mechanisms, GTK reads default values for
-- settings from @settings.ini@ files in @\/etc\/gtk-4.0@, @$XDG_CONFIG_DIRS\/gtk-4.0@
-- and @$XDG_CONFIG_HOME\/gtk-4.0@. These files must be valid key files (see
-- @GKeyFile@), and have a section called Settings. Themes can also provide
-- default values for settings by installing a @settings.ini@ file
-- next to their @gtk.css@ file.
-- 
-- Applications can override system-wide settings by setting the property
-- of the @GtkSettings@ object with @/g_object_set()/@. This should be restricted
-- to special cases though; @GtkSettings@ are not meant as an application
-- configuration facility.
-- 
-- There is one @GtkSettings@ instance per display. It can be obtained with
-- 'GI.Gtk.Objects.Settings.settingsGetForDisplay', but in many cases, it is more
-- convenient to use 'GI.Gtk.Objects.Widget.widgetGetSettings'.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gtk.Objects.Settings
    ( 

-- * Exported types
    Settings(..)                            ,
    IsSettings                              ,
    toSettings                              ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [resetProperty]("GI.Gtk.Objects.Settings#g:method:resetProperty"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getData]("GI.GObject.Objects.Object#g:method:getData"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveSettingsMethod                   ,
#endif

-- ** getDefault #method:getDefault#

    settingsGetDefault                      ,


-- ** getForDisplay #method:getForDisplay#

    settingsGetForDisplay                   ,


-- ** resetProperty #method:resetProperty#

#if defined(ENABLE_OVERLOADING)
    SettingsResetPropertyMethodInfo         ,
#endif
    settingsResetProperty                   ,




 -- * Properties


-- ** gtkAlternativeButtonOrder #attr:gtkAlternativeButtonOrder#
-- | Whether buttons in dialogs should use the alternative button order.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkAlternativeButtonOrderPropertyInfo,
#endif
    constructSettingsGtkAlternativeButtonOrder,
    getSettingsGtkAlternativeButtonOrder    ,
    setSettingsGtkAlternativeButtonOrder    ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkAlternativeButtonOrder       ,
#endif


-- ** gtkAlternativeSortArrows #attr:gtkAlternativeSortArrows#
-- | Controls the direction of the sort indicators in sorted list and tree
-- views.
-- 
-- By default an arrow pointing down means the column is sorted
-- in ascending order. When set to 'P.True', this order will be inverted.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkAlternativeSortArrowsPropertyInfo,
#endif
    constructSettingsGtkAlternativeSortArrows,
    getSettingsGtkAlternativeSortArrows     ,
    setSettingsGtkAlternativeSortArrows     ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkAlternativeSortArrows        ,
#endif


-- ** gtkApplicationPreferDarkTheme #attr:gtkApplicationPreferDarkTheme#
-- | Whether the application prefers to use a dark theme.
-- 
-- If a GTK theme includes a dark variant, it will be used
-- instead of the configured theme.
-- 
-- Some applications benefit from minimizing the amount of light
-- pollution that interferes with the content. Good candidates for
-- dark themes are photo and video editors that make the actual
-- content get all the attention and minimize the distraction of
-- the chrome.
-- 
-- Dark themes should not be used for documents, where large spaces
-- are white\/light and the dark chrome creates too much contrast
-- (web browser, text editor...).

#if defined(ENABLE_OVERLOADING)
    SettingsGtkApplicationPreferDarkThemePropertyInfo,
#endif
    constructSettingsGtkApplicationPreferDarkTheme,
    getSettingsGtkApplicationPreferDarkTheme,
    setSettingsGtkApplicationPreferDarkTheme,
#if defined(ENABLE_OVERLOADING)
    settingsGtkApplicationPreferDarkTheme   ,
#endif


-- ** gtkCursorAspectRatio #attr:gtkCursorAspectRatio#
-- | The aspect ratio of the text caret.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkCursorAspectRatioPropertyInfo,
#endif
    constructSettingsGtkCursorAspectRatio   ,
    getSettingsGtkCursorAspectRatio         ,
    setSettingsGtkCursorAspectRatio         ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkCursorAspectRatio            ,
#endif


-- ** gtkCursorBlink #attr:gtkCursorBlink#
-- | Whether the cursor should blink.
-- 
-- Also see the [Settings:gtkCursorBlinkTimeout]("GI.Gtk.Objects.Settings#g:attr:gtkCursorBlinkTimeout") setting,
-- which allows more flexible control over cursor blinking.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkCursorBlinkPropertyInfo      ,
#endif
    constructSettingsGtkCursorBlink         ,
    getSettingsGtkCursorBlink               ,
    setSettingsGtkCursorBlink               ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkCursorBlink                  ,
#endif


-- ** gtkCursorBlinkTime #attr:gtkCursorBlinkTime#
-- | Length of the cursor blink cycle, in milliseconds.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkCursorBlinkTimePropertyInfo  ,
#endif
    constructSettingsGtkCursorBlinkTime     ,
    getSettingsGtkCursorBlinkTime           ,
    setSettingsGtkCursorBlinkTime           ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkCursorBlinkTime              ,
#endif


-- ** gtkCursorBlinkTimeout #attr:gtkCursorBlinkTimeout#
-- | Time after which the cursor stops blinking, in seconds.
-- 
-- The timer is reset after each user interaction.
-- 
-- Setting this to zero has the same effect as setting
-- [Settings:gtkCursorBlink]("GI.Gtk.Objects.Settings#g:attr:gtkCursorBlink") to 'P.False'.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkCursorBlinkTimeoutPropertyInfo,
#endif
    constructSettingsGtkCursorBlinkTimeout  ,
    getSettingsGtkCursorBlinkTimeout        ,
    setSettingsGtkCursorBlinkTimeout        ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkCursorBlinkTimeout           ,
#endif


-- ** gtkCursorThemeName #attr:gtkCursorThemeName#
-- | Name of the cursor theme to use.
-- 
-- Use 'P.Nothing' to use the default theme.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkCursorThemeNamePropertyInfo  ,
#endif
    clearSettingsGtkCursorThemeName         ,
    constructSettingsGtkCursorThemeName     ,
    getSettingsGtkCursorThemeName           ,
    setSettingsGtkCursorThemeName           ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkCursorThemeName              ,
#endif


-- ** gtkCursorThemeSize #attr:gtkCursorThemeSize#
-- | The size to use for cursors.
-- 
-- 0 means to use the default size.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkCursorThemeSizePropertyInfo  ,
#endif
    constructSettingsGtkCursorThemeSize     ,
    getSettingsGtkCursorThemeSize           ,
    setSettingsGtkCursorThemeSize           ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkCursorThemeSize              ,
#endif


-- ** gtkDecorationLayout #attr:gtkDecorationLayout#
-- | Determines which buttons should be put in the
-- titlebar of client-side decorated windows, and whether they
-- should be placed on the left or right.
-- 
-- The format of the string is button names, separated by commas.
-- A colon separates the buttons that should appear on the left
-- from those on the right. Recognized button names are minimize,
-- maximize, close, icon (the window icon) and menu (a menu button
-- for the fallback app menu).
-- 
-- For example, \"menu:minimize,maximize,close\" specifies a menu
-- on the left, and minimize, maximize and close buttons on the right.
-- 
-- Note that buttons will only be shown when they are meaningful.
-- E.g. a menu button only appears when the desktop shell does not
-- show the app menu, and a close button only appears on a window
-- that can be closed.
-- 
-- Also note that the setting can be overridden with the
-- [HeaderBar:decorationLayout]("GI.Gtk.Objects.HeaderBar#g:attr:decorationLayout") property.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkDecorationLayoutPropertyInfo ,
#endif
    clearSettingsGtkDecorationLayout        ,
    constructSettingsGtkDecorationLayout    ,
    getSettingsGtkDecorationLayout          ,
    setSettingsGtkDecorationLayout          ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkDecorationLayout             ,
#endif


-- ** gtkDialogsUseHeader #attr:gtkDialogsUseHeader#
-- | Whether builtin GTK dialogs such as the file chooser, the
-- color chooser or the font chooser will use a header bar at
-- the top to show action widgets, or an action area at the bottom.
-- 
-- This setting does not affect custom dialogs using @GtkDialog@
-- directly, or message dialogs.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkDialogsUseHeaderPropertyInfo ,
#endif
    constructSettingsGtkDialogsUseHeader    ,
    getSettingsGtkDialogsUseHeader          ,
    setSettingsGtkDialogsUseHeader          ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkDialogsUseHeader             ,
#endif


-- ** gtkDndDragThreshold #attr:gtkDndDragThreshold#
-- | The number of pixels the cursor can move before dragging.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkDndDragThresholdPropertyInfo ,
#endif
    constructSettingsGtkDndDragThreshold    ,
    getSettingsGtkDndDragThreshold          ,
    setSettingsGtkDndDragThreshold          ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkDndDragThreshold             ,
#endif


-- ** gtkDoubleClickDistance #attr:gtkDoubleClickDistance#
-- | The maximum distance allowed between two clicks for them to be considered
-- a double click, in pixels.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkDoubleClickDistancePropertyInfo,
#endif
    constructSettingsGtkDoubleClickDistance ,
    getSettingsGtkDoubleClickDistance       ,
    setSettingsGtkDoubleClickDistance       ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkDoubleClickDistance          ,
#endif


-- ** gtkDoubleClickTime #attr:gtkDoubleClickTime#
-- | The maximum time to allow between two clicks for them to be considered
-- a double click, in milliseconds.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkDoubleClickTimePropertyInfo  ,
#endif
    constructSettingsGtkDoubleClickTime     ,
    getSettingsGtkDoubleClickTime           ,
    setSettingsGtkDoubleClickTime           ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkDoubleClickTime              ,
#endif


-- ** gtkEnableAccels #attr:gtkEnableAccels#
-- | Whether menu items should have visible accelerators which can be
-- activated.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkEnableAccelsPropertyInfo     ,
#endif
    constructSettingsGtkEnableAccels        ,
    getSettingsGtkEnableAccels              ,
    setSettingsGtkEnableAccels              ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkEnableAccels                 ,
#endif


-- ** gtkEnableAnimations #attr:gtkEnableAnimations#
-- | Whether to enable toolkit-wide animations.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkEnableAnimationsPropertyInfo ,
#endif
    constructSettingsGtkEnableAnimations    ,
    getSettingsGtkEnableAnimations          ,
    setSettingsGtkEnableAnimations          ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkEnableAnimations             ,
#endif


-- ** gtkEnableEventSounds #attr:gtkEnableEventSounds#
-- | Whether to play any event sounds at all.
-- 
-- See the <http://www.freedesktop.org/wiki/Specifications/sound-theme-spec Sound Theme Specifications>
-- for more information on event sounds and sound themes.
-- 
-- GTK itself does not support event sounds, you have to use a loadable
-- module like the one that comes with libcanberra.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkEnableEventSoundsPropertyInfo,
#endif
    constructSettingsGtkEnableEventSounds   ,
    getSettingsGtkEnableEventSounds         ,
    setSettingsGtkEnableEventSounds         ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkEnableEventSounds            ,
#endif


-- ** gtkEnableInputFeedbackSounds #attr:gtkEnableInputFeedbackSounds#
-- | Whether to play event sounds as feedback to user input.
-- 
-- See the <http://www.freedesktop.org/wiki/Specifications/sound-theme-spec Sound Theme Specifications>
-- for more information on event sounds and sound themes.
-- 
-- GTK itself does not support event sounds, you have to use a loadable
-- module like the one that comes with libcanberra.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkEnableInputFeedbackSoundsPropertyInfo,
#endif
    constructSettingsGtkEnableInputFeedbackSounds,
    getSettingsGtkEnableInputFeedbackSounds ,
    setSettingsGtkEnableInputFeedbackSounds ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkEnableInputFeedbackSounds    ,
#endif


-- ** gtkEnablePrimaryPaste #attr:gtkEnablePrimaryPaste#
-- | Whether a middle click on a mouse should paste the
-- \'PRIMARY\' clipboard content at the cursor location.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkEnablePrimaryPastePropertyInfo,
#endif
    constructSettingsGtkEnablePrimaryPaste  ,
    getSettingsGtkEnablePrimaryPaste        ,
    setSettingsGtkEnablePrimaryPaste        ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkEnablePrimaryPaste           ,
#endif


-- ** gtkEntryPasswordHintTimeout #attr:gtkEntryPasswordHintTimeout#
-- | How long to show the last input character in hidden
-- entries.
-- 
-- This value is in milliseconds. 0 disables showing the
-- last char. 600 is a good value for enabling it.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkEntryPasswordHintTimeoutPropertyInfo,
#endif
    constructSettingsGtkEntryPasswordHintTimeout,
    getSettingsGtkEntryPasswordHintTimeout  ,
    setSettingsGtkEntryPasswordHintTimeout  ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkEntryPasswordHintTimeout     ,
#endif


-- ** gtkEntrySelectOnFocus #attr:gtkEntrySelectOnFocus#
-- | Whether to select the contents of an entry when it is focused.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkEntrySelectOnFocusPropertyInfo,
#endif
    constructSettingsGtkEntrySelectOnFocus  ,
    getSettingsGtkEntrySelectOnFocus        ,
    setSettingsGtkEntrySelectOnFocus        ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkEntrySelectOnFocus           ,
#endif


-- ** gtkErrorBell #attr:gtkErrorBell#
-- | When 'P.True', keyboard navigation and other input-related errors
-- will cause a beep.
-- 
-- Since the error bell is implemented using 'GI.Gdk.Objects.Surface.surfaceBeep', the
-- windowing system may offer ways to configure the error bell in many
-- ways, such as flashing the window or similar visual effects.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkErrorBellPropertyInfo        ,
#endif
    constructSettingsGtkErrorBell           ,
    getSettingsGtkErrorBell                 ,
    setSettingsGtkErrorBell                 ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkErrorBell                    ,
#endif


-- ** gtkFontName #attr:gtkFontName#
-- | The default font to use.
-- 
-- GTK uses the family name and size from this string.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkFontNamePropertyInfo         ,
#endif
    clearSettingsGtkFontName                ,
    constructSettingsGtkFontName            ,
    getSettingsGtkFontName                  ,
    setSettingsGtkFontName                  ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkFontName                     ,
#endif


-- ** gtkFontRendering #attr:gtkFontRendering#
-- | How GTK font rendering is set up.
-- 
-- When set to 'GI.Gtk.Enums.FontRenderingManual', GTK respects the low-level
-- font-related settings ([Settings:gtkHintFontMetrics]("GI.Gtk.Objects.Settings#g:attr:gtkHintFontMetrics"),
-- [Settings:gtkXftAntialias]("GI.Gtk.Objects.Settings#g:attr:gtkXftAntialias"), [Settings:gtkXftHinting]("GI.Gtk.Objects.Settings#g:attr:gtkXftHinting"),
-- [Settings:gtkXftHintstyle]("GI.Gtk.Objects.Settings#g:attr:gtkXftHintstyle") and [Settings:gtkXftRgba]("GI.Gtk.Objects.Settings#g:attr:gtkXftRgba"))
-- as much as practical.
-- 
-- When set to 'GI.Gtk.Enums.FontRenderingAutomatic', GTK will consider factors such
-- as screen resolution and scale in deciding how to render fonts.
-- 
-- /Since: 4.16/

#if defined(ENABLE_OVERLOADING)
    SettingsGtkFontRenderingPropertyInfo    ,
#endif
    constructSettingsGtkFontRendering       ,
    getSettingsGtkFontRendering             ,
    setSettingsGtkFontRendering             ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkFontRendering                ,
#endif


-- ** gtkFontconfigTimestamp #attr:gtkFontconfigTimestamp#
-- | Timestamp of the current fontconfig configuration.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkFontconfigTimestampPropertyInfo,
#endif
    constructSettingsGtkFontconfigTimestamp ,
    getSettingsGtkFontconfigTimestamp       ,
    setSettingsGtkFontconfigTimestamp       ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkFontconfigTimestamp          ,
#endif


-- ** gtkHintFontMetrics #attr:gtkHintFontMetrics#
-- | Whether hinting should be applied to font metrics.
-- 
-- Note that this also turns off subpixel positioning of glyphs,
-- since it conflicts with metrics hinting.
-- 
-- /Since: 4.6/

#if defined(ENABLE_OVERLOADING)
    SettingsGtkHintFontMetricsPropertyInfo  ,
#endif
    constructSettingsGtkHintFontMetrics     ,
    getSettingsGtkHintFontMetrics           ,
    setSettingsGtkHintFontMetrics           ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkHintFontMetrics              ,
#endif


-- ** gtkIconThemeName #attr:gtkIconThemeName#
-- | Name of the icon theme to use.
-- 
-- See t'GI.Gtk.Objects.IconTheme.IconTheme' for details about how
-- GTK handles icon themes.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkIconThemeNamePropertyInfo    ,
#endif
    clearSettingsGtkIconThemeName           ,
    constructSettingsGtkIconThemeName       ,
    getSettingsGtkIconThemeName             ,
    setSettingsGtkIconThemeName             ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkIconThemeName                ,
#endif


-- ** gtkImModule #attr:gtkImModule#
-- | Which IM (input method) module should be used by default.
-- 
-- This is the input method that will be used if the user has not
-- explicitly chosen another input method from the IM context menu.
-- This also can be a colon-separated list of input methods, which GTK
-- will try in turn until it finds one available on the system.
-- 
-- See t'GI.Gtk.Objects.IMContext.IMContext'.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkImModulePropertyInfo         ,
#endif
    clearSettingsGtkImModule                ,
    constructSettingsGtkImModule            ,
    getSettingsGtkImModule                  ,
    setSettingsGtkImModule                  ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkImModule                     ,
#endif


-- ** gtkInterfaceColorScheme #attr:gtkInterfaceColorScheme#
-- | The color scheme used for rendering the user interface.
-- 
-- This setting communicates the system-wide preference.
-- The color scheme that is actually used when applying CSS
-- styles can be set with the [CssProvider:prefersColorScheme]("GI.Gtk.Objects.CssProvider#g:attr:prefersColorScheme")
-- property.
-- 
-- /Since: 4.20/

#if defined(ENABLE_OVERLOADING)
    SettingsGtkInterfaceColorSchemePropertyInfo,
#endif
    constructSettingsGtkInterfaceColorScheme,
    getSettingsGtkInterfaceColorScheme      ,
    setSettingsGtkInterfaceColorScheme      ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkInterfaceColorScheme         ,
#endif


-- ** gtkInterfaceContrast #attr:gtkInterfaceContrast#
-- | The level of contrast to use for the user interface.
-- 
-- This setting communicates the system-wide preference.
-- The contrast level that is actually used when applying CSS
-- styles can be set with the [CssProvider:prefersContrast]("GI.Gtk.Objects.CssProvider#g:attr:prefersContrast")
-- property.
-- 
-- /Since: 4.20/

#if defined(ENABLE_OVERLOADING)
    SettingsGtkInterfaceContrastPropertyInfo,
#endif
    constructSettingsGtkInterfaceContrast   ,
    getSettingsGtkInterfaceContrast         ,
    setSettingsGtkInterfaceContrast         ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkInterfaceContrast            ,
#endif


-- ** gtkKeynavUseCaret #attr:gtkKeynavUseCaret#
-- | Whether GTK should make sure that text can be navigated with
-- a caret, even if it is not editable.
-- 
-- This is useful when using a screen reader.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkKeynavUseCaretPropertyInfo   ,
#endif
    constructSettingsGtkKeynavUseCaret      ,
    getSettingsGtkKeynavUseCaret            ,
    setSettingsGtkKeynavUseCaret            ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkKeynavUseCaret               ,
#endif


-- ** gtkLabelSelectOnFocus #attr:gtkLabelSelectOnFocus#
-- | Whether to select the contents of a selectable
-- label when it is focused.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkLabelSelectOnFocusPropertyInfo,
#endif
    constructSettingsGtkLabelSelectOnFocus  ,
    getSettingsGtkLabelSelectOnFocus        ,
    setSettingsGtkLabelSelectOnFocus        ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkLabelSelectOnFocus           ,
#endif


-- ** gtkLongPressTime #attr:gtkLongPressTime#
-- | The time for a button or touch press to be considered a “long press”.
-- 
-- See t'GI.Gtk.Objects.GestureLongPress.GestureLongPress'.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkLongPressTimePropertyInfo    ,
#endif
    constructSettingsGtkLongPressTime       ,
    getSettingsGtkLongPressTime             ,
    setSettingsGtkLongPressTime             ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkLongPressTime                ,
#endif


-- ** gtkOverlayScrolling #attr:gtkOverlayScrolling#
-- | Whether scrolled windows may use overlaid scrolling indicators.
-- 
-- If this is set to 'P.False', scrolled windows will have permanent
-- scrollbars.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkOverlayScrollingPropertyInfo ,
#endif
    constructSettingsGtkOverlayScrolling    ,
    getSettingsGtkOverlayScrolling          ,
    setSettingsGtkOverlayScrolling          ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkOverlayScrolling             ,
#endif


-- ** gtkPrimaryButtonWarpsSlider #attr:gtkPrimaryButtonWarpsSlider#
-- | If the value of this setting is 'P.True', clicking the primary button in a
-- @GtkRange@ trough will move the slider, and hence set the range’s value, to
-- the point that you clicked.
-- 
-- If it is 'P.False', a primary click will cause the slider\/value to move
-- by the range’s page-size towards the point clicked.
-- 
-- Whichever action you choose for the primary button, the other action will
-- be available by holding Shift and primary-clicking, or clicking the middle
-- mouse button.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkPrimaryButtonWarpsSliderPropertyInfo,
#endif
    constructSettingsGtkPrimaryButtonWarpsSlider,
    getSettingsGtkPrimaryButtonWarpsSlider  ,
    setSettingsGtkPrimaryButtonWarpsSlider  ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkPrimaryButtonWarpsSlider     ,
#endif


-- ** gtkPrintBackends #attr:gtkPrintBackends#
-- | A comma-separated list of print backends to use in the print
-- dialog.
-- 
-- Available print backends depend on the GTK installation,
-- and may include \"file\", \"cups\", \"lpr\" or \"papi\".

#if defined(ENABLE_OVERLOADING)
    SettingsGtkPrintBackendsPropertyInfo    ,
#endif
    clearSettingsGtkPrintBackends           ,
    constructSettingsGtkPrintBackends       ,
    getSettingsGtkPrintBackends             ,
    setSettingsGtkPrintBackends             ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkPrintBackends                ,
#endif


-- ** gtkPrintPreviewCommand #attr:gtkPrintPreviewCommand#
-- | A command to run for displaying the print preview.
-- 
-- The command should contain a @%f@ placeholder, which will get
-- replaced by the path to the pdf file. The command may also
-- contain a @%s@ placeholder, which will get replaced by the
-- path to a file containing the print settings in the format
-- produced by 'GI.Gtk.Objects.PrintSettings.printSettingsToFile'.
-- 
-- The preview application is responsible for removing the pdf
-- file and the print settings file when it is done.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkPrintPreviewCommandPropertyInfo,
#endif
    clearSettingsGtkPrintPreviewCommand     ,
    constructSettingsGtkPrintPreviewCommand ,
    getSettingsGtkPrintPreviewCommand       ,
    setSettingsGtkPrintPreviewCommand       ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkPrintPreviewCommand          ,
#endif


-- ** gtkRecentFilesEnabled #attr:gtkRecentFilesEnabled#
-- | Whether GTK should keep track of items inside the recently used
-- resources list.
-- 
-- If set to 'P.False', the list will always be empty.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkRecentFilesEnabledPropertyInfo,
#endif
    constructSettingsGtkRecentFilesEnabled  ,
    getSettingsGtkRecentFilesEnabled        ,
    setSettingsGtkRecentFilesEnabled        ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkRecentFilesEnabled           ,
#endif


-- ** gtkRecentFilesMaxAge #attr:gtkRecentFilesMaxAge#
-- | The maximum age, in days, of the items inside the recently used
-- resources list.
-- 
-- Items older than this setting will be excised from the list.
-- If set to 0, the list will always be empty; if set to -1, no
-- item will be removed.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkRecentFilesMaxAgePropertyInfo,
#endif
    constructSettingsGtkRecentFilesMaxAge   ,
    getSettingsGtkRecentFilesMaxAge         ,
    setSettingsGtkRecentFilesMaxAge         ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkRecentFilesMaxAge            ,
#endif


-- ** gtkShellShowsAppMenu #attr:gtkShellShowsAppMenu#
-- | Set to 'P.True' if the desktop environment is displaying
-- the app menu, 'P.False' if the app should display it itself.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkShellShowsAppMenuPropertyInfo,
#endif
    constructSettingsGtkShellShowsAppMenu   ,
    getSettingsGtkShellShowsAppMenu         ,
    setSettingsGtkShellShowsAppMenu         ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkShellShowsAppMenu            ,
#endif


-- ** gtkShellShowsDesktop #attr:gtkShellShowsDesktop#
-- | Set to 'P.True' if the desktop environment is displaying
-- the desktop folder, 'P.False' if not.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkShellShowsDesktopPropertyInfo,
#endif
    constructSettingsGtkShellShowsDesktop   ,
    getSettingsGtkShellShowsDesktop         ,
    setSettingsGtkShellShowsDesktop         ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkShellShowsDesktop            ,
#endif


-- ** gtkShellShowsMenubar #attr:gtkShellShowsMenubar#
-- | Set to 'P.True' if the desktop environment is displaying
-- the menubar, 'P.False' if the app should display it itself.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkShellShowsMenubarPropertyInfo,
#endif
    constructSettingsGtkShellShowsMenubar   ,
    getSettingsGtkShellShowsMenubar         ,
    setSettingsGtkShellShowsMenubar         ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkShellShowsMenubar            ,
#endif


-- ** gtkShowStatusShapes #attr:gtkShowStatusShapes#
-- | When 'P.True', widgets like switches include shapes to indicate their on\/off state.
-- 
-- /Since: 4.14/

#if defined(ENABLE_OVERLOADING)
    SettingsGtkShowStatusShapesPropertyInfo ,
#endif
    constructSettingsGtkShowStatusShapes    ,
    getSettingsGtkShowStatusShapes          ,
    setSettingsGtkShowStatusShapes          ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkShowStatusShapes             ,
#endif


-- ** gtkSoundThemeName #attr:gtkSoundThemeName#
-- | The XDG sound theme to use for event sounds.
-- 
-- See the <http://www.freedesktop.org/wiki/Specifications/sound-theme-spec Sound Theme Specifications>
-- for more information on event sounds and sound themes.
-- 
-- GTK itself does not support event sounds, you have to use
-- a loadable module like the one that comes with libcanberra.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkSoundThemeNamePropertyInfo   ,
#endif
    clearSettingsGtkSoundThemeName          ,
    constructSettingsGtkSoundThemeName      ,
    getSettingsGtkSoundThemeName            ,
    setSettingsGtkSoundThemeName            ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkSoundThemeName               ,
#endif


-- ** gtkSplitCursor #attr:gtkSplitCursor#
-- | Whether two cursors should be displayed for mixed left-to-right and
-- right-to-left text.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkSplitCursorPropertyInfo      ,
#endif
    constructSettingsGtkSplitCursor         ,
    getSettingsGtkSplitCursor               ,
    setSettingsGtkSplitCursor               ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkSplitCursor                  ,
#endif


-- ** gtkThemeName #attr:gtkThemeName#
-- | Name of the theme to load.
-- 
-- See t'GI.Gtk.Objects.CssProvider.CssProvider' for details about how
-- GTK finds the CSS stylesheet for a theme.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkThemeNamePropertyInfo        ,
#endif
    clearSettingsGtkThemeName               ,
    constructSettingsGtkThemeName           ,
    getSettingsGtkThemeName                 ,
    setSettingsGtkThemeName                 ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkThemeName                    ,
#endif


-- ** gtkTitlebarDoubleClick #attr:gtkTitlebarDoubleClick#
-- | Determines the action to take when a double-click
-- occurs on the titlebar of client-side decorated windows.
-- 
-- Recognized actions are minimize, toggle-maximize, menu, lower
-- or none.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkTitlebarDoubleClickPropertyInfo,
#endif
    clearSettingsGtkTitlebarDoubleClick     ,
    constructSettingsGtkTitlebarDoubleClick ,
    getSettingsGtkTitlebarDoubleClick       ,
    setSettingsGtkTitlebarDoubleClick       ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkTitlebarDoubleClick          ,
#endif


-- ** gtkTitlebarMiddleClick #attr:gtkTitlebarMiddleClick#
-- | Determines the action to take when a middle-click
-- occurs on the titlebar of client-side decorated windows.
-- 
-- Recognized actions are minimize, toggle-maximize, menu, lower
-- or none.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkTitlebarMiddleClickPropertyInfo,
#endif
    clearSettingsGtkTitlebarMiddleClick     ,
    constructSettingsGtkTitlebarMiddleClick ,
    getSettingsGtkTitlebarMiddleClick       ,
    setSettingsGtkTitlebarMiddleClick       ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkTitlebarMiddleClick          ,
#endif


-- ** gtkTitlebarRightClick #attr:gtkTitlebarRightClick#
-- | Determines the action to take when a right-click
-- occurs on the titlebar of client-side decorated windows.
-- 
-- Recognized actions are minimize, toggle-maximize, menu, lower
-- or none.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkTitlebarRightClickPropertyInfo,
#endif
    clearSettingsGtkTitlebarRightClick      ,
    constructSettingsGtkTitlebarRightClick  ,
    getSettingsGtkTitlebarRightClick        ,
    setSettingsGtkTitlebarRightClick        ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkTitlebarRightClick           ,
#endif


-- ** gtkXftAntialias #attr:gtkXftAntialias#
-- | Whether to antialias fonts.
-- 
-- The values are 0 for no, 1 for yes, or -1 for the system default.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkXftAntialiasPropertyInfo     ,
#endif
    constructSettingsGtkXftAntialias        ,
    getSettingsGtkXftAntialias              ,
    setSettingsGtkXftAntialias              ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkXftAntialias                 ,
#endif


-- ** gtkXftDpi #attr:gtkXftDpi#
-- | The font resolution, in 1024 * dots\/inch.
-- 
-- -1 to use the default value.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkXftDpiPropertyInfo           ,
#endif
    constructSettingsGtkXftDpi              ,
    getSettingsGtkXftDpi                    ,
    setSettingsGtkXftDpi                    ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkXftDpi                       ,
#endif


-- ** gtkXftHinting #attr:gtkXftHinting#
-- | Whether to enable font hinting.
-- 
-- The values are 0 for no, 1 for yes, or -1 for the system default.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkXftHintingPropertyInfo       ,
#endif
    constructSettingsGtkXftHinting          ,
    getSettingsGtkXftHinting                ,
    setSettingsGtkXftHinting                ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkXftHinting                   ,
#endif


-- ** gtkXftHintstyle #attr:gtkXftHintstyle#
-- | What degree of font hinting to use.
-- 
-- The possible vaues are hintnone, hintslight,
-- hintmedium, hintfull.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkXftHintstylePropertyInfo     ,
#endif
    clearSettingsGtkXftHintstyle            ,
    constructSettingsGtkXftHintstyle        ,
    getSettingsGtkXftHintstyle              ,
    setSettingsGtkXftHintstyle              ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkXftHintstyle                 ,
#endif


-- ** gtkXftRgba #attr:gtkXftRgba#
-- | The type of subpixel antialiasing to use.
-- 
-- The possible values are none, rgb, bgr, vrgb, vbgr.
-- 
-- Note that GSK does not support subpixel antialiasing, and this
-- setting has no effect on font rendering in GTK.

#if defined(ENABLE_OVERLOADING)
    SettingsGtkXftRgbaPropertyInfo          ,
#endif
    clearSettingsGtkXftRgba                 ,
    constructSettingsGtkXftRgba             ,
    getSettingsGtkXftRgba                   ,
    setSettingsGtkXftRgba                   ,
#if defined(ENABLE_OVERLOADING)
    settingsGtkXftRgba                      ,
#endif




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Objects.Display as Gdk.Display
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.StyleProvider as Gtk.StyleProvider

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Objects.Display as Gdk.Display
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.StyleProvider as Gtk.StyleProvider

#endif

-- | Memory-managed wrapper type.
newtype Settings = Settings (SP.ManagedPtr Settings)
    deriving (Eq)

instance SP.ManagedPtrNewtype Settings where
    toManagedPtr (Settings p) = p

foreign import ccall "gtk_settings_get_type"
    c_gtk_settings_get_type :: IO B.Types.GType

instance B.Types.TypedObject Settings where
    glibType = c_gtk_settings_get_type

instance B.Types.GObject Settings

-- | Type class for types which can be safely cast to t'Settings', for instance with `toSettings`.
class (SP.GObject o, O.IsDescendantOf Settings o) => IsSettings o
instance (SP.GObject o, O.IsDescendantOf Settings o) => IsSettings o

instance O.HasParentTypes Settings
type instance O.ParentTypes Settings = '[GObject.Object.Object, Gtk.StyleProvider.StyleProvider]

-- | Cast to t'Settings', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toSettings :: (MIO.MonadIO m, IsSettings o) => o -> m Settings
toSettings = MIO.liftIO . B.ManagedPtr.unsafeCastTo Settings

-- | Convert t'Settings' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Settings) where
    gvalueGType_ = c_gtk_settings_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Settings)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Settings)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Settings ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveSettingsMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveSettingsMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveSettingsMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveSettingsMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveSettingsMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveSettingsMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveSettingsMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveSettingsMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveSettingsMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveSettingsMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveSettingsMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveSettingsMethod "resetProperty" o = SettingsResetPropertyMethodInfo
    ResolveSettingsMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveSettingsMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveSettingsMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveSettingsMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveSettingsMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveSettingsMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveSettingsMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveSettingsMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveSettingsMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveSettingsMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveSettingsMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveSettingsMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveSettingsMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveSettingsMethod t Settings, O.OverloadedMethod info Settings p) => OL.IsLabel t (Settings -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveSettingsMethod t Settings, O.OverloadedMethod info Settings p, R.HasField t Settings p) => R.HasField t Settings p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveSettingsMethod t Settings, O.OverloadedMethodInfo info Settings) => OL.IsLabel t (O.MethodProxy info Settings) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- VVV Prop "gtk-alternative-button-order"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-alternative-button-order@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkAlternativeButtonOrder
-- @
getSettingsGtkAlternativeButtonOrder :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkAlternativeButtonOrder obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-alternative-button-order"

-- | Set the value of the “@gtk-alternative-button-order@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkAlternativeButtonOrder 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkAlternativeButtonOrder :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkAlternativeButtonOrder obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-alternative-button-order" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-alternative-button-order@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkAlternativeButtonOrder :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkAlternativeButtonOrder val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-alternative-button-order" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkAlternativeButtonOrderPropertyInfo
instance AttrInfo SettingsGtkAlternativeButtonOrderPropertyInfo where
    type AttrAllowedOps SettingsGtkAlternativeButtonOrderPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkAlternativeButtonOrderPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkAlternativeButtonOrderPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkAlternativeButtonOrderPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkAlternativeButtonOrderPropertyInfo = Bool
    type AttrGetType SettingsGtkAlternativeButtonOrderPropertyInfo = Bool
    type AttrLabel SettingsGtkAlternativeButtonOrderPropertyInfo = "gtk-alternative-button-order"
    type AttrOrigin SettingsGtkAlternativeButtonOrderPropertyInfo = Settings
    attrGet = getSettingsGtkAlternativeButtonOrder
    attrSet = setSettingsGtkAlternativeButtonOrder
    attrPut = setSettingsGtkAlternativeButtonOrder
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkAlternativeButtonOrder
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkAlternativeButtonOrder"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkAlternativeButtonOrder"
        })
#endif

-- VVV Prop "gtk-alternative-sort-arrows"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-alternative-sort-arrows@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkAlternativeSortArrows
-- @
getSettingsGtkAlternativeSortArrows :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkAlternativeSortArrows obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-alternative-sort-arrows"

-- | Set the value of the “@gtk-alternative-sort-arrows@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkAlternativeSortArrows 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkAlternativeSortArrows :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkAlternativeSortArrows obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-alternative-sort-arrows" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-alternative-sort-arrows@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkAlternativeSortArrows :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkAlternativeSortArrows val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-alternative-sort-arrows" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkAlternativeSortArrowsPropertyInfo
instance AttrInfo SettingsGtkAlternativeSortArrowsPropertyInfo where
    type AttrAllowedOps SettingsGtkAlternativeSortArrowsPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkAlternativeSortArrowsPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkAlternativeSortArrowsPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkAlternativeSortArrowsPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkAlternativeSortArrowsPropertyInfo = Bool
    type AttrGetType SettingsGtkAlternativeSortArrowsPropertyInfo = Bool
    type AttrLabel SettingsGtkAlternativeSortArrowsPropertyInfo = "gtk-alternative-sort-arrows"
    type AttrOrigin SettingsGtkAlternativeSortArrowsPropertyInfo = Settings
    attrGet = getSettingsGtkAlternativeSortArrows
    attrSet = setSettingsGtkAlternativeSortArrows
    attrPut = setSettingsGtkAlternativeSortArrows
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkAlternativeSortArrows
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkAlternativeSortArrows"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkAlternativeSortArrows"
        })
#endif

-- VVV Prop "gtk-application-prefer-dark-theme"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-application-prefer-dark-theme@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkApplicationPreferDarkTheme
-- @
getSettingsGtkApplicationPreferDarkTheme :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkApplicationPreferDarkTheme obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-application-prefer-dark-theme"

-- | Set the value of the “@gtk-application-prefer-dark-theme@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkApplicationPreferDarkTheme 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkApplicationPreferDarkTheme :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkApplicationPreferDarkTheme obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-application-prefer-dark-theme" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-application-prefer-dark-theme@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkApplicationPreferDarkTheme :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkApplicationPreferDarkTheme val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-application-prefer-dark-theme" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkApplicationPreferDarkThemePropertyInfo
instance AttrInfo SettingsGtkApplicationPreferDarkThemePropertyInfo where
    type AttrAllowedOps SettingsGtkApplicationPreferDarkThemePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkApplicationPreferDarkThemePropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkApplicationPreferDarkThemePropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkApplicationPreferDarkThemePropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkApplicationPreferDarkThemePropertyInfo = Bool
    type AttrGetType SettingsGtkApplicationPreferDarkThemePropertyInfo = Bool
    type AttrLabel SettingsGtkApplicationPreferDarkThemePropertyInfo = "gtk-application-prefer-dark-theme"
    type AttrOrigin SettingsGtkApplicationPreferDarkThemePropertyInfo = Settings
    attrGet = getSettingsGtkApplicationPreferDarkTheme
    attrSet = setSettingsGtkApplicationPreferDarkTheme
    attrPut = setSettingsGtkApplicationPreferDarkTheme
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkApplicationPreferDarkTheme
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkApplicationPreferDarkTheme"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkApplicationPreferDarkTheme"
        })
#endif

-- VVV Prop "gtk-cursor-aspect-ratio"
   -- Type: TBasicType TDouble
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-cursor-aspect-ratio@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkCursorAspectRatio
-- @
getSettingsGtkCursorAspectRatio :: (MonadIO m, IsSettings o) => o -> m Double
getSettingsGtkCursorAspectRatio obj = MIO.liftIO $ B.Properties.getObjectPropertyDouble obj "gtk-cursor-aspect-ratio"

-- | Set the value of the “@gtk-cursor-aspect-ratio@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkCursorAspectRatio 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkCursorAspectRatio :: (MonadIO m, IsSettings o) => o -> Double -> m ()
setSettingsGtkCursorAspectRatio obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyDouble obj "gtk-cursor-aspect-ratio" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-cursor-aspect-ratio@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkCursorAspectRatio :: (IsSettings o, MIO.MonadIO m) => Double -> m (GValueConstruct o)
constructSettingsGtkCursorAspectRatio val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyDouble "gtk-cursor-aspect-ratio" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkCursorAspectRatioPropertyInfo
instance AttrInfo SettingsGtkCursorAspectRatioPropertyInfo where
    type AttrAllowedOps SettingsGtkCursorAspectRatioPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkCursorAspectRatioPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkCursorAspectRatioPropertyInfo = (~) Double
    type AttrTransferTypeConstraint SettingsGtkCursorAspectRatioPropertyInfo = (~) Double
    type AttrTransferType SettingsGtkCursorAspectRatioPropertyInfo = Double
    type AttrGetType SettingsGtkCursorAspectRatioPropertyInfo = Double
    type AttrLabel SettingsGtkCursorAspectRatioPropertyInfo = "gtk-cursor-aspect-ratio"
    type AttrOrigin SettingsGtkCursorAspectRatioPropertyInfo = Settings
    attrGet = getSettingsGtkCursorAspectRatio
    attrSet = setSettingsGtkCursorAspectRatio
    attrPut = setSettingsGtkCursorAspectRatio
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkCursorAspectRatio
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkCursorAspectRatio"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkCursorAspectRatio"
        })
#endif

-- VVV Prop "gtk-cursor-blink"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-cursor-blink@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkCursorBlink
-- @
getSettingsGtkCursorBlink :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkCursorBlink obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-cursor-blink"

-- | Set the value of the “@gtk-cursor-blink@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkCursorBlink 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkCursorBlink :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkCursorBlink obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-cursor-blink" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-cursor-blink@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkCursorBlink :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkCursorBlink val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-cursor-blink" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkCursorBlinkPropertyInfo
instance AttrInfo SettingsGtkCursorBlinkPropertyInfo where
    type AttrAllowedOps SettingsGtkCursorBlinkPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkCursorBlinkPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkCursorBlinkPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkCursorBlinkPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkCursorBlinkPropertyInfo = Bool
    type AttrGetType SettingsGtkCursorBlinkPropertyInfo = Bool
    type AttrLabel SettingsGtkCursorBlinkPropertyInfo = "gtk-cursor-blink"
    type AttrOrigin SettingsGtkCursorBlinkPropertyInfo = Settings
    attrGet = getSettingsGtkCursorBlink
    attrSet = setSettingsGtkCursorBlink
    attrPut = setSettingsGtkCursorBlink
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkCursorBlink
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkCursorBlink"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkCursorBlink"
        })
#endif

-- VVV Prop "gtk-cursor-blink-time"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-cursor-blink-time@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkCursorBlinkTime
-- @
getSettingsGtkCursorBlinkTime :: (MonadIO m, IsSettings o) => o -> m Int32
getSettingsGtkCursorBlinkTime obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "gtk-cursor-blink-time"

-- | Set the value of the “@gtk-cursor-blink-time@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkCursorBlinkTime 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkCursorBlinkTime :: (MonadIO m, IsSettings o) => o -> Int32 -> m ()
setSettingsGtkCursorBlinkTime obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "gtk-cursor-blink-time" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-cursor-blink-time@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkCursorBlinkTime :: (IsSettings o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructSettingsGtkCursorBlinkTime val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "gtk-cursor-blink-time" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkCursorBlinkTimePropertyInfo
instance AttrInfo SettingsGtkCursorBlinkTimePropertyInfo where
    type AttrAllowedOps SettingsGtkCursorBlinkTimePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkCursorBlinkTimePropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkCursorBlinkTimePropertyInfo = (~) Int32
    type AttrTransferTypeConstraint SettingsGtkCursorBlinkTimePropertyInfo = (~) Int32
    type AttrTransferType SettingsGtkCursorBlinkTimePropertyInfo = Int32
    type AttrGetType SettingsGtkCursorBlinkTimePropertyInfo = Int32
    type AttrLabel SettingsGtkCursorBlinkTimePropertyInfo = "gtk-cursor-blink-time"
    type AttrOrigin SettingsGtkCursorBlinkTimePropertyInfo = Settings
    attrGet = getSettingsGtkCursorBlinkTime
    attrSet = setSettingsGtkCursorBlinkTime
    attrPut = setSettingsGtkCursorBlinkTime
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkCursorBlinkTime
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkCursorBlinkTime"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkCursorBlinkTime"
        })
#endif

-- VVV Prop "gtk-cursor-blink-timeout"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-cursor-blink-timeout@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkCursorBlinkTimeout
-- @
getSettingsGtkCursorBlinkTimeout :: (MonadIO m, IsSettings o) => o -> m Int32
getSettingsGtkCursorBlinkTimeout obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "gtk-cursor-blink-timeout"

-- | Set the value of the “@gtk-cursor-blink-timeout@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkCursorBlinkTimeout 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkCursorBlinkTimeout :: (MonadIO m, IsSettings o) => o -> Int32 -> m ()
setSettingsGtkCursorBlinkTimeout obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "gtk-cursor-blink-timeout" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-cursor-blink-timeout@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkCursorBlinkTimeout :: (IsSettings o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructSettingsGtkCursorBlinkTimeout val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "gtk-cursor-blink-timeout" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkCursorBlinkTimeoutPropertyInfo
instance AttrInfo SettingsGtkCursorBlinkTimeoutPropertyInfo where
    type AttrAllowedOps SettingsGtkCursorBlinkTimeoutPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkCursorBlinkTimeoutPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkCursorBlinkTimeoutPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint SettingsGtkCursorBlinkTimeoutPropertyInfo = (~) Int32
    type AttrTransferType SettingsGtkCursorBlinkTimeoutPropertyInfo = Int32
    type AttrGetType SettingsGtkCursorBlinkTimeoutPropertyInfo = Int32
    type AttrLabel SettingsGtkCursorBlinkTimeoutPropertyInfo = "gtk-cursor-blink-timeout"
    type AttrOrigin SettingsGtkCursorBlinkTimeoutPropertyInfo = Settings
    attrGet = getSettingsGtkCursorBlinkTimeout
    attrSet = setSettingsGtkCursorBlinkTimeout
    attrPut = setSettingsGtkCursorBlinkTimeout
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkCursorBlinkTimeout
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkCursorBlinkTimeout"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkCursorBlinkTimeout"
        })
#endif

-- VVV Prop "gtk-cursor-theme-name"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-cursor-theme-name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkCursorThemeName
-- @
getSettingsGtkCursorThemeName :: (MonadIO m, IsSettings o) => o -> m (Maybe T.Text)
getSettingsGtkCursorThemeName obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "gtk-cursor-theme-name"

-- | Set the value of the “@gtk-cursor-theme-name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkCursorThemeName 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkCursorThemeName :: (MonadIO m, IsSettings o) => o -> T.Text -> m ()
setSettingsGtkCursorThemeName obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "gtk-cursor-theme-name" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-cursor-theme-name@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkCursorThemeName :: (IsSettings o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructSettingsGtkCursorThemeName val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "gtk-cursor-theme-name" (P.Just val)

-- | Set the value of the “@gtk-cursor-theme-name@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #gtkCursorThemeName
-- @
clearSettingsGtkCursorThemeName :: (MonadIO m, IsSettings o) => o -> m ()
clearSettingsGtkCursorThemeName obj = liftIO $ B.Properties.setObjectPropertyString obj "gtk-cursor-theme-name" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data SettingsGtkCursorThemeNamePropertyInfo
instance AttrInfo SettingsGtkCursorThemeNamePropertyInfo where
    type AttrAllowedOps SettingsGtkCursorThemeNamePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsGtkCursorThemeNamePropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkCursorThemeNamePropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint SettingsGtkCursorThemeNamePropertyInfo = (~) T.Text
    type AttrTransferType SettingsGtkCursorThemeNamePropertyInfo = T.Text
    type AttrGetType SettingsGtkCursorThemeNamePropertyInfo = (Maybe T.Text)
    type AttrLabel SettingsGtkCursorThemeNamePropertyInfo = "gtk-cursor-theme-name"
    type AttrOrigin SettingsGtkCursorThemeNamePropertyInfo = Settings
    attrGet = getSettingsGtkCursorThemeName
    attrSet = setSettingsGtkCursorThemeName
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkCursorThemeName
    attrClear = clearSettingsGtkCursorThemeName
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkCursorThemeName"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkCursorThemeName"
        })
#endif

-- VVV Prop "gtk-cursor-theme-size"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-cursor-theme-size@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkCursorThemeSize
-- @
getSettingsGtkCursorThemeSize :: (MonadIO m, IsSettings o) => o -> m Int32
getSettingsGtkCursorThemeSize obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "gtk-cursor-theme-size"

-- | Set the value of the “@gtk-cursor-theme-size@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkCursorThemeSize 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkCursorThemeSize :: (MonadIO m, IsSettings o) => o -> Int32 -> m ()
setSettingsGtkCursorThemeSize obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "gtk-cursor-theme-size" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-cursor-theme-size@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkCursorThemeSize :: (IsSettings o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructSettingsGtkCursorThemeSize val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "gtk-cursor-theme-size" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkCursorThemeSizePropertyInfo
instance AttrInfo SettingsGtkCursorThemeSizePropertyInfo where
    type AttrAllowedOps SettingsGtkCursorThemeSizePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkCursorThemeSizePropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkCursorThemeSizePropertyInfo = (~) Int32
    type AttrTransferTypeConstraint SettingsGtkCursorThemeSizePropertyInfo = (~) Int32
    type AttrTransferType SettingsGtkCursorThemeSizePropertyInfo = Int32
    type AttrGetType SettingsGtkCursorThemeSizePropertyInfo = Int32
    type AttrLabel SettingsGtkCursorThemeSizePropertyInfo = "gtk-cursor-theme-size"
    type AttrOrigin SettingsGtkCursorThemeSizePropertyInfo = Settings
    attrGet = getSettingsGtkCursorThemeSize
    attrSet = setSettingsGtkCursorThemeSize
    attrPut = setSettingsGtkCursorThemeSize
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkCursorThemeSize
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkCursorThemeSize"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkCursorThemeSize"
        })
#endif

-- VVV Prop "gtk-decoration-layout"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-decoration-layout@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkDecorationLayout
-- @
getSettingsGtkDecorationLayout :: (MonadIO m, IsSettings o) => o -> m (Maybe T.Text)
getSettingsGtkDecorationLayout obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "gtk-decoration-layout"

-- | Set the value of the “@gtk-decoration-layout@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkDecorationLayout 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkDecorationLayout :: (MonadIO m, IsSettings o) => o -> T.Text -> m ()
setSettingsGtkDecorationLayout obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "gtk-decoration-layout" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-decoration-layout@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkDecorationLayout :: (IsSettings o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructSettingsGtkDecorationLayout val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "gtk-decoration-layout" (P.Just val)

-- | Set the value of the “@gtk-decoration-layout@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #gtkDecorationLayout
-- @
clearSettingsGtkDecorationLayout :: (MonadIO m, IsSettings o) => o -> m ()
clearSettingsGtkDecorationLayout obj = liftIO $ B.Properties.setObjectPropertyString obj "gtk-decoration-layout" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data SettingsGtkDecorationLayoutPropertyInfo
instance AttrInfo SettingsGtkDecorationLayoutPropertyInfo where
    type AttrAllowedOps SettingsGtkDecorationLayoutPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsGtkDecorationLayoutPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkDecorationLayoutPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint SettingsGtkDecorationLayoutPropertyInfo = (~) T.Text
    type AttrTransferType SettingsGtkDecorationLayoutPropertyInfo = T.Text
    type AttrGetType SettingsGtkDecorationLayoutPropertyInfo = (Maybe T.Text)
    type AttrLabel SettingsGtkDecorationLayoutPropertyInfo = "gtk-decoration-layout"
    type AttrOrigin SettingsGtkDecorationLayoutPropertyInfo = Settings
    attrGet = getSettingsGtkDecorationLayout
    attrSet = setSettingsGtkDecorationLayout
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkDecorationLayout
    attrClear = clearSettingsGtkDecorationLayout
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkDecorationLayout"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkDecorationLayout"
        })
#endif

-- VVV Prop "gtk-dialogs-use-header"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-dialogs-use-header@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkDialogsUseHeader
-- @
getSettingsGtkDialogsUseHeader :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkDialogsUseHeader obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-dialogs-use-header"

-- | Set the value of the “@gtk-dialogs-use-header@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkDialogsUseHeader 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkDialogsUseHeader :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkDialogsUseHeader obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-dialogs-use-header" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-dialogs-use-header@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkDialogsUseHeader :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkDialogsUseHeader val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-dialogs-use-header" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkDialogsUseHeaderPropertyInfo
instance AttrInfo SettingsGtkDialogsUseHeaderPropertyInfo where
    type AttrAllowedOps SettingsGtkDialogsUseHeaderPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkDialogsUseHeaderPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkDialogsUseHeaderPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkDialogsUseHeaderPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkDialogsUseHeaderPropertyInfo = Bool
    type AttrGetType SettingsGtkDialogsUseHeaderPropertyInfo = Bool
    type AttrLabel SettingsGtkDialogsUseHeaderPropertyInfo = "gtk-dialogs-use-header"
    type AttrOrigin SettingsGtkDialogsUseHeaderPropertyInfo = Settings
    attrGet = getSettingsGtkDialogsUseHeader
    attrSet = setSettingsGtkDialogsUseHeader
    attrPut = setSettingsGtkDialogsUseHeader
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkDialogsUseHeader
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkDialogsUseHeader"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkDialogsUseHeader"
        })
#endif

-- VVV Prop "gtk-dnd-drag-threshold"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-dnd-drag-threshold@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkDndDragThreshold
-- @
getSettingsGtkDndDragThreshold :: (MonadIO m, IsSettings o) => o -> m Int32
getSettingsGtkDndDragThreshold obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "gtk-dnd-drag-threshold"

-- | Set the value of the “@gtk-dnd-drag-threshold@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkDndDragThreshold 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkDndDragThreshold :: (MonadIO m, IsSettings o) => o -> Int32 -> m ()
setSettingsGtkDndDragThreshold obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "gtk-dnd-drag-threshold" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-dnd-drag-threshold@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkDndDragThreshold :: (IsSettings o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructSettingsGtkDndDragThreshold val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "gtk-dnd-drag-threshold" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkDndDragThresholdPropertyInfo
instance AttrInfo SettingsGtkDndDragThresholdPropertyInfo where
    type AttrAllowedOps SettingsGtkDndDragThresholdPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkDndDragThresholdPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkDndDragThresholdPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint SettingsGtkDndDragThresholdPropertyInfo = (~) Int32
    type AttrTransferType SettingsGtkDndDragThresholdPropertyInfo = Int32
    type AttrGetType SettingsGtkDndDragThresholdPropertyInfo = Int32
    type AttrLabel SettingsGtkDndDragThresholdPropertyInfo = "gtk-dnd-drag-threshold"
    type AttrOrigin SettingsGtkDndDragThresholdPropertyInfo = Settings
    attrGet = getSettingsGtkDndDragThreshold
    attrSet = setSettingsGtkDndDragThreshold
    attrPut = setSettingsGtkDndDragThreshold
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkDndDragThreshold
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkDndDragThreshold"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkDndDragThreshold"
        })
#endif

-- VVV Prop "gtk-double-click-distance"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-double-click-distance@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkDoubleClickDistance
-- @
getSettingsGtkDoubleClickDistance :: (MonadIO m, IsSettings o) => o -> m Int32
getSettingsGtkDoubleClickDistance obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "gtk-double-click-distance"

-- | Set the value of the “@gtk-double-click-distance@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkDoubleClickDistance 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkDoubleClickDistance :: (MonadIO m, IsSettings o) => o -> Int32 -> m ()
setSettingsGtkDoubleClickDistance obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "gtk-double-click-distance" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-double-click-distance@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkDoubleClickDistance :: (IsSettings o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructSettingsGtkDoubleClickDistance val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "gtk-double-click-distance" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkDoubleClickDistancePropertyInfo
instance AttrInfo SettingsGtkDoubleClickDistancePropertyInfo where
    type AttrAllowedOps SettingsGtkDoubleClickDistancePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkDoubleClickDistancePropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkDoubleClickDistancePropertyInfo = (~) Int32
    type AttrTransferTypeConstraint SettingsGtkDoubleClickDistancePropertyInfo = (~) Int32
    type AttrTransferType SettingsGtkDoubleClickDistancePropertyInfo = Int32
    type AttrGetType SettingsGtkDoubleClickDistancePropertyInfo = Int32
    type AttrLabel SettingsGtkDoubleClickDistancePropertyInfo = "gtk-double-click-distance"
    type AttrOrigin SettingsGtkDoubleClickDistancePropertyInfo = Settings
    attrGet = getSettingsGtkDoubleClickDistance
    attrSet = setSettingsGtkDoubleClickDistance
    attrPut = setSettingsGtkDoubleClickDistance
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkDoubleClickDistance
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkDoubleClickDistance"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkDoubleClickDistance"
        })
#endif

-- VVV Prop "gtk-double-click-time"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-double-click-time@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkDoubleClickTime
-- @
getSettingsGtkDoubleClickTime :: (MonadIO m, IsSettings o) => o -> m Int32
getSettingsGtkDoubleClickTime obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "gtk-double-click-time"

-- | Set the value of the “@gtk-double-click-time@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkDoubleClickTime 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkDoubleClickTime :: (MonadIO m, IsSettings o) => o -> Int32 -> m ()
setSettingsGtkDoubleClickTime obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "gtk-double-click-time" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-double-click-time@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkDoubleClickTime :: (IsSettings o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructSettingsGtkDoubleClickTime val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "gtk-double-click-time" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkDoubleClickTimePropertyInfo
instance AttrInfo SettingsGtkDoubleClickTimePropertyInfo where
    type AttrAllowedOps SettingsGtkDoubleClickTimePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkDoubleClickTimePropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkDoubleClickTimePropertyInfo = (~) Int32
    type AttrTransferTypeConstraint SettingsGtkDoubleClickTimePropertyInfo = (~) Int32
    type AttrTransferType SettingsGtkDoubleClickTimePropertyInfo = Int32
    type AttrGetType SettingsGtkDoubleClickTimePropertyInfo = Int32
    type AttrLabel SettingsGtkDoubleClickTimePropertyInfo = "gtk-double-click-time"
    type AttrOrigin SettingsGtkDoubleClickTimePropertyInfo = Settings
    attrGet = getSettingsGtkDoubleClickTime
    attrSet = setSettingsGtkDoubleClickTime
    attrPut = setSettingsGtkDoubleClickTime
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkDoubleClickTime
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkDoubleClickTime"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkDoubleClickTime"
        })
#endif

-- VVV Prop "gtk-enable-accels"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-enable-accels@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkEnableAccels
-- @
getSettingsGtkEnableAccels :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkEnableAccels obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-enable-accels"

-- | Set the value of the “@gtk-enable-accels@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkEnableAccels 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkEnableAccels :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkEnableAccels obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-enable-accels" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-enable-accels@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkEnableAccels :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkEnableAccels val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-enable-accels" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkEnableAccelsPropertyInfo
instance AttrInfo SettingsGtkEnableAccelsPropertyInfo where
    type AttrAllowedOps SettingsGtkEnableAccelsPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkEnableAccelsPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkEnableAccelsPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkEnableAccelsPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkEnableAccelsPropertyInfo = Bool
    type AttrGetType SettingsGtkEnableAccelsPropertyInfo = Bool
    type AttrLabel SettingsGtkEnableAccelsPropertyInfo = "gtk-enable-accels"
    type AttrOrigin SettingsGtkEnableAccelsPropertyInfo = Settings
    attrGet = getSettingsGtkEnableAccels
    attrSet = setSettingsGtkEnableAccels
    attrPut = setSettingsGtkEnableAccels
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkEnableAccels
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkEnableAccels"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkEnableAccels"
        })
#endif

-- VVV Prop "gtk-enable-animations"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-enable-animations@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkEnableAnimations
-- @
getSettingsGtkEnableAnimations :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkEnableAnimations obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-enable-animations"

-- | Set the value of the “@gtk-enable-animations@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkEnableAnimations 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkEnableAnimations :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkEnableAnimations obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-enable-animations" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-enable-animations@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkEnableAnimations :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkEnableAnimations val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-enable-animations" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkEnableAnimationsPropertyInfo
instance AttrInfo SettingsGtkEnableAnimationsPropertyInfo where
    type AttrAllowedOps SettingsGtkEnableAnimationsPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkEnableAnimationsPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkEnableAnimationsPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkEnableAnimationsPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkEnableAnimationsPropertyInfo = Bool
    type AttrGetType SettingsGtkEnableAnimationsPropertyInfo = Bool
    type AttrLabel SettingsGtkEnableAnimationsPropertyInfo = "gtk-enable-animations"
    type AttrOrigin SettingsGtkEnableAnimationsPropertyInfo = Settings
    attrGet = getSettingsGtkEnableAnimations
    attrSet = setSettingsGtkEnableAnimations
    attrPut = setSettingsGtkEnableAnimations
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkEnableAnimations
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkEnableAnimations"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkEnableAnimations"
        })
#endif

-- VVV Prop "gtk-enable-event-sounds"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-enable-event-sounds@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkEnableEventSounds
-- @
getSettingsGtkEnableEventSounds :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkEnableEventSounds obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-enable-event-sounds"

-- | Set the value of the “@gtk-enable-event-sounds@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkEnableEventSounds 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkEnableEventSounds :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkEnableEventSounds obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-enable-event-sounds" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-enable-event-sounds@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkEnableEventSounds :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkEnableEventSounds val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-enable-event-sounds" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkEnableEventSoundsPropertyInfo
instance AttrInfo SettingsGtkEnableEventSoundsPropertyInfo where
    type AttrAllowedOps SettingsGtkEnableEventSoundsPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkEnableEventSoundsPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkEnableEventSoundsPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkEnableEventSoundsPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkEnableEventSoundsPropertyInfo = Bool
    type AttrGetType SettingsGtkEnableEventSoundsPropertyInfo = Bool
    type AttrLabel SettingsGtkEnableEventSoundsPropertyInfo = "gtk-enable-event-sounds"
    type AttrOrigin SettingsGtkEnableEventSoundsPropertyInfo = Settings
    attrGet = getSettingsGtkEnableEventSounds
    attrSet = setSettingsGtkEnableEventSounds
    attrPut = setSettingsGtkEnableEventSounds
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkEnableEventSounds
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkEnableEventSounds"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkEnableEventSounds"
        })
#endif

-- VVV Prop "gtk-enable-input-feedback-sounds"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-enable-input-feedback-sounds@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkEnableInputFeedbackSounds
-- @
getSettingsGtkEnableInputFeedbackSounds :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkEnableInputFeedbackSounds obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-enable-input-feedback-sounds"

-- | Set the value of the “@gtk-enable-input-feedback-sounds@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkEnableInputFeedbackSounds 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkEnableInputFeedbackSounds :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkEnableInputFeedbackSounds obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-enable-input-feedback-sounds" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-enable-input-feedback-sounds@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkEnableInputFeedbackSounds :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkEnableInputFeedbackSounds val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-enable-input-feedback-sounds" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkEnableInputFeedbackSoundsPropertyInfo
instance AttrInfo SettingsGtkEnableInputFeedbackSoundsPropertyInfo where
    type AttrAllowedOps SettingsGtkEnableInputFeedbackSoundsPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkEnableInputFeedbackSoundsPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkEnableInputFeedbackSoundsPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkEnableInputFeedbackSoundsPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkEnableInputFeedbackSoundsPropertyInfo = Bool
    type AttrGetType SettingsGtkEnableInputFeedbackSoundsPropertyInfo = Bool
    type AttrLabel SettingsGtkEnableInputFeedbackSoundsPropertyInfo = "gtk-enable-input-feedback-sounds"
    type AttrOrigin SettingsGtkEnableInputFeedbackSoundsPropertyInfo = Settings
    attrGet = getSettingsGtkEnableInputFeedbackSounds
    attrSet = setSettingsGtkEnableInputFeedbackSounds
    attrPut = setSettingsGtkEnableInputFeedbackSounds
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkEnableInputFeedbackSounds
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkEnableInputFeedbackSounds"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkEnableInputFeedbackSounds"
        })
#endif

-- VVV Prop "gtk-enable-primary-paste"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-enable-primary-paste@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkEnablePrimaryPaste
-- @
getSettingsGtkEnablePrimaryPaste :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkEnablePrimaryPaste obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-enable-primary-paste"

-- | Set the value of the “@gtk-enable-primary-paste@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkEnablePrimaryPaste 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkEnablePrimaryPaste :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkEnablePrimaryPaste obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-enable-primary-paste" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-enable-primary-paste@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkEnablePrimaryPaste :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkEnablePrimaryPaste val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-enable-primary-paste" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkEnablePrimaryPastePropertyInfo
instance AttrInfo SettingsGtkEnablePrimaryPastePropertyInfo where
    type AttrAllowedOps SettingsGtkEnablePrimaryPastePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkEnablePrimaryPastePropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkEnablePrimaryPastePropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkEnablePrimaryPastePropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkEnablePrimaryPastePropertyInfo = Bool
    type AttrGetType SettingsGtkEnablePrimaryPastePropertyInfo = Bool
    type AttrLabel SettingsGtkEnablePrimaryPastePropertyInfo = "gtk-enable-primary-paste"
    type AttrOrigin SettingsGtkEnablePrimaryPastePropertyInfo = Settings
    attrGet = getSettingsGtkEnablePrimaryPaste
    attrSet = setSettingsGtkEnablePrimaryPaste
    attrPut = setSettingsGtkEnablePrimaryPaste
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkEnablePrimaryPaste
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkEnablePrimaryPaste"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkEnablePrimaryPaste"
        })
#endif

-- VVV Prop "gtk-entry-password-hint-timeout"
   -- Type: TBasicType TUInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-entry-password-hint-timeout@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkEntryPasswordHintTimeout
-- @
getSettingsGtkEntryPasswordHintTimeout :: (MonadIO m, IsSettings o) => o -> m Word32
getSettingsGtkEntryPasswordHintTimeout obj = MIO.liftIO $ B.Properties.getObjectPropertyUInt32 obj "gtk-entry-password-hint-timeout"

-- | Set the value of the “@gtk-entry-password-hint-timeout@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkEntryPasswordHintTimeout 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkEntryPasswordHintTimeout :: (MonadIO m, IsSettings o) => o -> Word32 -> m ()
setSettingsGtkEntryPasswordHintTimeout obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyUInt32 obj "gtk-entry-password-hint-timeout" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-entry-password-hint-timeout@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkEntryPasswordHintTimeout :: (IsSettings o, MIO.MonadIO m) => Word32 -> m (GValueConstruct o)
constructSettingsGtkEntryPasswordHintTimeout val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyUInt32 "gtk-entry-password-hint-timeout" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkEntryPasswordHintTimeoutPropertyInfo
instance AttrInfo SettingsGtkEntryPasswordHintTimeoutPropertyInfo where
    type AttrAllowedOps SettingsGtkEntryPasswordHintTimeoutPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkEntryPasswordHintTimeoutPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkEntryPasswordHintTimeoutPropertyInfo = (~) Word32
    type AttrTransferTypeConstraint SettingsGtkEntryPasswordHintTimeoutPropertyInfo = (~) Word32
    type AttrTransferType SettingsGtkEntryPasswordHintTimeoutPropertyInfo = Word32
    type AttrGetType SettingsGtkEntryPasswordHintTimeoutPropertyInfo = Word32
    type AttrLabel SettingsGtkEntryPasswordHintTimeoutPropertyInfo = "gtk-entry-password-hint-timeout"
    type AttrOrigin SettingsGtkEntryPasswordHintTimeoutPropertyInfo = Settings
    attrGet = getSettingsGtkEntryPasswordHintTimeout
    attrSet = setSettingsGtkEntryPasswordHintTimeout
    attrPut = setSettingsGtkEntryPasswordHintTimeout
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkEntryPasswordHintTimeout
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkEntryPasswordHintTimeout"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkEntryPasswordHintTimeout"
        })
#endif

-- VVV Prop "gtk-entry-select-on-focus"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-entry-select-on-focus@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkEntrySelectOnFocus
-- @
getSettingsGtkEntrySelectOnFocus :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkEntrySelectOnFocus obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-entry-select-on-focus"

-- | Set the value of the “@gtk-entry-select-on-focus@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkEntrySelectOnFocus 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkEntrySelectOnFocus :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkEntrySelectOnFocus obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-entry-select-on-focus" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-entry-select-on-focus@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkEntrySelectOnFocus :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkEntrySelectOnFocus val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-entry-select-on-focus" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkEntrySelectOnFocusPropertyInfo
instance AttrInfo SettingsGtkEntrySelectOnFocusPropertyInfo where
    type AttrAllowedOps SettingsGtkEntrySelectOnFocusPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkEntrySelectOnFocusPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkEntrySelectOnFocusPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkEntrySelectOnFocusPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkEntrySelectOnFocusPropertyInfo = Bool
    type AttrGetType SettingsGtkEntrySelectOnFocusPropertyInfo = Bool
    type AttrLabel SettingsGtkEntrySelectOnFocusPropertyInfo = "gtk-entry-select-on-focus"
    type AttrOrigin SettingsGtkEntrySelectOnFocusPropertyInfo = Settings
    attrGet = getSettingsGtkEntrySelectOnFocus
    attrSet = setSettingsGtkEntrySelectOnFocus
    attrPut = setSettingsGtkEntrySelectOnFocus
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkEntrySelectOnFocus
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkEntrySelectOnFocus"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkEntrySelectOnFocus"
        })
#endif

-- VVV Prop "gtk-error-bell"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-error-bell@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkErrorBell
-- @
getSettingsGtkErrorBell :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkErrorBell obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-error-bell"

-- | Set the value of the “@gtk-error-bell@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkErrorBell 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkErrorBell :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkErrorBell obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-error-bell" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-error-bell@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkErrorBell :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkErrorBell val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-error-bell" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkErrorBellPropertyInfo
instance AttrInfo SettingsGtkErrorBellPropertyInfo where
    type AttrAllowedOps SettingsGtkErrorBellPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkErrorBellPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkErrorBellPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkErrorBellPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkErrorBellPropertyInfo = Bool
    type AttrGetType SettingsGtkErrorBellPropertyInfo = Bool
    type AttrLabel SettingsGtkErrorBellPropertyInfo = "gtk-error-bell"
    type AttrOrigin SettingsGtkErrorBellPropertyInfo = Settings
    attrGet = getSettingsGtkErrorBell
    attrSet = setSettingsGtkErrorBell
    attrPut = setSettingsGtkErrorBell
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkErrorBell
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkErrorBell"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkErrorBell"
        })
#endif

-- VVV Prop "gtk-font-name"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-font-name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkFontName
-- @
getSettingsGtkFontName :: (MonadIO m, IsSettings o) => o -> m (Maybe T.Text)
getSettingsGtkFontName obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "gtk-font-name"

-- | Set the value of the “@gtk-font-name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkFontName 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkFontName :: (MonadIO m, IsSettings o) => o -> T.Text -> m ()
setSettingsGtkFontName obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "gtk-font-name" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-font-name@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkFontName :: (IsSettings o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructSettingsGtkFontName val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "gtk-font-name" (P.Just val)

-- | Set the value of the “@gtk-font-name@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #gtkFontName
-- @
clearSettingsGtkFontName :: (MonadIO m, IsSettings o) => o -> m ()
clearSettingsGtkFontName obj = liftIO $ B.Properties.setObjectPropertyString obj "gtk-font-name" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data SettingsGtkFontNamePropertyInfo
instance AttrInfo SettingsGtkFontNamePropertyInfo where
    type AttrAllowedOps SettingsGtkFontNamePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsGtkFontNamePropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkFontNamePropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint SettingsGtkFontNamePropertyInfo = (~) T.Text
    type AttrTransferType SettingsGtkFontNamePropertyInfo = T.Text
    type AttrGetType SettingsGtkFontNamePropertyInfo = (Maybe T.Text)
    type AttrLabel SettingsGtkFontNamePropertyInfo = "gtk-font-name"
    type AttrOrigin SettingsGtkFontNamePropertyInfo = Settings
    attrGet = getSettingsGtkFontName
    attrSet = setSettingsGtkFontName
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkFontName
    attrClear = clearSettingsGtkFontName
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkFontName"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkFontName"
        })
#endif

-- VVV Prop "gtk-font-rendering"
   -- Type: TInterface (Name {namespace = "Gtk", name = "FontRendering"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-font-rendering@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkFontRendering
-- @
getSettingsGtkFontRendering :: (MonadIO m, IsSettings o) => o -> m Gtk.Enums.FontRendering
getSettingsGtkFontRendering obj = MIO.liftIO $ B.Properties.getObjectPropertyEnum obj "gtk-font-rendering"

-- | Set the value of the “@gtk-font-rendering@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkFontRendering 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkFontRendering :: (MonadIO m, IsSettings o) => o -> Gtk.Enums.FontRendering -> m ()
setSettingsGtkFontRendering obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyEnum obj "gtk-font-rendering" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-font-rendering@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkFontRendering :: (IsSettings o, MIO.MonadIO m) => Gtk.Enums.FontRendering -> m (GValueConstruct o)
constructSettingsGtkFontRendering val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyEnum "gtk-font-rendering" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkFontRenderingPropertyInfo
instance AttrInfo SettingsGtkFontRenderingPropertyInfo where
    type AttrAllowedOps SettingsGtkFontRenderingPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkFontRenderingPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkFontRenderingPropertyInfo = (~) Gtk.Enums.FontRendering
    type AttrTransferTypeConstraint SettingsGtkFontRenderingPropertyInfo = (~) Gtk.Enums.FontRendering
    type AttrTransferType SettingsGtkFontRenderingPropertyInfo = Gtk.Enums.FontRendering
    type AttrGetType SettingsGtkFontRenderingPropertyInfo = Gtk.Enums.FontRendering
    type AttrLabel SettingsGtkFontRenderingPropertyInfo = "gtk-font-rendering"
    type AttrOrigin SettingsGtkFontRenderingPropertyInfo = Settings
    attrGet = getSettingsGtkFontRendering
    attrSet = setSettingsGtkFontRendering
    attrPut = setSettingsGtkFontRendering
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkFontRendering
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkFontRendering"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkFontRendering"
        })
#endif

-- VVV Prop "gtk-fontconfig-timestamp"
   -- Type: TBasicType TUInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-fontconfig-timestamp@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkFontconfigTimestamp
-- @
getSettingsGtkFontconfigTimestamp :: (MonadIO m, IsSettings o) => o -> m Word32
getSettingsGtkFontconfigTimestamp obj = MIO.liftIO $ B.Properties.getObjectPropertyUInt32 obj "gtk-fontconfig-timestamp"

-- | Set the value of the “@gtk-fontconfig-timestamp@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkFontconfigTimestamp 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkFontconfigTimestamp :: (MonadIO m, IsSettings o) => o -> Word32 -> m ()
setSettingsGtkFontconfigTimestamp obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyUInt32 obj "gtk-fontconfig-timestamp" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-fontconfig-timestamp@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkFontconfigTimestamp :: (IsSettings o, MIO.MonadIO m) => Word32 -> m (GValueConstruct o)
constructSettingsGtkFontconfigTimestamp val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyUInt32 "gtk-fontconfig-timestamp" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkFontconfigTimestampPropertyInfo
instance AttrInfo SettingsGtkFontconfigTimestampPropertyInfo where
    type AttrAllowedOps SettingsGtkFontconfigTimestampPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkFontconfigTimestampPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkFontconfigTimestampPropertyInfo = (~) Word32
    type AttrTransferTypeConstraint SettingsGtkFontconfigTimestampPropertyInfo = (~) Word32
    type AttrTransferType SettingsGtkFontconfigTimestampPropertyInfo = Word32
    type AttrGetType SettingsGtkFontconfigTimestampPropertyInfo = Word32
    type AttrLabel SettingsGtkFontconfigTimestampPropertyInfo = "gtk-fontconfig-timestamp"
    type AttrOrigin SettingsGtkFontconfigTimestampPropertyInfo = Settings
    attrGet = getSettingsGtkFontconfigTimestamp
    attrSet = setSettingsGtkFontconfigTimestamp
    attrPut = setSettingsGtkFontconfigTimestamp
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkFontconfigTimestamp
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkFontconfigTimestamp"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkFontconfigTimestamp"
        })
#endif

-- VVV Prop "gtk-hint-font-metrics"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-hint-font-metrics@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkHintFontMetrics
-- @
getSettingsGtkHintFontMetrics :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkHintFontMetrics obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-hint-font-metrics"

-- | Set the value of the “@gtk-hint-font-metrics@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkHintFontMetrics 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkHintFontMetrics :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkHintFontMetrics obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-hint-font-metrics" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-hint-font-metrics@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkHintFontMetrics :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkHintFontMetrics val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-hint-font-metrics" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkHintFontMetricsPropertyInfo
instance AttrInfo SettingsGtkHintFontMetricsPropertyInfo where
    type AttrAllowedOps SettingsGtkHintFontMetricsPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkHintFontMetricsPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkHintFontMetricsPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkHintFontMetricsPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkHintFontMetricsPropertyInfo = Bool
    type AttrGetType SettingsGtkHintFontMetricsPropertyInfo = Bool
    type AttrLabel SettingsGtkHintFontMetricsPropertyInfo = "gtk-hint-font-metrics"
    type AttrOrigin SettingsGtkHintFontMetricsPropertyInfo = Settings
    attrGet = getSettingsGtkHintFontMetrics
    attrSet = setSettingsGtkHintFontMetrics
    attrPut = setSettingsGtkHintFontMetrics
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkHintFontMetrics
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkHintFontMetrics"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkHintFontMetrics"
        })
#endif

-- VVV Prop "gtk-icon-theme-name"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-icon-theme-name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkIconThemeName
-- @
getSettingsGtkIconThemeName :: (MonadIO m, IsSettings o) => o -> m (Maybe T.Text)
getSettingsGtkIconThemeName obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "gtk-icon-theme-name"

-- | Set the value of the “@gtk-icon-theme-name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkIconThemeName 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkIconThemeName :: (MonadIO m, IsSettings o) => o -> T.Text -> m ()
setSettingsGtkIconThemeName obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "gtk-icon-theme-name" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-icon-theme-name@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkIconThemeName :: (IsSettings o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructSettingsGtkIconThemeName val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "gtk-icon-theme-name" (P.Just val)

-- | Set the value of the “@gtk-icon-theme-name@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #gtkIconThemeName
-- @
clearSettingsGtkIconThemeName :: (MonadIO m, IsSettings o) => o -> m ()
clearSettingsGtkIconThemeName obj = liftIO $ B.Properties.setObjectPropertyString obj "gtk-icon-theme-name" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data SettingsGtkIconThemeNamePropertyInfo
instance AttrInfo SettingsGtkIconThemeNamePropertyInfo where
    type AttrAllowedOps SettingsGtkIconThemeNamePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsGtkIconThemeNamePropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkIconThemeNamePropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint SettingsGtkIconThemeNamePropertyInfo = (~) T.Text
    type AttrTransferType SettingsGtkIconThemeNamePropertyInfo = T.Text
    type AttrGetType SettingsGtkIconThemeNamePropertyInfo = (Maybe T.Text)
    type AttrLabel SettingsGtkIconThemeNamePropertyInfo = "gtk-icon-theme-name"
    type AttrOrigin SettingsGtkIconThemeNamePropertyInfo = Settings
    attrGet = getSettingsGtkIconThemeName
    attrSet = setSettingsGtkIconThemeName
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkIconThemeName
    attrClear = clearSettingsGtkIconThemeName
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkIconThemeName"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkIconThemeName"
        })
#endif

-- VVV Prop "gtk-im-module"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-im-module@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkImModule
-- @
getSettingsGtkImModule :: (MonadIO m, IsSettings o) => o -> m (Maybe T.Text)
getSettingsGtkImModule obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "gtk-im-module"

-- | Set the value of the “@gtk-im-module@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkImModule 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkImModule :: (MonadIO m, IsSettings o) => o -> T.Text -> m ()
setSettingsGtkImModule obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "gtk-im-module" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-im-module@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkImModule :: (IsSettings o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructSettingsGtkImModule val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "gtk-im-module" (P.Just val)

-- | Set the value of the “@gtk-im-module@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #gtkImModule
-- @
clearSettingsGtkImModule :: (MonadIO m, IsSettings o) => o -> m ()
clearSettingsGtkImModule obj = liftIO $ B.Properties.setObjectPropertyString obj "gtk-im-module" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data SettingsGtkImModulePropertyInfo
instance AttrInfo SettingsGtkImModulePropertyInfo where
    type AttrAllowedOps SettingsGtkImModulePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsGtkImModulePropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkImModulePropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint SettingsGtkImModulePropertyInfo = (~) T.Text
    type AttrTransferType SettingsGtkImModulePropertyInfo = T.Text
    type AttrGetType SettingsGtkImModulePropertyInfo = (Maybe T.Text)
    type AttrLabel SettingsGtkImModulePropertyInfo = "gtk-im-module"
    type AttrOrigin SettingsGtkImModulePropertyInfo = Settings
    attrGet = getSettingsGtkImModule
    attrSet = setSettingsGtkImModule
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkImModule
    attrClear = clearSettingsGtkImModule
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkImModule"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkImModule"
        })
#endif

-- VVV Prop "gtk-interface-color-scheme"
   -- Type: TInterface (Name {namespace = "Gtk", name = "InterfaceColorScheme"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-interface-color-scheme@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkInterfaceColorScheme
-- @
getSettingsGtkInterfaceColorScheme :: (MonadIO m, IsSettings o) => o -> m Gtk.Enums.InterfaceColorScheme
getSettingsGtkInterfaceColorScheme obj = MIO.liftIO $ B.Properties.getObjectPropertyEnum obj "gtk-interface-color-scheme"

-- | Set the value of the “@gtk-interface-color-scheme@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkInterfaceColorScheme 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkInterfaceColorScheme :: (MonadIO m, IsSettings o) => o -> Gtk.Enums.InterfaceColorScheme -> m ()
setSettingsGtkInterfaceColorScheme obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyEnum obj "gtk-interface-color-scheme" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-interface-color-scheme@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkInterfaceColorScheme :: (IsSettings o, MIO.MonadIO m) => Gtk.Enums.InterfaceColorScheme -> m (GValueConstruct o)
constructSettingsGtkInterfaceColorScheme val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyEnum "gtk-interface-color-scheme" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkInterfaceColorSchemePropertyInfo
instance AttrInfo SettingsGtkInterfaceColorSchemePropertyInfo where
    type AttrAllowedOps SettingsGtkInterfaceColorSchemePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkInterfaceColorSchemePropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkInterfaceColorSchemePropertyInfo = (~) Gtk.Enums.InterfaceColorScheme
    type AttrTransferTypeConstraint SettingsGtkInterfaceColorSchemePropertyInfo = (~) Gtk.Enums.InterfaceColorScheme
    type AttrTransferType SettingsGtkInterfaceColorSchemePropertyInfo = Gtk.Enums.InterfaceColorScheme
    type AttrGetType SettingsGtkInterfaceColorSchemePropertyInfo = Gtk.Enums.InterfaceColorScheme
    type AttrLabel SettingsGtkInterfaceColorSchemePropertyInfo = "gtk-interface-color-scheme"
    type AttrOrigin SettingsGtkInterfaceColorSchemePropertyInfo = Settings
    attrGet = getSettingsGtkInterfaceColorScheme
    attrSet = setSettingsGtkInterfaceColorScheme
    attrPut = setSettingsGtkInterfaceColorScheme
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkInterfaceColorScheme
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkInterfaceColorScheme"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkInterfaceColorScheme"
        })
#endif

-- VVV Prop "gtk-interface-contrast"
   -- Type: TInterface (Name {namespace = "Gtk", name = "InterfaceContrast"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-interface-contrast@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkInterfaceContrast
-- @
getSettingsGtkInterfaceContrast :: (MonadIO m, IsSettings o) => o -> m Gtk.Enums.InterfaceContrast
getSettingsGtkInterfaceContrast obj = MIO.liftIO $ B.Properties.getObjectPropertyEnum obj "gtk-interface-contrast"

-- | Set the value of the “@gtk-interface-contrast@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkInterfaceContrast 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkInterfaceContrast :: (MonadIO m, IsSettings o) => o -> Gtk.Enums.InterfaceContrast -> m ()
setSettingsGtkInterfaceContrast obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyEnum obj "gtk-interface-contrast" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-interface-contrast@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkInterfaceContrast :: (IsSettings o, MIO.MonadIO m) => Gtk.Enums.InterfaceContrast -> m (GValueConstruct o)
constructSettingsGtkInterfaceContrast val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyEnum "gtk-interface-contrast" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkInterfaceContrastPropertyInfo
instance AttrInfo SettingsGtkInterfaceContrastPropertyInfo where
    type AttrAllowedOps SettingsGtkInterfaceContrastPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkInterfaceContrastPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkInterfaceContrastPropertyInfo = (~) Gtk.Enums.InterfaceContrast
    type AttrTransferTypeConstraint SettingsGtkInterfaceContrastPropertyInfo = (~) Gtk.Enums.InterfaceContrast
    type AttrTransferType SettingsGtkInterfaceContrastPropertyInfo = Gtk.Enums.InterfaceContrast
    type AttrGetType SettingsGtkInterfaceContrastPropertyInfo = Gtk.Enums.InterfaceContrast
    type AttrLabel SettingsGtkInterfaceContrastPropertyInfo = "gtk-interface-contrast"
    type AttrOrigin SettingsGtkInterfaceContrastPropertyInfo = Settings
    attrGet = getSettingsGtkInterfaceContrast
    attrSet = setSettingsGtkInterfaceContrast
    attrPut = setSettingsGtkInterfaceContrast
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkInterfaceContrast
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkInterfaceContrast"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkInterfaceContrast"
        })
#endif

-- VVV Prop "gtk-keynav-use-caret"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-keynav-use-caret@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkKeynavUseCaret
-- @
getSettingsGtkKeynavUseCaret :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkKeynavUseCaret obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-keynav-use-caret"

-- | Set the value of the “@gtk-keynav-use-caret@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkKeynavUseCaret 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkKeynavUseCaret :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkKeynavUseCaret obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-keynav-use-caret" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-keynav-use-caret@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkKeynavUseCaret :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkKeynavUseCaret val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-keynav-use-caret" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkKeynavUseCaretPropertyInfo
instance AttrInfo SettingsGtkKeynavUseCaretPropertyInfo where
    type AttrAllowedOps SettingsGtkKeynavUseCaretPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkKeynavUseCaretPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkKeynavUseCaretPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkKeynavUseCaretPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkKeynavUseCaretPropertyInfo = Bool
    type AttrGetType SettingsGtkKeynavUseCaretPropertyInfo = Bool
    type AttrLabel SettingsGtkKeynavUseCaretPropertyInfo = "gtk-keynav-use-caret"
    type AttrOrigin SettingsGtkKeynavUseCaretPropertyInfo = Settings
    attrGet = getSettingsGtkKeynavUseCaret
    attrSet = setSettingsGtkKeynavUseCaret
    attrPut = setSettingsGtkKeynavUseCaret
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkKeynavUseCaret
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkKeynavUseCaret"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkKeynavUseCaret"
        })
#endif

-- VVV Prop "gtk-label-select-on-focus"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-label-select-on-focus@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkLabelSelectOnFocus
-- @
getSettingsGtkLabelSelectOnFocus :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkLabelSelectOnFocus obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-label-select-on-focus"

-- | Set the value of the “@gtk-label-select-on-focus@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkLabelSelectOnFocus 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkLabelSelectOnFocus :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkLabelSelectOnFocus obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-label-select-on-focus" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-label-select-on-focus@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkLabelSelectOnFocus :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkLabelSelectOnFocus val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-label-select-on-focus" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkLabelSelectOnFocusPropertyInfo
instance AttrInfo SettingsGtkLabelSelectOnFocusPropertyInfo where
    type AttrAllowedOps SettingsGtkLabelSelectOnFocusPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkLabelSelectOnFocusPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkLabelSelectOnFocusPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkLabelSelectOnFocusPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkLabelSelectOnFocusPropertyInfo = Bool
    type AttrGetType SettingsGtkLabelSelectOnFocusPropertyInfo = Bool
    type AttrLabel SettingsGtkLabelSelectOnFocusPropertyInfo = "gtk-label-select-on-focus"
    type AttrOrigin SettingsGtkLabelSelectOnFocusPropertyInfo = Settings
    attrGet = getSettingsGtkLabelSelectOnFocus
    attrSet = setSettingsGtkLabelSelectOnFocus
    attrPut = setSettingsGtkLabelSelectOnFocus
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkLabelSelectOnFocus
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkLabelSelectOnFocus"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkLabelSelectOnFocus"
        })
#endif

-- VVV Prop "gtk-long-press-time"
   -- Type: TBasicType TUInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-long-press-time@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkLongPressTime
-- @
getSettingsGtkLongPressTime :: (MonadIO m, IsSettings o) => o -> m Word32
getSettingsGtkLongPressTime obj = MIO.liftIO $ B.Properties.getObjectPropertyUInt32 obj "gtk-long-press-time"

-- | Set the value of the “@gtk-long-press-time@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkLongPressTime 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkLongPressTime :: (MonadIO m, IsSettings o) => o -> Word32 -> m ()
setSettingsGtkLongPressTime obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyUInt32 obj "gtk-long-press-time" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-long-press-time@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkLongPressTime :: (IsSettings o, MIO.MonadIO m) => Word32 -> m (GValueConstruct o)
constructSettingsGtkLongPressTime val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyUInt32 "gtk-long-press-time" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkLongPressTimePropertyInfo
instance AttrInfo SettingsGtkLongPressTimePropertyInfo where
    type AttrAllowedOps SettingsGtkLongPressTimePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkLongPressTimePropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkLongPressTimePropertyInfo = (~) Word32
    type AttrTransferTypeConstraint SettingsGtkLongPressTimePropertyInfo = (~) Word32
    type AttrTransferType SettingsGtkLongPressTimePropertyInfo = Word32
    type AttrGetType SettingsGtkLongPressTimePropertyInfo = Word32
    type AttrLabel SettingsGtkLongPressTimePropertyInfo = "gtk-long-press-time"
    type AttrOrigin SettingsGtkLongPressTimePropertyInfo = Settings
    attrGet = getSettingsGtkLongPressTime
    attrSet = setSettingsGtkLongPressTime
    attrPut = setSettingsGtkLongPressTime
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkLongPressTime
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkLongPressTime"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkLongPressTime"
        })
#endif

-- VVV Prop "gtk-overlay-scrolling"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-overlay-scrolling@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkOverlayScrolling
-- @
getSettingsGtkOverlayScrolling :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkOverlayScrolling obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-overlay-scrolling"

-- | Set the value of the “@gtk-overlay-scrolling@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkOverlayScrolling 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkOverlayScrolling :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkOverlayScrolling obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-overlay-scrolling" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-overlay-scrolling@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkOverlayScrolling :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkOverlayScrolling val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-overlay-scrolling" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkOverlayScrollingPropertyInfo
instance AttrInfo SettingsGtkOverlayScrollingPropertyInfo where
    type AttrAllowedOps SettingsGtkOverlayScrollingPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkOverlayScrollingPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkOverlayScrollingPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkOverlayScrollingPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkOverlayScrollingPropertyInfo = Bool
    type AttrGetType SettingsGtkOverlayScrollingPropertyInfo = Bool
    type AttrLabel SettingsGtkOverlayScrollingPropertyInfo = "gtk-overlay-scrolling"
    type AttrOrigin SettingsGtkOverlayScrollingPropertyInfo = Settings
    attrGet = getSettingsGtkOverlayScrolling
    attrSet = setSettingsGtkOverlayScrolling
    attrPut = setSettingsGtkOverlayScrolling
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkOverlayScrolling
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkOverlayScrolling"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkOverlayScrolling"
        })
#endif

-- VVV Prop "gtk-primary-button-warps-slider"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-primary-button-warps-slider@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkPrimaryButtonWarpsSlider
-- @
getSettingsGtkPrimaryButtonWarpsSlider :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkPrimaryButtonWarpsSlider obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-primary-button-warps-slider"

-- | Set the value of the “@gtk-primary-button-warps-slider@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkPrimaryButtonWarpsSlider 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkPrimaryButtonWarpsSlider :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkPrimaryButtonWarpsSlider obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-primary-button-warps-slider" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-primary-button-warps-slider@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkPrimaryButtonWarpsSlider :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkPrimaryButtonWarpsSlider val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-primary-button-warps-slider" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkPrimaryButtonWarpsSliderPropertyInfo
instance AttrInfo SettingsGtkPrimaryButtonWarpsSliderPropertyInfo where
    type AttrAllowedOps SettingsGtkPrimaryButtonWarpsSliderPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkPrimaryButtonWarpsSliderPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkPrimaryButtonWarpsSliderPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkPrimaryButtonWarpsSliderPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkPrimaryButtonWarpsSliderPropertyInfo = Bool
    type AttrGetType SettingsGtkPrimaryButtonWarpsSliderPropertyInfo = Bool
    type AttrLabel SettingsGtkPrimaryButtonWarpsSliderPropertyInfo = "gtk-primary-button-warps-slider"
    type AttrOrigin SettingsGtkPrimaryButtonWarpsSliderPropertyInfo = Settings
    attrGet = getSettingsGtkPrimaryButtonWarpsSlider
    attrSet = setSettingsGtkPrimaryButtonWarpsSlider
    attrPut = setSettingsGtkPrimaryButtonWarpsSlider
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkPrimaryButtonWarpsSlider
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkPrimaryButtonWarpsSlider"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkPrimaryButtonWarpsSlider"
        })
#endif

-- VVV Prop "gtk-print-backends"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-print-backends@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkPrintBackends
-- @
getSettingsGtkPrintBackends :: (MonadIO m, IsSettings o) => o -> m (Maybe T.Text)
getSettingsGtkPrintBackends obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "gtk-print-backends"

-- | Set the value of the “@gtk-print-backends@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkPrintBackends 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkPrintBackends :: (MonadIO m, IsSettings o) => o -> T.Text -> m ()
setSettingsGtkPrintBackends obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "gtk-print-backends" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-print-backends@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkPrintBackends :: (IsSettings o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructSettingsGtkPrintBackends val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "gtk-print-backends" (P.Just val)

-- | Set the value of the “@gtk-print-backends@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #gtkPrintBackends
-- @
clearSettingsGtkPrintBackends :: (MonadIO m, IsSettings o) => o -> m ()
clearSettingsGtkPrintBackends obj = liftIO $ B.Properties.setObjectPropertyString obj "gtk-print-backends" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data SettingsGtkPrintBackendsPropertyInfo
instance AttrInfo SettingsGtkPrintBackendsPropertyInfo where
    type AttrAllowedOps SettingsGtkPrintBackendsPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsGtkPrintBackendsPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkPrintBackendsPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint SettingsGtkPrintBackendsPropertyInfo = (~) T.Text
    type AttrTransferType SettingsGtkPrintBackendsPropertyInfo = T.Text
    type AttrGetType SettingsGtkPrintBackendsPropertyInfo = (Maybe T.Text)
    type AttrLabel SettingsGtkPrintBackendsPropertyInfo = "gtk-print-backends"
    type AttrOrigin SettingsGtkPrintBackendsPropertyInfo = Settings
    attrGet = getSettingsGtkPrintBackends
    attrSet = setSettingsGtkPrintBackends
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkPrintBackends
    attrClear = clearSettingsGtkPrintBackends
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkPrintBackends"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkPrintBackends"
        })
#endif

-- VVV Prop "gtk-print-preview-command"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-print-preview-command@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkPrintPreviewCommand
-- @
getSettingsGtkPrintPreviewCommand :: (MonadIO m, IsSettings o) => o -> m (Maybe T.Text)
getSettingsGtkPrintPreviewCommand obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "gtk-print-preview-command"

-- | Set the value of the “@gtk-print-preview-command@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkPrintPreviewCommand 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkPrintPreviewCommand :: (MonadIO m, IsSettings o) => o -> T.Text -> m ()
setSettingsGtkPrintPreviewCommand obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "gtk-print-preview-command" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-print-preview-command@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkPrintPreviewCommand :: (IsSettings o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructSettingsGtkPrintPreviewCommand val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "gtk-print-preview-command" (P.Just val)

-- | Set the value of the “@gtk-print-preview-command@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #gtkPrintPreviewCommand
-- @
clearSettingsGtkPrintPreviewCommand :: (MonadIO m, IsSettings o) => o -> m ()
clearSettingsGtkPrintPreviewCommand obj = liftIO $ B.Properties.setObjectPropertyString obj "gtk-print-preview-command" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data SettingsGtkPrintPreviewCommandPropertyInfo
instance AttrInfo SettingsGtkPrintPreviewCommandPropertyInfo where
    type AttrAllowedOps SettingsGtkPrintPreviewCommandPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsGtkPrintPreviewCommandPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkPrintPreviewCommandPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint SettingsGtkPrintPreviewCommandPropertyInfo = (~) T.Text
    type AttrTransferType SettingsGtkPrintPreviewCommandPropertyInfo = T.Text
    type AttrGetType SettingsGtkPrintPreviewCommandPropertyInfo = (Maybe T.Text)
    type AttrLabel SettingsGtkPrintPreviewCommandPropertyInfo = "gtk-print-preview-command"
    type AttrOrigin SettingsGtkPrintPreviewCommandPropertyInfo = Settings
    attrGet = getSettingsGtkPrintPreviewCommand
    attrSet = setSettingsGtkPrintPreviewCommand
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkPrintPreviewCommand
    attrClear = clearSettingsGtkPrintPreviewCommand
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkPrintPreviewCommand"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkPrintPreviewCommand"
        })
#endif

-- VVV Prop "gtk-recent-files-enabled"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-recent-files-enabled@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkRecentFilesEnabled
-- @
getSettingsGtkRecentFilesEnabled :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkRecentFilesEnabled obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-recent-files-enabled"

-- | Set the value of the “@gtk-recent-files-enabled@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkRecentFilesEnabled 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkRecentFilesEnabled :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkRecentFilesEnabled obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-recent-files-enabled" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-recent-files-enabled@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkRecentFilesEnabled :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkRecentFilesEnabled val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-recent-files-enabled" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkRecentFilesEnabledPropertyInfo
instance AttrInfo SettingsGtkRecentFilesEnabledPropertyInfo where
    type AttrAllowedOps SettingsGtkRecentFilesEnabledPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkRecentFilesEnabledPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkRecentFilesEnabledPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkRecentFilesEnabledPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkRecentFilesEnabledPropertyInfo = Bool
    type AttrGetType SettingsGtkRecentFilesEnabledPropertyInfo = Bool
    type AttrLabel SettingsGtkRecentFilesEnabledPropertyInfo = "gtk-recent-files-enabled"
    type AttrOrigin SettingsGtkRecentFilesEnabledPropertyInfo = Settings
    attrGet = getSettingsGtkRecentFilesEnabled
    attrSet = setSettingsGtkRecentFilesEnabled
    attrPut = setSettingsGtkRecentFilesEnabled
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkRecentFilesEnabled
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkRecentFilesEnabled"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkRecentFilesEnabled"
        })
#endif

-- VVV Prop "gtk-recent-files-max-age"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-recent-files-max-age@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkRecentFilesMaxAge
-- @
getSettingsGtkRecentFilesMaxAge :: (MonadIO m, IsSettings o) => o -> m Int32
getSettingsGtkRecentFilesMaxAge obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "gtk-recent-files-max-age"

-- | Set the value of the “@gtk-recent-files-max-age@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkRecentFilesMaxAge 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkRecentFilesMaxAge :: (MonadIO m, IsSettings o) => o -> Int32 -> m ()
setSettingsGtkRecentFilesMaxAge obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "gtk-recent-files-max-age" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-recent-files-max-age@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkRecentFilesMaxAge :: (IsSettings o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructSettingsGtkRecentFilesMaxAge val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "gtk-recent-files-max-age" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkRecentFilesMaxAgePropertyInfo
instance AttrInfo SettingsGtkRecentFilesMaxAgePropertyInfo where
    type AttrAllowedOps SettingsGtkRecentFilesMaxAgePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkRecentFilesMaxAgePropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkRecentFilesMaxAgePropertyInfo = (~) Int32
    type AttrTransferTypeConstraint SettingsGtkRecentFilesMaxAgePropertyInfo = (~) Int32
    type AttrTransferType SettingsGtkRecentFilesMaxAgePropertyInfo = Int32
    type AttrGetType SettingsGtkRecentFilesMaxAgePropertyInfo = Int32
    type AttrLabel SettingsGtkRecentFilesMaxAgePropertyInfo = "gtk-recent-files-max-age"
    type AttrOrigin SettingsGtkRecentFilesMaxAgePropertyInfo = Settings
    attrGet = getSettingsGtkRecentFilesMaxAge
    attrSet = setSettingsGtkRecentFilesMaxAge
    attrPut = setSettingsGtkRecentFilesMaxAge
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkRecentFilesMaxAge
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkRecentFilesMaxAge"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkRecentFilesMaxAge"
        })
#endif

-- VVV Prop "gtk-shell-shows-app-menu"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-shell-shows-app-menu@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkShellShowsAppMenu
-- @
getSettingsGtkShellShowsAppMenu :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkShellShowsAppMenu obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-shell-shows-app-menu"

-- | Set the value of the “@gtk-shell-shows-app-menu@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkShellShowsAppMenu 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkShellShowsAppMenu :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkShellShowsAppMenu obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-shell-shows-app-menu" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-shell-shows-app-menu@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkShellShowsAppMenu :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkShellShowsAppMenu val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-shell-shows-app-menu" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkShellShowsAppMenuPropertyInfo
instance AttrInfo SettingsGtkShellShowsAppMenuPropertyInfo where
    type AttrAllowedOps SettingsGtkShellShowsAppMenuPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkShellShowsAppMenuPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkShellShowsAppMenuPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkShellShowsAppMenuPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkShellShowsAppMenuPropertyInfo = Bool
    type AttrGetType SettingsGtkShellShowsAppMenuPropertyInfo = Bool
    type AttrLabel SettingsGtkShellShowsAppMenuPropertyInfo = "gtk-shell-shows-app-menu"
    type AttrOrigin SettingsGtkShellShowsAppMenuPropertyInfo = Settings
    attrGet = getSettingsGtkShellShowsAppMenu
    attrSet = setSettingsGtkShellShowsAppMenu
    attrPut = setSettingsGtkShellShowsAppMenu
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkShellShowsAppMenu
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkShellShowsAppMenu"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkShellShowsAppMenu"
        })
#endif

-- VVV Prop "gtk-shell-shows-desktop"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-shell-shows-desktop@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkShellShowsDesktop
-- @
getSettingsGtkShellShowsDesktop :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkShellShowsDesktop obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-shell-shows-desktop"

-- | Set the value of the “@gtk-shell-shows-desktop@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkShellShowsDesktop 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkShellShowsDesktop :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkShellShowsDesktop obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-shell-shows-desktop" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-shell-shows-desktop@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkShellShowsDesktop :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkShellShowsDesktop val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-shell-shows-desktop" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkShellShowsDesktopPropertyInfo
instance AttrInfo SettingsGtkShellShowsDesktopPropertyInfo where
    type AttrAllowedOps SettingsGtkShellShowsDesktopPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkShellShowsDesktopPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkShellShowsDesktopPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkShellShowsDesktopPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkShellShowsDesktopPropertyInfo = Bool
    type AttrGetType SettingsGtkShellShowsDesktopPropertyInfo = Bool
    type AttrLabel SettingsGtkShellShowsDesktopPropertyInfo = "gtk-shell-shows-desktop"
    type AttrOrigin SettingsGtkShellShowsDesktopPropertyInfo = Settings
    attrGet = getSettingsGtkShellShowsDesktop
    attrSet = setSettingsGtkShellShowsDesktop
    attrPut = setSettingsGtkShellShowsDesktop
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkShellShowsDesktop
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkShellShowsDesktop"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkShellShowsDesktop"
        })
#endif

-- VVV Prop "gtk-shell-shows-menubar"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-shell-shows-menubar@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkShellShowsMenubar
-- @
getSettingsGtkShellShowsMenubar :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkShellShowsMenubar obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-shell-shows-menubar"

-- | Set the value of the “@gtk-shell-shows-menubar@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkShellShowsMenubar 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkShellShowsMenubar :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkShellShowsMenubar obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-shell-shows-menubar" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-shell-shows-menubar@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkShellShowsMenubar :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkShellShowsMenubar val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-shell-shows-menubar" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkShellShowsMenubarPropertyInfo
instance AttrInfo SettingsGtkShellShowsMenubarPropertyInfo where
    type AttrAllowedOps SettingsGtkShellShowsMenubarPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkShellShowsMenubarPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkShellShowsMenubarPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkShellShowsMenubarPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkShellShowsMenubarPropertyInfo = Bool
    type AttrGetType SettingsGtkShellShowsMenubarPropertyInfo = Bool
    type AttrLabel SettingsGtkShellShowsMenubarPropertyInfo = "gtk-shell-shows-menubar"
    type AttrOrigin SettingsGtkShellShowsMenubarPropertyInfo = Settings
    attrGet = getSettingsGtkShellShowsMenubar
    attrSet = setSettingsGtkShellShowsMenubar
    attrPut = setSettingsGtkShellShowsMenubar
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkShellShowsMenubar
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkShellShowsMenubar"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkShellShowsMenubar"
        })
#endif

-- VVV Prop "gtk-show-status-shapes"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-show-status-shapes@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkShowStatusShapes
-- @
getSettingsGtkShowStatusShapes :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkShowStatusShapes obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-show-status-shapes"

-- | Set the value of the “@gtk-show-status-shapes@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkShowStatusShapes 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkShowStatusShapes :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkShowStatusShapes obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-show-status-shapes" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-show-status-shapes@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkShowStatusShapes :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkShowStatusShapes val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-show-status-shapes" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkShowStatusShapesPropertyInfo
instance AttrInfo SettingsGtkShowStatusShapesPropertyInfo where
    type AttrAllowedOps SettingsGtkShowStatusShapesPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkShowStatusShapesPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkShowStatusShapesPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkShowStatusShapesPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkShowStatusShapesPropertyInfo = Bool
    type AttrGetType SettingsGtkShowStatusShapesPropertyInfo = Bool
    type AttrLabel SettingsGtkShowStatusShapesPropertyInfo = "gtk-show-status-shapes"
    type AttrOrigin SettingsGtkShowStatusShapesPropertyInfo = Settings
    attrGet = getSettingsGtkShowStatusShapes
    attrSet = setSettingsGtkShowStatusShapes
    attrPut = setSettingsGtkShowStatusShapes
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkShowStatusShapes
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkShowStatusShapes"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkShowStatusShapes"
        })
#endif

-- VVV Prop "gtk-sound-theme-name"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-sound-theme-name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkSoundThemeName
-- @
getSettingsGtkSoundThemeName :: (MonadIO m, IsSettings o) => o -> m (Maybe T.Text)
getSettingsGtkSoundThemeName obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "gtk-sound-theme-name"

-- | Set the value of the “@gtk-sound-theme-name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkSoundThemeName 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkSoundThemeName :: (MonadIO m, IsSettings o) => o -> T.Text -> m ()
setSettingsGtkSoundThemeName obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "gtk-sound-theme-name" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-sound-theme-name@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkSoundThemeName :: (IsSettings o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructSettingsGtkSoundThemeName val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "gtk-sound-theme-name" (P.Just val)

-- | Set the value of the “@gtk-sound-theme-name@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #gtkSoundThemeName
-- @
clearSettingsGtkSoundThemeName :: (MonadIO m, IsSettings o) => o -> m ()
clearSettingsGtkSoundThemeName obj = liftIO $ B.Properties.setObjectPropertyString obj "gtk-sound-theme-name" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data SettingsGtkSoundThemeNamePropertyInfo
instance AttrInfo SettingsGtkSoundThemeNamePropertyInfo where
    type AttrAllowedOps SettingsGtkSoundThemeNamePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsGtkSoundThemeNamePropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkSoundThemeNamePropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint SettingsGtkSoundThemeNamePropertyInfo = (~) T.Text
    type AttrTransferType SettingsGtkSoundThemeNamePropertyInfo = T.Text
    type AttrGetType SettingsGtkSoundThemeNamePropertyInfo = (Maybe T.Text)
    type AttrLabel SettingsGtkSoundThemeNamePropertyInfo = "gtk-sound-theme-name"
    type AttrOrigin SettingsGtkSoundThemeNamePropertyInfo = Settings
    attrGet = getSettingsGtkSoundThemeName
    attrSet = setSettingsGtkSoundThemeName
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkSoundThemeName
    attrClear = clearSettingsGtkSoundThemeName
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkSoundThemeName"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkSoundThemeName"
        })
#endif

-- VVV Prop "gtk-split-cursor"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-split-cursor@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkSplitCursor
-- @
getSettingsGtkSplitCursor :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsGtkSplitCursor obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "gtk-split-cursor"

-- | Set the value of the “@gtk-split-cursor@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkSplitCursor 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkSplitCursor :: (MonadIO m, IsSettings o) => o -> Bool -> m ()
setSettingsGtkSplitCursor obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "gtk-split-cursor" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-split-cursor@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkSplitCursor :: (IsSettings o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructSettingsGtkSplitCursor val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "gtk-split-cursor" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkSplitCursorPropertyInfo
instance AttrInfo SettingsGtkSplitCursorPropertyInfo where
    type AttrAllowedOps SettingsGtkSplitCursorPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkSplitCursorPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkSplitCursorPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint SettingsGtkSplitCursorPropertyInfo = (~) Bool
    type AttrTransferType SettingsGtkSplitCursorPropertyInfo = Bool
    type AttrGetType SettingsGtkSplitCursorPropertyInfo = Bool
    type AttrLabel SettingsGtkSplitCursorPropertyInfo = "gtk-split-cursor"
    type AttrOrigin SettingsGtkSplitCursorPropertyInfo = Settings
    attrGet = getSettingsGtkSplitCursor
    attrSet = setSettingsGtkSplitCursor
    attrPut = setSettingsGtkSplitCursor
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkSplitCursor
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkSplitCursor"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkSplitCursor"
        })
#endif

-- VVV Prop "gtk-theme-name"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-theme-name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkThemeName
-- @
getSettingsGtkThemeName :: (MonadIO m, IsSettings o) => o -> m (Maybe T.Text)
getSettingsGtkThemeName obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "gtk-theme-name"

-- | Set the value of the “@gtk-theme-name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkThemeName 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkThemeName :: (MonadIO m, IsSettings o) => o -> T.Text -> m ()
setSettingsGtkThemeName obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "gtk-theme-name" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-theme-name@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkThemeName :: (IsSettings o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructSettingsGtkThemeName val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "gtk-theme-name" (P.Just val)

-- | Set the value of the “@gtk-theme-name@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #gtkThemeName
-- @
clearSettingsGtkThemeName :: (MonadIO m, IsSettings o) => o -> m ()
clearSettingsGtkThemeName obj = liftIO $ B.Properties.setObjectPropertyString obj "gtk-theme-name" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data SettingsGtkThemeNamePropertyInfo
instance AttrInfo SettingsGtkThemeNamePropertyInfo where
    type AttrAllowedOps SettingsGtkThemeNamePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsGtkThemeNamePropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkThemeNamePropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint SettingsGtkThemeNamePropertyInfo = (~) T.Text
    type AttrTransferType SettingsGtkThemeNamePropertyInfo = T.Text
    type AttrGetType SettingsGtkThemeNamePropertyInfo = (Maybe T.Text)
    type AttrLabel SettingsGtkThemeNamePropertyInfo = "gtk-theme-name"
    type AttrOrigin SettingsGtkThemeNamePropertyInfo = Settings
    attrGet = getSettingsGtkThemeName
    attrSet = setSettingsGtkThemeName
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkThemeName
    attrClear = clearSettingsGtkThemeName
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkThemeName"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkThemeName"
        })
#endif

-- VVV Prop "gtk-titlebar-double-click"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-titlebar-double-click@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkTitlebarDoubleClick
-- @
getSettingsGtkTitlebarDoubleClick :: (MonadIO m, IsSettings o) => o -> m (Maybe T.Text)
getSettingsGtkTitlebarDoubleClick obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "gtk-titlebar-double-click"

-- | Set the value of the “@gtk-titlebar-double-click@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkTitlebarDoubleClick 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkTitlebarDoubleClick :: (MonadIO m, IsSettings o) => o -> T.Text -> m ()
setSettingsGtkTitlebarDoubleClick obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "gtk-titlebar-double-click" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-titlebar-double-click@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkTitlebarDoubleClick :: (IsSettings o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructSettingsGtkTitlebarDoubleClick val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "gtk-titlebar-double-click" (P.Just val)

-- | Set the value of the “@gtk-titlebar-double-click@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #gtkTitlebarDoubleClick
-- @
clearSettingsGtkTitlebarDoubleClick :: (MonadIO m, IsSettings o) => o -> m ()
clearSettingsGtkTitlebarDoubleClick obj = liftIO $ B.Properties.setObjectPropertyString obj "gtk-titlebar-double-click" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data SettingsGtkTitlebarDoubleClickPropertyInfo
instance AttrInfo SettingsGtkTitlebarDoubleClickPropertyInfo where
    type AttrAllowedOps SettingsGtkTitlebarDoubleClickPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsGtkTitlebarDoubleClickPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkTitlebarDoubleClickPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint SettingsGtkTitlebarDoubleClickPropertyInfo = (~) T.Text
    type AttrTransferType SettingsGtkTitlebarDoubleClickPropertyInfo = T.Text
    type AttrGetType SettingsGtkTitlebarDoubleClickPropertyInfo = (Maybe T.Text)
    type AttrLabel SettingsGtkTitlebarDoubleClickPropertyInfo = "gtk-titlebar-double-click"
    type AttrOrigin SettingsGtkTitlebarDoubleClickPropertyInfo = Settings
    attrGet = getSettingsGtkTitlebarDoubleClick
    attrSet = setSettingsGtkTitlebarDoubleClick
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkTitlebarDoubleClick
    attrClear = clearSettingsGtkTitlebarDoubleClick
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkTitlebarDoubleClick"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkTitlebarDoubleClick"
        })
#endif

-- VVV Prop "gtk-titlebar-middle-click"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-titlebar-middle-click@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkTitlebarMiddleClick
-- @
getSettingsGtkTitlebarMiddleClick :: (MonadIO m, IsSettings o) => o -> m (Maybe T.Text)
getSettingsGtkTitlebarMiddleClick obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "gtk-titlebar-middle-click"

-- | Set the value of the “@gtk-titlebar-middle-click@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkTitlebarMiddleClick 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkTitlebarMiddleClick :: (MonadIO m, IsSettings o) => o -> T.Text -> m ()
setSettingsGtkTitlebarMiddleClick obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "gtk-titlebar-middle-click" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-titlebar-middle-click@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkTitlebarMiddleClick :: (IsSettings o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructSettingsGtkTitlebarMiddleClick val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "gtk-titlebar-middle-click" (P.Just val)

-- | Set the value of the “@gtk-titlebar-middle-click@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #gtkTitlebarMiddleClick
-- @
clearSettingsGtkTitlebarMiddleClick :: (MonadIO m, IsSettings o) => o -> m ()
clearSettingsGtkTitlebarMiddleClick obj = liftIO $ B.Properties.setObjectPropertyString obj "gtk-titlebar-middle-click" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data SettingsGtkTitlebarMiddleClickPropertyInfo
instance AttrInfo SettingsGtkTitlebarMiddleClickPropertyInfo where
    type AttrAllowedOps SettingsGtkTitlebarMiddleClickPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsGtkTitlebarMiddleClickPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkTitlebarMiddleClickPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint SettingsGtkTitlebarMiddleClickPropertyInfo = (~) T.Text
    type AttrTransferType SettingsGtkTitlebarMiddleClickPropertyInfo = T.Text
    type AttrGetType SettingsGtkTitlebarMiddleClickPropertyInfo = (Maybe T.Text)
    type AttrLabel SettingsGtkTitlebarMiddleClickPropertyInfo = "gtk-titlebar-middle-click"
    type AttrOrigin SettingsGtkTitlebarMiddleClickPropertyInfo = Settings
    attrGet = getSettingsGtkTitlebarMiddleClick
    attrSet = setSettingsGtkTitlebarMiddleClick
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkTitlebarMiddleClick
    attrClear = clearSettingsGtkTitlebarMiddleClick
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkTitlebarMiddleClick"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkTitlebarMiddleClick"
        })
#endif

-- VVV Prop "gtk-titlebar-right-click"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-titlebar-right-click@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkTitlebarRightClick
-- @
getSettingsGtkTitlebarRightClick :: (MonadIO m, IsSettings o) => o -> m (Maybe T.Text)
getSettingsGtkTitlebarRightClick obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "gtk-titlebar-right-click"

-- | Set the value of the “@gtk-titlebar-right-click@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkTitlebarRightClick 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkTitlebarRightClick :: (MonadIO m, IsSettings o) => o -> T.Text -> m ()
setSettingsGtkTitlebarRightClick obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "gtk-titlebar-right-click" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-titlebar-right-click@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkTitlebarRightClick :: (IsSettings o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructSettingsGtkTitlebarRightClick val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "gtk-titlebar-right-click" (P.Just val)

-- | Set the value of the “@gtk-titlebar-right-click@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #gtkTitlebarRightClick
-- @
clearSettingsGtkTitlebarRightClick :: (MonadIO m, IsSettings o) => o -> m ()
clearSettingsGtkTitlebarRightClick obj = liftIO $ B.Properties.setObjectPropertyString obj "gtk-titlebar-right-click" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data SettingsGtkTitlebarRightClickPropertyInfo
instance AttrInfo SettingsGtkTitlebarRightClickPropertyInfo where
    type AttrAllowedOps SettingsGtkTitlebarRightClickPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsGtkTitlebarRightClickPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkTitlebarRightClickPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint SettingsGtkTitlebarRightClickPropertyInfo = (~) T.Text
    type AttrTransferType SettingsGtkTitlebarRightClickPropertyInfo = T.Text
    type AttrGetType SettingsGtkTitlebarRightClickPropertyInfo = (Maybe T.Text)
    type AttrLabel SettingsGtkTitlebarRightClickPropertyInfo = "gtk-titlebar-right-click"
    type AttrOrigin SettingsGtkTitlebarRightClickPropertyInfo = Settings
    attrGet = getSettingsGtkTitlebarRightClick
    attrSet = setSettingsGtkTitlebarRightClick
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkTitlebarRightClick
    attrClear = clearSettingsGtkTitlebarRightClick
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkTitlebarRightClick"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkTitlebarRightClick"
        })
#endif

-- VVV Prop "gtk-xft-antialias"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-xft-antialias@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkXftAntialias
-- @
getSettingsGtkXftAntialias :: (MonadIO m, IsSettings o) => o -> m Int32
getSettingsGtkXftAntialias obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "gtk-xft-antialias"

-- | Set the value of the “@gtk-xft-antialias@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkXftAntialias 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkXftAntialias :: (MonadIO m, IsSettings o) => o -> Int32 -> m ()
setSettingsGtkXftAntialias obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "gtk-xft-antialias" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-xft-antialias@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkXftAntialias :: (IsSettings o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructSettingsGtkXftAntialias val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "gtk-xft-antialias" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkXftAntialiasPropertyInfo
instance AttrInfo SettingsGtkXftAntialiasPropertyInfo where
    type AttrAllowedOps SettingsGtkXftAntialiasPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkXftAntialiasPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkXftAntialiasPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint SettingsGtkXftAntialiasPropertyInfo = (~) Int32
    type AttrTransferType SettingsGtkXftAntialiasPropertyInfo = Int32
    type AttrGetType SettingsGtkXftAntialiasPropertyInfo = Int32
    type AttrLabel SettingsGtkXftAntialiasPropertyInfo = "gtk-xft-antialias"
    type AttrOrigin SettingsGtkXftAntialiasPropertyInfo = Settings
    attrGet = getSettingsGtkXftAntialias
    attrSet = setSettingsGtkXftAntialias
    attrPut = setSettingsGtkXftAntialias
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkXftAntialias
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkXftAntialias"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkXftAntialias"
        })
#endif

-- VVV Prop "gtk-xft-dpi"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-xft-dpi@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkXftDpi
-- @
getSettingsGtkXftDpi :: (MonadIO m, IsSettings o) => o -> m Int32
getSettingsGtkXftDpi obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "gtk-xft-dpi"

-- | Set the value of the “@gtk-xft-dpi@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkXftDpi 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkXftDpi :: (MonadIO m, IsSettings o) => o -> Int32 -> m ()
setSettingsGtkXftDpi obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "gtk-xft-dpi" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-xft-dpi@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkXftDpi :: (IsSettings o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructSettingsGtkXftDpi val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "gtk-xft-dpi" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkXftDpiPropertyInfo
instance AttrInfo SettingsGtkXftDpiPropertyInfo where
    type AttrAllowedOps SettingsGtkXftDpiPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkXftDpiPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkXftDpiPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint SettingsGtkXftDpiPropertyInfo = (~) Int32
    type AttrTransferType SettingsGtkXftDpiPropertyInfo = Int32
    type AttrGetType SettingsGtkXftDpiPropertyInfo = Int32
    type AttrLabel SettingsGtkXftDpiPropertyInfo = "gtk-xft-dpi"
    type AttrOrigin SettingsGtkXftDpiPropertyInfo = Settings
    attrGet = getSettingsGtkXftDpi
    attrSet = setSettingsGtkXftDpi
    attrPut = setSettingsGtkXftDpi
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkXftDpi
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkXftDpi"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkXftDpi"
        })
#endif

-- VVV Prop "gtk-xft-hinting"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-xft-hinting@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkXftHinting
-- @
getSettingsGtkXftHinting :: (MonadIO m, IsSettings o) => o -> m Int32
getSettingsGtkXftHinting obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "gtk-xft-hinting"

-- | Set the value of the “@gtk-xft-hinting@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkXftHinting 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkXftHinting :: (MonadIO m, IsSettings o) => o -> Int32 -> m ()
setSettingsGtkXftHinting obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "gtk-xft-hinting" val

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-xft-hinting@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkXftHinting :: (IsSettings o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructSettingsGtkXftHinting val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "gtk-xft-hinting" val

#if defined(ENABLE_OVERLOADING)
data SettingsGtkXftHintingPropertyInfo
instance AttrInfo SettingsGtkXftHintingPropertyInfo where
    type AttrAllowedOps SettingsGtkXftHintingPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint SettingsGtkXftHintingPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkXftHintingPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint SettingsGtkXftHintingPropertyInfo = (~) Int32
    type AttrTransferType SettingsGtkXftHintingPropertyInfo = Int32
    type AttrGetType SettingsGtkXftHintingPropertyInfo = Int32
    type AttrLabel SettingsGtkXftHintingPropertyInfo = "gtk-xft-hinting"
    type AttrOrigin SettingsGtkXftHintingPropertyInfo = Settings
    attrGet = getSettingsGtkXftHinting
    attrSet = setSettingsGtkXftHinting
    attrPut = setSettingsGtkXftHinting
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkXftHinting
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkXftHinting"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkXftHinting"
        })
#endif

-- VVV Prop "gtk-xft-hintstyle"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-xft-hintstyle@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkXftHintstyle
-- @
getSettingsGtkXftHintstyle :: (MonadIO m, IsSettings o) => o -> m (Maybe T.Text)
getSettingsGtkXftHintstyle obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "gtk-xft-hintstyle"

-- | Set the value of the “@gtk-xft-hintstyle@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkXftHintstyle 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkXftHintstyle :: (MonadIO m, IsSettings o) => o -> T.Text -> m ()
setSettingsGtkXftHintstyle obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "gtk-xft-hintstyle" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-xft-hintstyle@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkXftHintstyle :: (IsSettings o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructSettingsGtkXftHintstyle val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "gtk-xft-hintstyle" (P.Just val)

-- | Set the value of the “@gtk-xft-hintstyle@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #gtkXftHintstyle
-- @
clearSettingsGtkXftHintstyle :: (MonadIO m, IsSettings o) => o -> m ()
clearSettingsGtkXftHintstyle obj = liftIO $ B.Properties.setObjectPropertyString obj "gtk-xft-hintstyle" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data SettingsGtkXftHintstylePropertyInfo
instance AttrInfo SettingsGtkXftHintstylePropertyInfo where
    type AttrAllowedOps SettingsGtkXftHintstylePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsGtkXftHintstylePropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkXftHintstylePropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint SettingsGtkXftHintstylePropertyInfo = (~) T.Text
    type AttrTransferType SettingsGtkXftHintstylePropertyInfo = T.Text
    type AttrGetType SettingsGtkXftHintstylePropertyInfo = (Maybe T.Text)
    type AttrLabel SettingsGtkXftHintstylePropertyInfo = "gtk-xft-hintstyle"
    type AttrOrigin SettingsGtkXftHintstylePropertyInfo = Settings
    attrGet = getSettingsGtkXftHintstyle
    attrSet = setSettingsGtkXftHintstyle
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkXftHintstyle
    attrClear = clearSettingsGtkXftHintstyle
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkXftHintstyle"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkXftHintstyle"
        })
#endif

-- VVV Prop "gtk-xft-rgba"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@gtk-xft-rgba@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #gtkXftRgba
-- @
getSettingsGtkXftRgba :: (MonadIO m, IsSettings o) => o -> m (Maybe T.Text)
getSettingsGtkXftRgba obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "gtk-xft-rgba"

-- | Set the value of the “@gtk-xft-rgba@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' settings [ #gtkXftRgba 'Data.GI.Base.Attributes.:=' value ]
-- @
setSettingsGtkXftRgba :: (MonadIO m, IsSettings o) => o -> T.Text -> m ()
setSettingsGtkXftRgba obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "gtk-xft-rgba" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@gtk-xft-rgba@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsGtkXftRgba :: (IsSettings o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructSettingsGtkXftRgba val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "gtk-xft-rgba" (P.Just val)

-- | Set the value of the “@gtk-xft-rgba@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #gtkXftRgba
-- @
clearSettingsGtkXftRgba :: (MonadIO m, IsSettings o) => o -> m ()
clearSettingsGtkXftRgba obj = liftIO $ B.Properties.setObjectPropertyString obj "gtk-xft-rgba" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data SettingsGtkXftRgbaPropertyInfo
instance AttrInfo SettingsGtkXftRgbaPropertyInfo where
    type AttrAllowedOps SettingsGtkXftRgbaPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsGtkXftRgbaPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsGtkXftRgbaPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint SettingsGtkXftRgbaPropertyInfo = (~) T.Text
    type AttrTransferType SettingsGtkXftRgbaPropertyInfo = T.Text
    type AttrGetType SettingsGtkXftRgbaPropertyInfo = (Maybe T.Text)
    type AttrLabel SettingsGtkXftRgbaPropertyInfo = "gtk-xft-rgba"
    type AttrOrigin SettingsGtkXftRgbaPropertyInfo = Settings
    attrGet = getSettingsGtkXftRgba
    attrSet = setSettingsGtkXftRgba
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsGtkXftRgba
    attrClear = clearSettingsGtkXftRgba
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.gtkXftRgba"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#g:attr:gtkXftRgba"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Settings
type instance O.AttributeList Settings = SettingsAttributeList
type SettingsAttributeList = ('[ '("gtkAlternativeButtonOrder", SettingsGtkAlternativeButtonOrderPropertyInfo), '("gtkAlternativeSortArrows", SettingsGtkAlternativeSortArrowsPropertyInfo), '("gtkApplicationPreferDarkTheme", SettingsGtkApplicationPreferDarkThemePropertyInfo), '("gtkCursorAspectRatio", SettingsGtkCursorAspectRatioPropertyInfo), '("gtkCursorBlink", SettingsGtkCursorBlinkPropertyInfo), '("gtkCursorBlinkTime", SettingsGtkCursorBlinkTimePropertyInfo), '("gtkCursorBlinkTimeout", SettingsGtkCursorBlinkTimeoutPropertyInfo), '("gtkCursorThemeName", SettingsGtkCursorThemeNamePropertyInfo), '("gtkCursorThemeSize", SettingsGtkCursorThemeSizePropertyInfo), '("gtkDecorationLayout", SettingsGtkDecorationLayoutPropertyInfo), '("gtkDialogsUseHeader", SettingsGtkDialogsUseHeaderPropertyInfo), '("gtkDndDragThreshold", SettingsGtkDndDragThresholdPropertyInfo), '("gtkDoubleClickDistance", SettingsGtkDoubleClickDistancePropertyInfo), '("gtkDoubleClickTime", SettingsGtkDoubleClickTimePropertyInfo), '("gtkEnableAccels", SettingsGtkEnableAccelsPropertyInfo), '("gtkEnableAnimations", SettingsGtkEnableAnimationsPropertyInfo), '("gtkEnableEventSounds", SettingsGtkEnableEventSoundsPropertyInfo), '("gtkEnableInputFeedbackSounds", SettingsGtkEnableInputFeedbackSoundsPropertyInfo), '("gtkEnablePrimaryPaste", SettingsGtkEnablePrimaryPastePropertyInfo), '("gtkEntryPasswordHintTimeout", SettingsGtkEntryPasswordHintTimeoutPropertyInfo), '("gtkEntrySelectOnFocus", SettingsGtkEntrySelectOnFocusPropertyInfo), '("gtkErrorBell", SettingsGtkErrorBellPropertyInfo), '("gtkFontName", SettingsGtkFontNamePropertyInfo), '("gtkFontRendering", SettingsGtkFontRenderingPropertyInfo), '("gtkFontconfigTimestamp", SettingsGtkFontconfigTimestampPropertyInfo), '("gtkHintFontMetrics", SettingsGtkHintFontMetricsPropertyInfo), '("gtkIconThemeName", SettingsGtkIconThemeNamePropertyInfo), '("gtkImModule", SettingsGtkImModulePropertyInfo), '("gtkInterfaceColorScheme", SettingsGtkInterfaceColorSchemePropertyInfo), '("gtkInterfaceContrast", SettingsGtkInterfaceContrastPropertyInfo), '("gtkKeynavUseCaret", SettingsGtkKeynavUseCaretPropertyInfo), '("gtkLabelSelectOnFocus", SettingsGtkLabelSelectOnFocusPropertyInfo), '("gtkLongPressTime", SettingsGtkLongPressTimePropertyInfo), '("gtkOverlayScrolling", SettingsGtkOverlayScrollingPropertyInfo), '("gtkPrimaryButtonWarpsSlider", SettingsGtkPrimaryButtonWarpsSliderPropertyInfo), '("gtkPrintBackends", SettingsGtkPrintBackendsPropertyInfo), '("gtkPrintPreviewCommand", SettingsGtkPrintPreviewCommandPropertyInfo), '("gtkRecentFilesEnabled", SettingsGtkRecentFilesEnabledPropertyInfo), '("gtkRecentFilesMaxAge", SettingsGtkRecentFilesMaxAgePropertyInfo), '("gtkShellShowsAppMenu", SettingsGtkShellShowsAppMenuPropertyInfo), '("gtkShellShowsDesktop", SettingsGtkShellShowsDesktopPropertyInfo), '("gtkShellShowsMenubar", SettingsGtkShellShowsMenubarPropertyInfo), '("gtkShowStatusShapes", SettingsGtkShowStatusShapesPropertyInfo), '("gtkSoundThemeName", SettingsGtkSoundThemeNamePropertyInfo), '("gtkSplitCursor", SettingsGtkSplitCursorPropertyInfo), '("gtkThemeName", SettingsGtkThemeNamePropertyInfo), '("gtkTitlebarDoubleClick", SettingsGtkTitlebarDoubleClickPropertyInfo), '("gtkTitlebarMiddleClick", SettingsGtkTitlebarMiddleClickPropertyInfo), '("gtkTitlebarRightClick", SettingsGtkTitlebarRightClickPropertyInfo), '("gtkXftAntialias", SettingsGtkXftAntialiasPropertyInfo), '("gtkXftDpi", SettingsGtkXftDpiPropertyInfo), '("gtkXftHinting", SettingsGtkXftHintingPropertyInfo), '("gtkXftHintstyle", SettingsGtkXftHintstylePropertyInfo), '("gtkXftRgba", SettingsGtkXftRgbaPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
settingsGtkAlternativeButtonOrder :: AttrLabelProxy "gtkAlternativeButtonOrder"
settingsGtkAlternativeButtonOrder = AttrLabelProxy

settingsGtkAlternativeSortArrows :: AttrLabelProxy "gtkAlternativeSortArrows"
settingsGtkAlternativeSortArrows = AttrLabelProxy

settingsGtkApplicationPreferDarkTheme :: AttrLabelProxy "gtkApplicationPreferDarkTheme"
settingsGtkApplicationPreferDarkTheme = AttrLabelProxy

settingsGtkCursorAspectRatio :: AttrLabelProxy "gtkCursorAspectRatio"
settingsGtkCursorAspectRatio = AttrLabelProxy

settingsGtkCursorBlink :: AttrLabelProxy "gtkCursorBlink"
settingsGtkCursorBlink = AttrLabelProxy

settingsGtkCursorBlinkTime :: AttrLabelProxy "gtkCursorBlinkTime"
settingsGtkCursorBlinkTime = AttrLabelProxy

settingsGtkCursorBlinkTimeout :: AttrLabelProxy "gtkCursorBlinkTimeout"
settingsGtkCursorBlinkTimeout = AttrLabelProxy

settingsGtkCursorThemeName :: AttrLabelProxy "gtkCursorThemeName"
settingsGtkCursorThemeName = AttrLabelProxy

settingsGtkCursorThemeSize :: AttrLabelProxy "gtkCursorThemeSize"
settingsGtkCursorThemeSize = AttrLabelProxy

settingsGtkDecorationLayout :: AttrLabelProxy "gtkDecorationLayout"
settingsGtkDecorationLayout = AttrLabelProxy

settingsGtkDialogsUseHeader :: AttrLabelProxy "gtkDialogsUseHeader"
settingsGtkDialogsUseHeader = AttrLabelProxy

settingsGtkDndDragThreshold :: AttrLabelProxy "gtkDndDragThreshold"
settingsGtkDndDragThreshold = AttrLabelProxy

settingsGtkDoubleClickDistance :: AttrLabelProxy "gtkDoubleClickDistance"
settingsGtkDoubleClickDistance = AttrLabelProxy

settingsGtkDoubleClickTime :: AttrLabelProxy "gtkDoubleClickTime"
settingsGtkDoubleClickTime = AttrLabelProxy

settingsGtkEnableAccels :: AttrLabelProxy "gtkEnableAccels"
settingsGtkEnableAccels = AttrLabelProxy

settingsGtkEnableAnimations :: AttrLabelProxy "gtkEnableAnimations"
settingsGtkEnableAnimations = AttrLabelProxy

settingsGtkEnableEventSounds :: AttrLabelProxy "gtkEnableEventSounds"
settingsGtkEnableEventSounds = AttrLabelProxy

settingsGtkEnableInputFeedbackSounds :: AttrLabelProxy "gtkEnableInputFeedbackSounds"
settingsGtkEnableInputFeedbackSounds = AttrLabelProxy

settingsGtkEnablePrimaryPaste :: AttrLabelProxy "gtkEnablePrimaryPaste"
settingsGtkEnablePrimaryPaste = AttrLabelProxy

settingsGtkEntryPasswordHintTimeout :: AttrLabelProxy "gtkEntryPasswordHintTimeout"
settingsGtkEntryPasswordHintTimeout = AttrLabelProxy

settingsGtkEntrySelectOnFocus :: AttrLabelProxy "gtkEntrySelectOnFocus"
settingsGtkEntrySelectOnFocus = AttrLabelProxy

settingsGtkErrorBell :: AttrLabelProxy "gtkErrorBell"
settingsGtkErrorBell = AttrLabelProxy

settingsGtkFontName :: AttrLabelProxy "gtkFontName"
settingsGtkFontName = AttrLabelProxy

settingsGtkFontRendering :: AttrLabelProxy "gtkFontRendering"
settingsGtkFontRendering = AttrLabelProxy

settingsGtkFontconfigTimestamp :: AttrLabelProxy "gtkFontconfigTimestamp"
settingsGtkFontconfigTimestamp = AttrLabelProxy

settingsGtkHintFontMetrics :: AttrLabelProxy "gtkHintFontMetrics"
settingsGtkHintFontMetrics = AttrLabelProxy

settingsGtkIconThemeName :: AttrLabelProxy "gtkIconThemeName"
settingsGtkIconThemeName = AttrLabelProxy

settingsGtkImModule :: AttrLabelProxy "gtkImModule"
settingsGtkImModule = AttrLabelProxy

settingsGtkInterfaceColorScheme :: AttrLabelProxy "gtkInterfaceColorScheme"
settingsGtkInterfaceColorScheme = AttrLabelProxy

settingsGtkInterfaceContrast :: AttrLabelProxy "gtkInterfaceContrast"
settingsGtkInterfaceContrast = AttrLabelProxy

settingsGtkKeynavUseCaret :: AttrLabelProxy "gtkKeynavUseCaret"
settingsGtkKeynavUseCaret = AttrLabelProxy

settingsGtkLabelSelectOnFocus :: AttrLabelProxy "gtkLabelSelectOnFocus"
settingsGtkLabelSelectOnFocus = AttrLabelProxy

settingsGtkLongPressTime :: AttrLabelProxy "gtkLongPressTime"
settingsGtkLongPressTime = AttrLabelProxy

settingsGtkOverlayScrolling :: AttrLabelProxy "gtkOverlayScrolling"
settingsGtkOverlayScrolling = AttrLabelProxy

settingsGtkPrimaryButtonWarpsSlider :: AttrLabelProxy "gtkPrimaryButtonWarpsSlider"
settingsGtkPrimaryButtonWarpsSlider = AttrLabelProxy

settingsGtkPrintBackends :: AttrLabelProxy "gtkPrintBackends"
settingsGtkPrintBackends = AttrLabelProxy

settingsGtkPrintPreviewCommand :: AttrLabelProxy "gtkPrintPreviewCommand"
settingsGtkPrintPreviewCommand = AttrLabelProxy

settingsGtkRecentFilesEnabled :: AttrLabelProxy "gtkRecentFilesEnabled"
settingsGtkRecentFilesEnabled = AttrLabelProxy

settingsGtkRecentFilesMaxAge :: AttrLabelProxy "gtkRecentFilesMaxAge"
settingsGtkRecentFilesMaxAge = AttrLabelProxy

settingsGtkShellShowsAppMenu :: AttrLabelProxy "gtkShellShowsAppMenu"
settingsGtkShellShowsAppMenu = AttrLabelProxy

settingsGtkShellShowsDesktop :: AttrLabelProxy "gtkShellShowsDesktop"
settingsGtkShellShowsDesktop = AttrLabelProxy

settingsGtkShellShowsMenubar :: AttrLabelProxy "gtkShellShowsMenubar"
settingsGtkShellShowsMenubar = AttrLabelProxy

settingsGtkShowStatusShapes :: AttrLabelProxy "gtkShowStatusShapes"
settingsGtkShowStatusShapes = AttrLabelProxy

settingsGtkSoundThemeName :: AttrLabelProxy "gtkSoundThemeName"
settingsGtkSoundThemeName = AttrLabelProxy

settingsGtkSplitCursor :: AttrLabelProxy "gtkSplitCursor"
settingsGtkSplitCursor = AttrLabelProxy

settingsGtkThemeName :: AttrLabelProxy "gtkThemeName"
settingsGtkThemeName = AttrLabelProxy

settingsGtkTitlebarDoubleClick :: AttrLabelProxy "gtkTitlebarDoubleClick"
settingsGtkTitlebarDoubleClick = AttrLabelProxy

settingsGtkTitlebarMiddleClick :: AttrLabelProxy "gtkTitlebarMiddleClick"
settingsGtkTitlebarMiddleClick = AttrLabelProxy

settingsGtkTitlebarRightClick :: AttrLabelProxy "gtkTitlebarRightClick"
settingsGtkTitlebarRightClick = AttrLabelProxy

settingsGtkXftAntialias :: AttrLabelProxy "gtkXftAntialias"
settingsGtkXftAntialias = AttrLabelProxy

settingsGtkXftDpi :: AttrLabelProxy "gtkXftDpi"
settingsGtkXftDpi = AttrLabelProxy

settingsGtkXftHinting :: AttrLabelProxy "gtkXftHinting"
settingsGtkXftHinting = AttrLabelProxy

settingsGtkXftHintstyle :: AttrLabelProxy "gtkXftHintstyle"
settingsGtkXftHintstyle = AttrLabelProxy

settingsGtkXftRgba :: AttrLabelProxy "gtkXftRgba"
settingsGtkXftRgba = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Settings = SettingsSignalList
type SettingsSignalList = ('[ '("gtkPrivateChanged", Gtk.StyleProvider.StyleProviderGtkPrivateChangedSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method Settings::reset_property
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkSettings` object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the setting to reset"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_settings_reset_property" gtk_settings_reset_property :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gtk", name = "Settings"})
    CString ->                              -- name : TBasicType TUTF8
    IO ()

-- | Undoes the effect of calling @/g_object_set()/@ to install an
-- application-specific value for a setting.
-- 
-- After this call, the setting will again follow the session-wide
-- value for this setting.
settingsResetProperty ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: a @GtkSettings@ object
    -> T.Text
    -- ^ /@name@/: the name of the setting to reset
    -> m ()
settingsResetProperty settings name = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    name' <- textToCString name
    gtk_settings_reset_property settings' name'
    touchManagedPtr settings
    freeMem name'
    return ()

#if defined(ENABLE_OVERLOADING)
data SettingsResetPropertyMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsResetPropertyMethodInfo a signature where
    overloadedMethod = settingsResetProperty

instance O.OverloadedMethodInfo SettingsResetPropertyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Settings.settingsResetProperty",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Settings.html#v:settingsResetProperty"
        })


#endif

-- method Settings::get_default
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Settings" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_settings_get_default" gtk_settings_get_default :: 
    IO (Ptr Settings)

-- | Gets the @GtkSettings@ object for the default display, creating
-- it if necessary.
-- 
-- See 'GI.Gtk.Objects.Settings.settingsGetForDisplay'.
settingsGetDefault ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m (Maybe Settings)
    -- ^ __Returns:__ a @GtkSettings@ object. If there is
    --   no default display, then returns 'P.Nothing'.
settingsGetDefault  = liftIO $ do
    result <- gtk_settings_get_default
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Settings) result'
        return result''
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method Settings::get_for_display
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "display"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Display" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkDisplay`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Settings" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_settings_get_for_display" gtk_settings_get_for_display :: 
    Ptr Gdk.Display.Display ->              -- display : TInterface (Name {namespace = "Gdk", name = "Display"})
    IO (Ptr Settings)

-- | Gets the @GtkSettings@ object for /@display@/, creating it if necessary.
settingsGetForDisplay ::
    (B.CallStack.HasCallStack, MonadIO m, Gdk.Display.IsDisplay a) =>
    a
    -- ^ /@display@/: a @GdkDisplay@
    -> m Settings
    -- ^ __Returns:__ a @GtkSettings@ object
settingsGetForDisplay display = liftIO $ do
    display' <- unsafeManagedPtrCastPtr display
    result <- gtk_settings_get_for_display display'
    checkUnexpectedReturnNULL "settingsGetForDisplay" result
    result' <- (newObject Settings) result
    touchManagedPtr display
    return result'

#if defined(ENABLE_OVERLOADING)
#endif


