// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A0__F48

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 64 },
    { .BLOCK_M = 64, .BLOCK_N = 64 },
    { .BLOCK_M = 64, .BLOCK_N = 64 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_64_0_F_F_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x1107585cu, 0xf6ee869du, 0, 46 }, // 1107585cf6ee869d = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950__P__16_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0xe2c052c5u, 0x9c9f70aau, 0, 6 }, // e2c052c59c9f70aa = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950__P__16_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0xdfe0a617u, 0x0ebb1139u, 0, 155 }, // dfe0a6170ebb1139 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950__P__16_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0x3905737eu, 0xf09131f6u, 0, 23 }, // 3905737ef09131f6 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950__P__16_16__CO__wave4_warp2_stg1--Arch_gfx950
    { 0xbc5c1d5fu, 0x7bcf73a7u, 40, 63 }, // bc5c1d5f7bcf73a7 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xc42fffa2u, 0x2bdbccb2u, 40, 23 }, // c42fffa22bdbccb2 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950__P__32_16__CO__wave4_warp2_stg1--Arch_gfx950
    { 0x11132112u, 0x80669471u, 97, 63 }, // 1113211280669471 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950__P__32_32__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xbc4bd0c7u, 0x41e229d7u, 97, 6 }, // bc4bd0c741e229d7 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950__P__32_32__CO__wave3_warp2_stg1--Arch_gfx950
    { 0xb8c831bau, 0x46ec9afeu, 97, 155 }, // b8c831ba46ec9afe = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950__P__32_32__CO__wave3_warp4_stg1--Arch_gfx950
    { 0xe6191696u, 0xec323fffu, 120, 63 }, // e6191696ec323fff = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950__P__64_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x56598b6du, 0xe061f3cbu, 120, 6 }, // 56598b6de061f3cb = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950__P__64_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x213d36eeu, 0x1bddcc95u, 120, 103 }, // 213d36ee1bddcc95 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950__P__64_16__CO__wave4_warp4_stg1--Arch_gfx950
    { 0x4acfbd60u, 0x6909f4fcu, 126, 46 }, // 4acfbd606909f4fc = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950__P__64_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x6f5bae32u, 0x5132c584u, 126, 63 }, // 6f5bae325132c584 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950__P__64_32__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xcf32749cu, 0x5f12ec51u, 126, 6 }, // cf32749c5f12ec51 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950__P__64_32__CO__wave3_warp2_stg1--Arch_gfx950
    { 0xeff38507u, 0x2cba5eabu, 132, 46 }, // eff385072cba5eab = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950__P__64_64__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x047fa68du, 0x2ee83be3u, 132, 63 }, // 047fa68d2ee83be3 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950__P__64_64__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xbef980b4u, 0x48e19734u, 132, 80 }, // bef980b448e19734 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_0___gfx950__P__64_64__CO__wave2_warp4_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx950_mod0
{{ 8,12, 7,17,17,17,16,16,16,16},
 { 1,12,17,17,17,17,16,16,16,16},
 { 6, 6, 6, 6, 6, 6, 6, 6,15,15},
 { 2, 2, 7, 6,15,15,15,15,15,15},
 { 1, 0, 0, 6, 6,15,15,15,15,15},
 { 6, 4, 6, 6,13,12,12,16,16,16},
 {14,10, 9, 9,14,14,14,14,14,14},
 { 1, 3,10,14,14,14,14,14,14,14},
 { 0, 0, 5,14,14,14,14,14,16,16},
 { 0,11, 4,14,14,14,14,14,14,14}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A0__F48 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 0;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 0;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = 0
// BLOCK_DMODEL = 64
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = False
// PADDED_HEAD = False
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.float16' AND inputs$BLOCK_DMODEL = 64 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = False AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

