/*
   Copyright The containerd Authors.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

package namespace

import (
	"context"

	containerd "github.com/containerd/containerd/v2/client"
	"github.com/containerd/containerd/v2/pkg/namespaces"
	"github.com/containerd/log"

	"github.com/containerd/nerdctl/v2/pkg/api/types"
	"github.com/containerd/nerdctl/v2/pkg/formatter"
	"github.com/containerd/nerdctl/v2/pkg/inspecttypes/native"
)

func Inspect(ctx context.Context, client *containerd.Client, inspectedNamespaces []string, options types.NamespaceInspectOptions) error {
	result := []interface{}{}
	warns := []error{}

	for _, ns := range inspectedNamespaces {
		ctx = namespaces.WithNamespace(ctx, ns)
		namespaceService := client.NamespaceService()
		if err := namespaceExists(ctx, namespaceService, ns); err != nil {
			warns = append(warns, err)
			continue
		}
		labels, err := namespaceService.Labels(ctx, ns)
		if err != nil {
			return err
		}
		nsInspect := native.Namespace{
			Name:   ns,
			Labels: &labels,
		}
		result = append(result, nsInspect)
	}
	if err := formatter.FormatSlice(options.Format, options.Stdout, result); err != nil {
		return err
	}
	for _, warn := range warns {
		log.G(ctx).Warn(warn)
	}
	return nil
}
