﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/firehose/FirehoseRequest.h>
#include <aws/firehose/Firehose_EXPORTS.h>
#include <aws/firehose/model/Tag.h>

#include <utility>

namespace Aws {
namespace Firehose {
namespace Model {

/**
 */
class TagDeliveryStreamRequest : public FirehoseRequest {
 public:
  AWS_FIREHOSE_API TagDeliveryStreamRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "TagDeliveryStream"; }

  AWS_FIREHOSE_API Aws::String SerializePayload() const override;

  AWS_FIREHOSE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the Firehose stream to which you want to add the tags.</p>
   */
  inline const Aws::String& GetDeliveryStreamName() const { return m_deliveryStreamName; }
  inline bool DeliveryStreamNameHasBeenSet() const { return m_deliveryStreamNameHasBeenSet; }
  template <typename DeliveryStreamNameT = Aws::String>
  void SetDeliveryStreamName(DeliveryStreamNameT&& value) {
    m_deliveryStreamNameHasBeenSet = true;
    m_deliveryStreamName = std::forward<DeliveryStreamNameT>(value);
  }
  template <typename DeliveryStreamNameT = Aws::String>
  TagDeliveryStreamRequest& WithDeliveryStreamName(DeliveryStreamNameT&& value) {
    SetDeliveryStreamName(std::forward<DeliveryStreamNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A set of key-value pairs to use to create the tags.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  TagDeliveryStreamRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  TagDeliveryStreamRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_deliveryStreamName;

  Aws::Vector<Tag> m_tags;
  bool m_deliveryStreamNameHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Firehose
}  // namespace Aws
