// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_fuse.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_fuse__A6__F360

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_F_F_0___gfx120x)xyzw" };
#else
{ R"xyzw(amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_F_F_0___gfx120x)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp32@16_16_3_F_F_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1201)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x3f6632f5u, 0xaa137ab6u, 0, 126 }, // 3f6632f5aa137ab6 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_F_F_0___gfx120x__P__16_16__CO__wave1_warp4_stg1--Arch_gfx1201
    { 0xafb7cd57u, 0x49e67e4cu, 0, 69 }, // afb7cd5749e67e4c = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_F_F_0___gfx120x__P__16_16__CO__wave2_warp4_stg1--Arch_gfx1201
    { 0x15a1b5b7u, 0x61ea1871u, 0, 143 }, // 15a1b5b761ea1871 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_F_F_0___gfx120x__P__16_16__CO__wave3_warp2_stg1--Arch_gfx1201
    { 0xda974da3u, 0x5e696a60u, 0, 29 }, // da974da35e696a60 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_F_F_0___gfx120x__P__16_16__CO__wave3_warp4_stg1--Arch_gfx1201
    { 0xa0a9ca42u, 0x0578d459u, 0, 86 }, // a0a9ca420578d459 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_F_F_0___gfx120x__P__16_16__CO__wave4_warp2_stg1--Arch_gfx1201
    { 0x9001b4ffu, 0xd8940117u, 0, 46 }, // 9001b4ffd8940117 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_F_F_0___gfx120x__P__16_16__CO__wave4_warp4_stg1--Arch_gfx1201
    { 0xdd685c23u, 0xf2d89d88u, 23, 109 }, // dd685c23f2d89d88 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_F_F_0___gfx120x__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1201
    { 0xb2fd41f8u, 0x45ac69dcu, 23, 6 }, // b2fd41f845ac69dc = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_F_F_0___gfx120x__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1201
    { 0x355e077au, 0x4093de90u, 23, 143 }, // 355e077a4093de90 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_F_F_0___gfx120x__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1201
    { 0x4bb47bd1u, 0x8ffb02d5u, 23, 86 }, // 4bb47bd18ffb02d5 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_F_F_0___gfx120x__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1201
    { 0xeb3cbda6u, 0x6963a8a8u, 63, 126 }, // eb3cbda66963a8a8 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_F_F_0___gfx120x__P__32_32__CO__wave1_warp4_stg1--Arch_gfx1201
    { 0xde0cd029u, 0xc0752940u, 63, 69 }, // de0cd029c0752940 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_F_F_0___gfx120x__P__32_32__CO__wave2_warp4_stg1--Arch_gfx1201
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_fuse_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_fuse_packed_string);

static int8_t lut[1][7][7] =
{
// GPU gfx1201_mod0
{{3,3,5,3,5,3,5},
 {1,9,3,1,3,5,5},
 {4,6,3,5,5,5,0},
 {4,8,6,8,8,3,5},
 {2,8,9,7,6,6,9},
 {2,7,7,8,9,8,9},
 {2,6,6,8,7,7,8}}
// End of GPU gfx1201_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelFuseContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelFuseContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_fuse__A6__F360 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 4;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_fuse__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 4;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp32:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = "i32"
// BLOCK_DMODEL = 16
// CAUSAL_TYPE = 3
// ENABLE_DROPOUT = False
// PADDED_HEAD = False
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_fuse WHERE gpu IN ('gfx1201_mod0') AND inputs$Q_dtype = 'torch.float32' AND inputs$BLOCK_DMODEL = 16 AND inputs$CAUSAL_TYPE = 3 AND inputs$ENABLE_DROPOUT = False AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

