// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_fuse.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_fuse__A0__F276

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_128_0_T_F_0___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_128_0_T_F_0___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_128_0_T_F_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xc7bbc10cu, 0x6ae72276u, 0, 6 }, // c7bbc10c6ae72276 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_128_0_T_F_0___gfx950__P__16_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x4f656332u, 0xc2f20bbcu, 0, 143 }, // 4f656332c2f20bbc = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_128_0_T_F_0___gfx950__P__16_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x6652492du, 0xacb8073bu, 0, 46 }, // 6652492dacb8073b = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_128_0_T_F_0___gfx950__P__16_16__CO__wave4_warp4_stg1--Arch_gfx950
    { 0xc275abd4u, 0x562654c1u, 23, 109 }, // c275abd4562654c1 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_128_0_T_F_0___gfx950__P__32_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0xd69dd6b9u, 0x561ed87du, 23, 6 }, // d69dd6b9561ed87d = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_128_0_T_F_0___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xe1ddcdc2u, 0x6ce06220u, 23, 69 }, // e1ddcdc26ce06220 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_128_0_T_F_0___gfx950__P__32_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x55a78ce3u, 0xd37ea49fu, 63, 109 }, // 55a78ce3d37ea49f = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_128_0_T_F_0___gfx950__P__32_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x8ab1a998u, 0x435da0e1u, 103, 69 }, // 8ab1a998435da0e1 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_128_0_T_F_0___gfx950__P__64_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0xd951c8e2u, 0x5235dff0u, 160, 109 }, // d951c8e25235dff0 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_128_0_T_F_0___gfx950__P__64_32__CO__wave1_warp2_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_fuse_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_fuse_packed_string);

static int8_t lut[1][7][7] =
{
// GPU gfx950_mod0
{{4,7,3,6,4,4,4},
 {7,7,0,8,4,0,4},
 {3,4,4,0,0,0,0},
 {4,4,0,0,0,0,4},
 {4,4,4,4,1,0,8},
 {4,4,4,4,4,4,8},
 {4,4,4,4,4,8,8}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelFuseContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelFuseContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_fuse__A0__F276 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 2;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_fuse__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 2;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 128
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = False
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_fuse WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 128 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

