#ifndef REFS_H
#define REFS_H

#include "commit.h"
#include "repository.h"
#include "repo-settings.h"

struct fsck_options;
struct object_id;
struct ref_store;
struct strbuf;
struct string_list;
struct string_list_item;
struct worktree;

enum ref_storage_format ref_storage_format_by_name(const char *name);
const char *ref_storage_format_to_name(enum ref_storage_format ref_storage_format);

enum ref_transaction_error {
	/* Default error code */
	REF_TRANSACTION_ERROR_GENERIC = -1,
	/* Ref name conflict like A vs A/B */
	REF_TRANSACTION_ERROR_NAME_CONFLICT = -2,
	/* Ref to be created already exists */
	REF_TRANSACTION_ERROR_CREATE_EXISTS = -3,
	/* ref expected but doesn't exist */
	REF_TRANSACTION_ERROR_NONEXISTENT_REF = -4,
	/* Provided old_oid or old_target of reference doesn't match actual */
	REF_TRANSACTION_ERROR_INCORRECT_OLD_VALUE = -5,
	/* Provided new_oid or new_target is invalid */
	REF_TRANSACTION_ERROR_INVALID_NEW_VALUE = -6,
	/* Expected ref to be symref, but is a regular ref */
	REF_TRANSACTION_ERROR_EXPECTED_SYMREF = -7,
	/* Cannot create ref due to case-insensitive filesystem */
	REF_TRANSACTION_ERROR_CASE_CONFLICT = -8,
};

/*
 * Resolve a reference, recursively following symbolic references.
 *
 * Return the name of the non-symbolic reference that ultimately pointed
 * at the resolved object name.  The return value, if not NULL, is a
 * pointer into either a static buffer or the input ref.
 *
 * If oid is non-NULL, store the referred-to object's name in it.
 *
 * If the reference cannot be resolved to an object, the behavior
 * depends on the RESOLVE_REF_READING flag:
 *
 * - If RESOLVE_REF_READING is set, return NULL.
 *
 * - If RESOLVE_REF_READING is not set, clear oid and return the name of
 *   the last reference name in the chain, which will either be a non-symbolic
 *   reference or an undefined reference.  If this is a prelude to
 *   "writing" to the ref, the return value is the name of the ref
 *   that will actually be created or changed.
 *
 * If the RESOLVE_REF_NO_RECURSE flag is passed, only resolves one
 * level of symbolic reference.  The value stored in oid for a symbolic
 * reference will always be null_oid in this case, and the return
 * value is the reference that the symref refers to directly.
 *
 * If flags is non-NULL, set the value that it points to the
 * combination of REF_ISPACKED (if the reference was found among the
 * packed references), REF_ISSYMREF (if the initial reference was a
 * symbolic reference), REF_BAD_NAME (if the reference name is ill
 * formed --- see RESOLVE_REF_ALLOW_BAD_NAME below), and REF_ISBROKEN
 * (if the ref is malformed or has a bad name). See refs.h for more detail
 * on each flag.
 *
 * If ref is not a properly-formatted, normalized reference, return
 * NULL.  If more than MAXDEPTH recursive symbolic lookups are needed,
 * give up and return NULL.
 *
 * RESOLVE_REF_ALLOW_BAD_NAME allows resolving refs even when their
 * name is invalid according to git-check-ref-format(1).  If the name
 * is bad then the value stored in oid will be null_oid and the two
 * flags REF_ISBROKEN and REF_BAD_NAME will be set.
 *
 * Even with RESOLVE_REF_ALLOW_BAD_NAME, names that escape the refs/
 * directory and do not consist of all caps and underscores cannot be
 * resolved. The function returns NULL for such ref names.
 * Caps and underscores refers to the pseudorefs, such as HEAD,
 * FETCH_HEAD and friends, that all live outside of the refs/ directory.
 */
#define RESOLVE_REF_READING 0x01
#define RESOLVE_REF_NO_RECURSE 0x02
#define RESOLVE_REF_ALLOW_BAD_NAME 0x04

const char *refs_resolve_ref_unsafe(struct ref_store *refs,
				    const char *refname,
				    int resolve_flags,
				    struct object_id *oid,
				    int *flags);

char *refs_resolve_refdup(struct ref_store *refs,
			  const char *refname, int resolve_flags,
			  struct object_id *oid, int *flags);

int refs_read_ref_full(struct ref_store *refs, const char *refname,
		       int resolve_flags, struct object_id *oid, int *flags);

int refs_read_ref(struct ref_store *refs, const char *refname, struct object_id *oid);

#define NOT_A_SYMREF -2

/*
 * Read the symbolic ref named "refname" and write its immediate referent into
 * the provided buffer. Referent is left empty if "refname" is not a symbolic
 * ref. It does not resolve the symbolic reference recursively in case the
 * target is also a symbolic ref.
 *
 * Returns 0 on success, -2 if the "refname" is not a symbolic ref,
 * -1 otherwise.
 */
int refs_read_symbolic_ref(struct ref_store *ref_store, const char *refname,
			   struct strbuf *referent);

/*
 * Return 0 if a reference named refname could be created without
 * conflicting with the name of an existing reference. Otherwise,
 * return a negative value and write an explanation to err. If extras
 * is non-NULL, it is a list of additional refnames with which refname
 * is not allowed to conflict. If skip is non-NULL, ignore potential
 * conflicts with refs in skip (e.g., because they are scheduled for
 * deletion in the same operation). Behavior is undefined if the same
 * name is listed in both extras and skip.
 *
 * Two reference names conflict if one of them exactly matches the
 * leading components of the other; e.g., "foo/bar" conflicts with
 * both "foo" and with "foo/bar/baz" but not with "foo/bar" or
 * "foo/barbados".
 *
 * If `initial_transaction` is truish, then all collision checks with
 * preexisting refs are skipped.
 *
 * extras and skip must be sorted.
 */
enum ref_transaction_error refs_verify_refname_available(struct ref_store *refs,
						 const char *refname,
						 const struct string_list *extras,
						 const struct string_list *skip,
						 unsigned int initial_transaction,
						 struct strbuf *err);

int refs_ref_exists(struct ref_store *refs, const char *refname);

int should_autocreate_reflog(enum log_refs_config log_all_ref_updates,
			     const char *refname);

int is_branch(const char *refname);

#define REF_STORE_CREATE_ON_DISK_IS_WORKTREE (1 << 0)

int ref_store_create_on_disk(struct ref_store *refs, int flags, struct strbuf *err);

/*
 * Release all memory and resources associated with the ref store.
 */
void ref_store_release(struct ref_store *ref_store);

/*
 * Remove the ref store from disk. This deletes all associated data.
 */
int ref_store_remove_on_disk(struct ref_store *refs, struct strbuf *err);

/*
 * Return the peeled value of the oid currently being iterated via
 * for_each_ref(), etc. This is equivalent to calling:
 *
 *   peel_object(r, oid, &peeled);
 *
 * with the "oid" value given to the each_ref_fn callback, except
 * that some ref storage may be able to answer the query without
 * actually loading the object in memory.
 */
int peel_iterated_oid(struct repository *r,
		      const struct object_id *base, struct object_id *peeled);

/**
 * Resolve refname in the nested "gitlink" repository in the specified
 * submodule (which must be non-NULL). If the resolution is
 * successful, return 0 and set oid to the name of the object;
 * otherwise, return a non-zero value.
 */
int repo_resolve_gitlink_ref(struct repository *r,
			     const char *submodule, const char *refname,
			     struct object_id *oid);

/*
 * Return true iff abbrev_name is a possible abbreviation for
 * full_name according to the rules defined by ref_rev_parse_rules in
 * refs.c.
 */
int refname_match(const char *abbrev_name, const char *full_name);

/*
 * Given a 'prefix' expand it by the rules in 'ref_rev_parse_rules' and add
 * the results to 'prefixes'
 */
struct strvec;
void expand_ref_prefix(struct strvec *prefixes, const char *prefix);

int expand_ref(struct repository *r, const char *str, int len, struct object_id *oid, char **ref);
int repo_dwim_ref(struct repository *r, const char *str, int len,
		  struct object_id *oid, char **ref, int nonfatal_dangling_mark);
int repo_dwim_log(struct repository *r, const char *str, int len, struct object_id *oid, char **ref);

/*
 * Retrieves the default branch name for newly-initialized repositories.
 *
 * The return value is an allocated string.
 */
char *repo_default_branch_name(struct repository *r, int quiet);

/*
 * Copy "name" to "sb", expanding any special @-marks as handled by
 * repo_interpret_branch_name(). The result is a non-qualified branch name
 * (so "foo" or "origin/master" instead of "refs/heads/foo" or
 * "refs/remotes/origin/master").
 *
 * Note that the resulting name may not be a syntactically valid refname.
 *
 * If "allowed" is non-zero, restrict the set of allowed expansions. See
 * repo_interpret_branch_name() for details.
 */
void copy_branchname(struct strbuf *sb, const char *name,
		       unsigned allowed);

/*
 * Like copy_branchname() above, but confirm that the result is
 * syntactically valid to be used as a local branch name in refs/heads/.
 *
 * The return value is "0" if the result is valid, and "-1" otherwise.
 */
int check_branch_ref(struct strbuf *sb, const char *name);

/*
 * Similar for a tag name in refs/tags/.
 *
 * The return value is "0" if the result is valid, and "-1" otherwise.
 */
int check_tag_ref(struct strbuf *sb, const char *name);

/*
 * A ref_transaction represents a collection of reference updates that
 * should succeed or fail together.
 *
 * Calling sequence
 * ----------------
 *
 * - Allocate and initialize a `struct ref_transaction` by calling
 *   `ref_transaction_begin()`.
 *
 * - Specify the intended ref updates by calling one or more of the
 *   following functions:
 *   - `ref_transaction_update()`
 *   - `ref_transaction_create()`
 *   - `ref_transaction_delete()`
 *   - `ref_transaction_verify()`
 *
 * - Then either:
 *
 *   - Optionally call `ref_transaction_prepare()` to prepare the
 *     transaction. This locks all references, checks preconditions,
 *     etc. but doesn't finalize anything. If this step fails, the
 *     transaction has been closed and can only be freed. If this step
 *     succeeds, then `ref_transaction_commit()` is almost certain to
 *     succeed. However, you can still call `ref_transaction_abort()`
 *     if you decide not to commit the transaction after all.
 *
 *   - Call `ref_transaction_commit()` to execute the transaction,
 *     make the changes permanent, and release all locks. If you
 *     haven't already called `ref_transaction_prepare()`, then
 *     `ref_transaction_commit()` calls it for you.
 *
 *   Or
 *
 *   - Call `ref_transaction_begin()` with REF_TRANSACTION_FLAG_INITIAL if the
 *     ref database is known to be empty and have no other writers (e.g. during
 *     clone). This is likely to be much faster than without the flag.
 *
 * - Then finally, call `ref_transaction_free()` to free the
 *   `ref_transaction` data structure.
 *
 * At any time before calling `ref_transaction_commit()`, you can call
 * `ref_transaction_abort()` to abort the transaction, rollback any
 * locks, and free any associated resources (including the
 * `ref_transaction` data structure).
 *
 * Putting it all together, a complete reference update looks like
 *
 *         struct ref_transaction *transaction;
 *         struct strbuf err = STRBUF_INIT;
 *         int ret = 0;
 *
 *         transaction = ref_store_transaction_begin(refs, 0, &err);
 *         if (!transaction ||
 *             ref_transaction_update(...) ||
 *             ref_transaction_create(...) ||
 *             ...etc... ||
 *             ref_transaction_commit(transaction, &err)) {
 *                 error("%s", err.buf);
 *                 ret = -1;
 *         }
 *         ref_transaction_free(transaction);
 *         strbuf_release(&err);
 *         return ret;
 *
 * Error handling
 * --------------
 *
 * On error, transaction functions append a message about what
 * went wrong to the 'err' argument.  The message mentions what
 * ref was being updated (if any) when the error occurred so it
 * can be passed to 'die' or 'error' as-is.
 *
 * The message is appended to err without first clearing err.
 * err will not be '\n' terminated.
 *
 * Caveats
 * -------
 *
 * Note that no locks are taken, and no refs are read, until
 * `ref_transaction_prepare()` or `ref_transaction_commit()` is
 * called. So, for example, `ref_transaction_verify()` won't report a
 * verification failure until the commit is attempted.
 */
struct ref_transaction;

/*
 * Bit values set in the flags argument passed to each_ref_fn() and
 * stored in ref_iterator::flags. Other bits are for internal use
 * only:
 */
enum reference_status {
	/* Reference is a symbolic reference. */
	REF_ISSYMREF = (1 << 0),

	/* Reference is a packed reference. */
	REF_ISPACKED = (1 << 1),

	/*
	 * Reference cannot be resolved to an object name: dangling symbolic
	 * reference (directly or indirectly), corrupt reference file,
	 * reference exists but name is bad, or symbolic reference refers to
	 * ill-formatted reference name.
	 */
	REF_ISBROKEN = (1 << 2),

	/*
	 * Reference name is not well formed.
	 *
	 * See git-check-ref-format(1) for the definition of well formed ref names.
	 */
	REF_BAD_NAME = (1 << 3),
};

/* A reference passed to `for_each_ref()`-style callbacks. */
struct reference {
	/* The fully-qualified name of the reference. */
	const char *name;

	/* The target of a symbolic ref. `NULL` for direct references. */
	const char *target;

	/*
	 * The object ID of a reference. Either the direct object ID or the
	 * resolved object ID in the case of a symbolic ref. May be the zero
	 * object ID in case the symbolic ref cannot be resolved.
	 */
	const struct object_id *oid;

	/*
	 * An optional peeled object ID. This field _may_ be set for tags in
	 * case the peeled value is present in the backend. Please refer to
	 * `reference_get_peeled_oid()`.
	 */
	const struct object_id *peeled_oid;

	/* A bitfield of `enum reference_status` flags. */
	unsigned flags;
};

/*
 * Peel the tag to a non-tag commit. If present, this uses the peeled object ID
 * exposed by the reference backend. Otherwise, the object is peeled via the
 * object database, which is less efficient.
 *
 * Return `0` if the reference could be peeled, a negative error code
 * otherwise.
 */
int reference_get_peeled_oid(struct repository *repo,
			     const struct reference *ref,
			     struct object_id *peeled_oid);

/*
 * The signature for the callback function for the for_each_*()
 * functions below.  The memory pointed to by the `struct reference`
 * argument is only guaranteed to be valid for the duration of a
 * single callback invocation.
 */
typedef int each_ref_fn(const struct reference *ref, void *cb_data);

/*
 * The following functions invoke the specified callback function for
 * each reference indicated.  If the function ever returns a nonzero
 * value, stop the iteration and return that value.  Please note that
 * it is not safe to modify references while an iteration is in
 * progress, unless the same callback function invocation that
 * modifies the reference also returns a nonzero value to immediately
 * stop the iteration. Returned references are sorted.
 */
int refs_head_ref(struct ref_store *refs,
		  each_ref_fn fn, void *cb_data);
int refs_for_each_ref(struct ref_store *refs,
		      each_ref_fn fn, void *cb_data);
int refs_for_each_ref_in(struct ref_store *refs, const char *prefix,
			 each_ref_fn fn, void *cb_data);
int refs_for_each_tag_ref(struct ref_store *refs,
			  each_ref_fn fn, void *cb_data);
int refs_for_each_branch_ref(struct ref_store *refs,
			     each_ref_fn fn, void *cb_data);
int refs_for_each_remote_ref(struct ref_store *refs,
			     each_ref_fn fn, void *cb_data);
int refs_for_each_replace_ref(struct ref_store *refs,
			      each_ref_fn fn, void *cb_data);

/*
 * references matching any pattern in "exclude_patterns" are omitted from the
 * result set on a best-effort basis.
 */
int refs_for_each_fullref_in(struct ref_store *refs, const char *prefix,
			     const char **exclude_patterns,
			     each_ref_fn fn, void *cb_data);

/**
 * iterate all refs in "patterns" by partitioning patterns into disjoint sets
 * and iterating the longest-common prefix of each set.
 *
 * references matching any pattern in "exclude_patterns" are omitted from the
 * result set on a best-effort basis.
 *
 * callers should be prepared to ignore references that they did not ask for.
 */
int refs_for_each_fullref_in_prefixes(struct ref_store *refs,
				      const char *namespace,
				      const char **patterns,
				      const char **exclude_patterns,
				      each_ref_fn fn, void *cb_data);

/* iterates all refs that match the specified glob pattern. */
int refs_for_each_glob_ref(struct ref_store *refs, each_ref_fn fn,
			   const char *pattern, void *cb_data);

int refs_for_each_glob_ref_in(struct ref_store *refs, each_ref_fn fn,
			      const char *pattern, const char *prefix, void *cb_data);

int refs_head_ref_namespaced(struct ref_store *refs, each_ref_fn fn, void *cb_data);

/*
 * references matching any pattern in "exclude_patterns" are omitted from the
 * result set on a best-effort basis.
 */
int refs_for_each_namespaced_ref(struct ref_store *refs,
				 const char **exclude_patterns,
				 each_ref_fn fn, void *cb_data);

/* can be used to learn about broken ref and symref */
int refs_for_each_rawref(struct ref_store *refs, each_ref_fn fn, void *cb_data);
int refs_for_each_rawref_in(struct ref_store *refs, const char *prefix,
			    each_ref_fn fn, void *cb_data);

/*
 * Iterates over all refs including root refs, i.e. pseudorefs and HEAD.
 */
int refs_for_each_include_root_refs(struct ref_store *refs, each_ref_fn fn,
				    void *cb_data);

/*
 * Normalizes partial refs to their fully qualified form.
 * Will prepend <prefix> to the <pattern> if it doesn't start with 'refs/'.
 * <prefix> will default to 'refs/' if NULL.
 *
 * item.string will be set to the result.
 * item.util will be set to NULL if <pattern> contains glob characters, or
 * non-NULL if it doesn't.
 */
void normalize_glob_ref(struct string_list_item *item, const char *prefix,
			const char *pattern);

static inline const char *has_glob_specials(const char *pattern)
{
	return strpbrk(pattern, "?*[");
}

void refs_warn_dangling_symrefs(struct ref_store *refs, FILE *fp,
				const char *indent, int dry_run,
				const struct string_list *refnames);

/*
 * Flags for controlling behaviour of refs_optimize()
 * REFS_OPTIMIZE_PRUNE: Prune loose refs after packing
 * REFS_OPTIMIZE_AUTO: Pack refs on a best effort basis. The heuristics and end
 *                     result are decided by the ref backend. Backends may ignore
 *                     this flag and fall back to a normal repack.
 */
#define REFS_OPTIMIZE_PRUNE (1 << 0)
#define REFS_OPTIMIZE_AUTO  (1 << 1)

struct refs_optimize_opts {
	unsigned int flags;
	struct ref_exclusions *exclusions;
	struct string_list *includes;
};

/*
 * Optimize the ref store. The exact behavior is up to the backend.
 * For the files backend, this is equivalent to packing refs.
 */
int refs_optimize(struct ref_store *refs, struct refs_optimize_opts *opts);

/*
 * Check if refs backend can be optimized by calling 'refs_optimize'.
 */
int refs_optimize_required(struct ref_store *ref_store,
			   struct refs_optimize_opts *opts,
			   bool *required);

/*
 * Setup reflog before using. Fill in err and return -1 on failure.
 */
int refs_create_reflog(struct ref_store *refs, const char *refname,
		       struct strbuf *err);

/**
 * Reads log for the value of ref during at_time (in which case "cnt" should be
 * negative) or the reflog "cnt" entries from the top (in which case "at_time"
 * should be 0).
 *
 * If we found the reflog entry in question, returns 0 (and details of the
 * entry can be found in the out-parameters).
 *
 * If we ran out of reflog entries, the out-parameters are filled with the
 * details of the oldest entry we did find, and the function returns 1. Note
 * that there is one important special case here! If the reflog was empty
 * and the caller asked for the 0-th cnt, we will return "1" but leave the
 * "oid" field untouched.
 **/
int read_ref_at(struct ref_store *refs,
		const char *refname, unsigned int flags,
		timestamp_t at_time, int cnt,
		struct object_id *oid, char **msg,
		timestamp_t *cutoff_time, int *cutoff_tz, int *cutoff_cnt);

/** Check if a particular reflog exists */
int refs_reflog_exists(struct ref_store *refs, const char *refname);

/*
 * Delete the specified reference. If old_oid is non-NULL, then
 * verify that the current value of the reference is old_oid before
 * deleting it. If old_oid is NULL, delete the reference if it
 * exists, regardless of its old value. It is an error for old_oid to
 * be null_oid. msg and flags are passed through to
 * ref_transaction_delete().
 */
int refs_delete_ref(struct ref_store *refs, const char *msg,
		    const char *refname,
		    const struct object_id *old_oid,
		    unsigned int flags);

/*
 * Delete the specified references. If there are any problems, emit
 * errors but attempt to keep going (i.e., the deletes are not done in
 * an all-or-nothing transaction). msg and flags are passed through to
 * ref_transaction_delete().
 */
int refs_delete_refs(struct ref_store *refs, const char *msg,
		     struct string_list *refnames, unsigned int flags);

/** Delete a reflog */
int refs_delete_reflog(struct ref_store *refs, const char *refname);

/*
 * Callback to process a reflog entry found by the iteration functions (see
 * below).
 *
 * The committer parameter is a single string, in the form
 * "$GIT_COMMITTER_NAME <$GIT_COMMITTER_EMAIL>" (without double quotes).
 *
 * The timestamp parameter gives the time when entry was created as the number
 * of seconds since the UNIX epoch.
 *
 * The tz parameter gives the timezone offset for the user who created
 * the reflog entry, and its value gives a positive or negative offset
 * from UTC.  Its absolute value is formed by multiplying the hour
 * part by 100 and adding the minute part.  For example, 1 hour ahead
 * of UTC, CET == "+0100", is represented as positive one hundred (not
 * positive sixty).
 *
 * The msg parameter is a single complete line; a reflog message given
 * to refs_delete_ref, refs_update_ref, etc. is returned to the
 * callback normalized---each run of whitespaces are squashed into a
 * single whitespace, trailing whitespace, if exists, is trimmed, and
 * then a single LF is added at the end.
 *
 * The cb_data is a caller-supplied pointer given to the iterator
 * functions.
 */
typedef int each_reflog_ent_fn(const char *refname,
			       struct object_id *old_oid,
			       struct object_id *new_oid,
			       const char *committer,
			       timestamp_t timestamp,
			       int tz, const char *msg,
			       void *cb_data);

/* Iterate over reflog entries in the log for `refname`. */

/* oldest entry first */
int refs_for_each_reflog_ent(struct ref_store *refs, const char *refname,
			     each_reflog_ent_fn fn, void *cb_data);

/* youngest entry first */
int refs_for_each_reflog_ent_reverse(struct ref_store *refs,
				     const char *refname,
				     each_reflog_ent_fn fn,
				     void *cb_data);

/*
 * The signature for the callback function for the refs_for_each_reflog()
 * functions below. The memory pointed to by the refname argument is only
 * guaranteed to be valid for the duration of a single callback invocation.
 */
typedef int each_reflog_fn(const char *refname, void *cb_data);

/*
 * Calls the specified function for each reflog file until it returns nonzero,
 * and returns the value. Reflog file order is unspecified.
 */
int refs_for_each_reflog(struct ref_store *refs, each_reflog_fn fn, void *cb_data);

#define REFNAME_ALLOW_ONELEVEL 1
#define REFNAME_REFSPEC_PATTERN 2

/*
 * Return 0 iff refname has the correct format for a refname according
 * to the rules described in Documentation/git-check-ref-format.adoc.
 * If REFNAME_ALLOW_ONELEVEL is set in flags, then accept one-level
 * reference names.  If REFNAME_REFSPEC_PATTERN is set in flags, then
 * allow a single "*" wildcard character in the refspec. No leading or
 * repeated slashes are accepted.
 */
int check_refname_format(const char *refname, int flags);

/*
 * Check the reference database for consistency. Return 0 if refs and
 * reflogs are consistent, and non-zero otherwise. The errors will be
 * written to stderr.
 */
int refs_fsck(struct ref_store *refs, struct fsck_options *o,
	      struct worktree *wt);

/*
 * Apply the rules from check_refname_format, but mutate the result until it
 * is acceptable, and place the result in "out".
 */
void sanitize_refname_component(const char *refname, struct strbuf *out);

const char *prettify_refname(const char *refname);

char *refs_shorten_unambiguous_ref(struct ref_store *refs,
				   const char *refname, int strict);

/** rename ref, return 0 on success **/
int refs_rename_ref(struct ref_store *refs, const char *oldref,
		    const char *newref, const char *logmsg);

/** copy ref, return 0 on success **/
int refs_copy_existing_ref(struct ref_store *refs, const char *oldref,
		    const char *newref, const char *logmsg);

int refs_update_symref(struct ref_store *refs, const char *refname,
		       const char *target, const char *logmsg);

int refs_update_symref_extended(struct ref_store *refs, const char *refname,
		       const char *target, const char *logmsg,
		       struct strbuf *referent, int create_only);

enum action_on_err {
	UPDATE_REFS_MSG_ON_ERR,
	UPDATE_REFS_DIE_ON_ERR,
	UPDATE_REFS_QUIET_ON_ERR
};

enum ref_transaction_flag {
	/*
	 * The ref transaction is part of the initial creation of the ref store
	 * and can thus assume that the ref store is completely empty. This
	 * allows the backend to perform the transaction more efficiently by
	 * skipping certain checks.
	 *
	 * It is a bug to set this flag when there might be other processes
	 * accessing the repository or if there are existing references that
	 * might conflict with the ones being created. All old_oid values must
	 * either be absent or null_oid.
	 */
	REF_TRANSACTION_FLAG_INITIAL = (1 << 0),

	/*
	 * The transaction mechanism by default fails all updates if any conflict
	 * is detected. This flag allows transactions to partially apply updates
	 * while rejecting updates which do not match the expected state.
	 */
	REF_TRANSACTION_ALLOW_FAILURE = (1 << 1),
};

/*
 * Begin a reference transaction.  The reference transaction must
 * be freed by calling ref_transaction_free().
 */
struct ref_transaction *ref_store_transaction_begin(struct ref_store *refs,
						    unsigned int flags,
						    struct strbuf *err);

/*
 * Reference transaction updates
 *
 * The following four functions add a reference check or update to a
 * ref_transaction.  They have some common similar parameters:
 *
 *     transaction -- a pointer to an open ref_transaction, obtained
 *         from ref_transaction_begin().
 *
 *     refname -- the name of the reference to be affected.
 *
 *     new_oid -- the object ID that should be set to be the new value
 *         of the reference. Some functions allow this parameter to be
 *         NULL, meaning that the reference is not changed, or
 *         null_oid, meaning that the reference should be deleted. A
 *         copy of this value is made in the transaction.
 *
 *     old_oid -- the object ID that the reference must have before
 *         the update. Some functions allow this parameter to be NULL,
 *         meaning that the old value of the reference is not checked,
 *         or null_oid, meaning that the reference must not exist
 *         before the update. A copy of this value is made in the
 *         transaction.
 *
 *     new_target -- the target reference that the reference will be
 *         updated to point to. If the reference is a regular reference,
 *         it will be converted to a symbolic reference. Cannot be set
 *         together with `new_oid`. A copy of this value is made in the
 *         transaction.
 *
 *     old_target -- the reference that the reference must be pointing to.
 *         Canont be set together with `old_oid`. A copy of this value is
 *         made in the transaction.
 *
 *     flags -- flags affecting the update, passed to
 *         update_ref_lock(). Possible flags: REF_NO_DEREF,
 *         REF_FORCE_CREATE_REFLOG. See those constants for more
 *         information.
 *
 *     msg -- a message describing the change (for the reflog).
 *
 *     err -- a strbuf for receiving a description of any error that
 *         might have occurred.
 *
 * The functions make internal copies of refname and msg, so the
 * caller retains ownership of these parameters.
 *
 * The functions return 0 on success and non-zero on failure. A
 * failure means that the transaction as a whole has failed and needs
 * to be rolled back.
 */

/*
 * The following flags can be passed to ref_transaction_update() etc.
 * Internally, they are stored in `ref_update::flags`, along with some
 * internal flags.
 */

/*
 * Act on the ref directly; i.e., without dereferencing symbolic refs.
 * If this flag is not specified, then symbolic references are
 * dereferenced and the update is applied to the referent.
 */
#define REF_NO_DEREF (1 << 0)

/*
 * Force the creation of a reflog for this reference, even if it
 * didn't previously have a reflog.
 */
#define REF_FORCE_CREATE_REFLOG (1 << 1)

/*
 * Blindly write an object_id. This is useful for testing data corruption
 * scenarios.
 */
#define REF_SKIP_OID_VERIFICATION (1 << 10)

/*
 * Skip verifying refname. This is useful for testing data corruption scenarios.
 */
#define REF_SKIP_REFNAME_VERIFICATION (1 << 11)

/*
 * Skip creation of a reflog entry, even if it would have otherwise been
 * created.
 */
#define REF_SKIP_CREATE_REFLOG (1 << 12)

/*
 * When writing a REF_LOG_ONLY record, use the old and new object IDs provided
 * in the update instead of resolving the old object ID. The caller must also
 * set both REF_HAVE_OLD and REF_HAVE_NEW.
 */
#define REF_LOG_USE_PROVIDED_OIDS (1 << 13)

/*
 * Bitmask of all of the flags that are allowed to be passed in to
 * ref_transaction_update() and friends:
 */
#define REF_TRANSACTION_UPDATE_ALLOWED_FLAGS                                  \
	(REF_NO_DEREF | REF_FORCE_CREATE_REFLOG | REF_SKIP_OID_VERIFICATION | \
	 REF_SKIP_REFNAME_VERIFICATION | REF_SKIP_CREATE_REFLOG | REF_LOG_USE_PROVIDED_OIDS)

/*
 * Add a reference update to transaction. `new_oid` is the value that
 * the reference should have after the update, or `null_oid` if it
 * should be deleted. If `new_oid` is NULL, then the reference is not
 * changed at all. `old_oid` is the value that the reference must have
 * before the update, or `null_oid` if it must not have existed
 * beforehand. The old value is checked after the lock is taken to
 * prevent races. If the old value doesn't agree with old_oid, the
 * whole transaction fails. If old_oid is NULL, then the previous
 * value is not checked. If `old_target` is not NULL, treat the reference
 * as a symbolic ref and validate that its target before the update is
 * `old_target`. If the `new_target` is not NULL, then the reference
 * will be updated to a symbolic ref which targets `new_target`.
 * Together, these allow us to update between regular refs and symrefs.
 *
 * See the above comment "Reference transaction updates" for more
 * information.
 */
int ref_transaction_update(struct ref_transaction *transaction,
			   const char *refname,
			   const struct object_id *new_oid,
			   const struct object_id *old_oid,
			   const char *new_target,
			   const char *old_target,
			   unsigned int flags, const char *msg,
			   struct strbuf *err);

/*
 * Similar to `ref_transaction_update`, but this function is only for adding
 * a reflog update. Supports providing custom committer information. The index
 * field can be utiltized to order updates as desired. When set to zero, the
 * updates default to being ordered by refname.
 */
int ref_transaction_update_reflog(struct ref_transaction *transaction,
				  const char *refname,
				  const struct object_id *new_oid,
				  const struct object_id *old_oid,
				  const char *committer_info,
				  const char *msg,
				  uint64_t index,
				  struct strbuf *err);

/*
 * Add a reference creation to transaction. new_oid is the value that
 * the reference should have after the update; it must not be
 * null_oid. It is verified that the reference does not exist
 * already.
 *
 * See the above comment "Reference transaction updates" for more
 * information.
 */
int ref_transaction_create(struct ref_transaction *transaction,
			   const char *refname,
			   const struct object_id *new_oid,
			   const char *new_target,
			   unsigned int flags, const char *msg,
			   struct strbuf *err);

/*
 * Add a reference deletion to transaction. If old_oid is non-NULL,
 * then it holds the value that the reference should have had before
 * the update (which must not be null_oid).
 *
 * See the above comment "Reference transaction updates" for more
 * information.
 */
int ref_transaction_delete(struct ref_transaction *transaction,
			   const char *refname,
			   const struct object_id *old_oid,
			   const char *old_target,
			   unsigned int flags,
			   const char *msg,
			   struct strbuf *err);

/*
 * Verify, within a transaction, that refname has the value old_oid,
 * or, if old_oid is null_oid, then verify that the reference
 * doesn't exist. old_oid must be non-NULL.
 *
 * See the above comment "Reference transaction updates" for more
 * information.
 */
int ref_transaction_verify(struct ref_transaction *transaction,
			   const char *refname,
			   const struct object_id *old_oid,
			   const char *old_target,
			   unsigned int flags,
			   struct strbuf *err);

/*
 * Perform the preparatory stages of committing `transaction`. Acquire
 * any needed locks, check preconditions, etc.; basically, do as much
 * as possible to ensure that the transaction will be able to go
 * through, stopping just short of making any irrevocable or
 * user-visible changes. The updates that this function prepares can
 * be finished up by calling `ref_transaction_commit()` or rolled back
 * by calling `ref_transaction_abort()`.
 *
 * On success, return 0 and leave the transaction in "prepared" state.
 * On failure, abort the transaction, write an error message to `err`,
 * and return one of the `TRANSACTION_*` constants.
 *
 * Callers who don't need such fine-grained control over committing
 * reference transactions should just call `ref_transaction_commit()`.
 */
int ref_transaction_prepare(struct ref_transaction *transaction,
			    struct strbuf *err);

/*
 * Commit all of the changes that have been queued in transaction, as
 * atomically as possible. On success, return 0 and leave the
 * transaction in "closed" state. On failure, roll back the
 * transaction, write an error message to `err`, and return one of the
 * `TRANSACTION_*` constants
 */
int ref_transaction_commit(struct ref_transaction *transaction,
			   struct strbuf *err);

/*
 * Abort `transaction`, which has been begun and possibly prepared,
 * but not yet committed.
 */
int ref_transaction_abort(struct ref_transaction *transaction,
			  struct strbuf *err);

/*
 * Execute the given callback function for each of the reference updates which
 * have been queued in the given transaction. `old_oid` and `new_oid` may be
 * `NULL` pointers depending on whether the update has these object IDs set or
 * not.
 */
typedef void ref_transaction_for_each_queued_update_fn(const char *refname,
						       const struct object_id *old_oid,
						       const struct object_id *new_oid,
						       void *cb_data);
void ref_transaction_for_each_queued_update(struct ref_transaction *transaction,
					    ref_transaction_for_each_queued_update_fn cb,
					    void *cb_data);

/*
 * Execute the given callback function for each of the reference updates which
 * have been rejected in the given transaction.
 */
typedef void ref_transaction_for_each_rejected_update_fn(const char *refname,
							 const struct object_id *old_oid,
							 const struct object_id *new_oid,
							 const char *old_target,
							 const char *new_target,
							 enum ref_transaction_error err,
							 void *cb_data);
void ref_transaction_for_each_rejected_update(struct ref_transaction *transaction,
					      ref_transaction_for_each_rejected_update_fn cb,
					      void *cb_data);

/*
 * Translate errors to human readable error messages.
 */
const char *ref_transaction_error_msg(enum ref_transaction_error err);

/*
 * Free `*transaction` and all associated data.
 */
void ref_transaction_free(struct ref_transaction *transaction);

/**
 * Lock, update, and unlock a single reference. This function
 * basically does a transaction containing a single call to
 * ref_transaction_update(). The parameters to this function have the
 * same meaning as the corresponding parameters to
 * ref_transaction_update(). Handle errors as requested by the `onerr`
 * argument.
 */
int refs_update_ref(struct ref_store *refs, const char *msg, const char *refname,
		    const struct object_id *new_oid, const struct object_id *old_oid,
		    unsigned int flags, enum action_on_err onerr);

int parse_hide_refs_config(const char *var, const char *value, const char *,
			   struct strvec *);

/*
 * Check whether a ref is hidden. If no namespace is set, both the first and
 * the second parameter point to the full ref name. If a namespace is set and
 * the ref is inside that namespace, the first parameter is a pointer to the
 * name of the ref with the namespace prefix removed. If a namespace is set and
 * the ref is outside that namespace, the first parameter is NULL. The second
 * parameter always points to the full ref name.
 */
int ref_is_hidden(const char *, const char *, const struct strvec *);

/*
 * Returns an array of patterns to use as excluded_patterns, if none of the
 * hidden references use the token '!' or '^'.
 */
const char **hidden_refs_to_excludes(const struct strvec *hide_refs);

/*
 * Prefix all exclude patterns with the namespace, if any. This is required
 * because exclude patterns apply to the stripped reference name, not the full
 * reference name with the namespace.
 */
const char **get_namespaced_exclude_patterns(const char **exclude_patterns,
					     const char *namespace,
					     struct strvec *out);

/* Is this a per-worktree ref living in the refs/ namespace? */
int is_per_worktree_ref(const char *refname);

/* Describes how a refname relates to worktrees */
enum ref_worktree_type {
	REF_WORKTREE_CURRENT, /* implicitly per worktree, eg. HEAD or
				 refs/bisect/SOMETHING */
	REF_WORKTREE_MAIN, /* explicitly in main worktree, eg.
			      main-worktree/HEAD */
	REF_WORKTREE_OTHER, /* explicitly in named worktree, eg.
			       worktrees/bla/HEAD */
	REF_WORKTREE_SHARED, /* the default, eg. refs/heads/main */
};

/*
 * Parse a `maybe_worktree_ref` as a ref that possibly refers to a worktree ref
 * (ie. either REFNAME, main-worktree/REFNAME or worktree/WORKTREE/REFNAME). It
 * returns what kind of ref was found, and in case of REF_WORKTREE_OTHER, the
 * worktree name is returned in `worktree_name` (pointing into
 * `maybe_worktree_ref`) and `worktree_name_length`. The bare refname (the
 * refname stripped of prefixes) is returned in `bare_refname`. The
 * `worktree_name`, `worktree_name_length` and `bare_refname` arguments may be
 * NULL.
 */
enum ref_worktree_type parse_worktree_ref(const char *maybe_worktree_ref,
					  const char **worktree_name,
					  int *worktree_name_length,
					  const char **bare_refname);

enum expire_reflog_flags {
	EXPIRE_REFLOGS_DRY_RUN = 1 << 0,
	EXPIRE_REFLOGS_UPDATE_REF = 1 << 1,
	EXPIRE_REFLOGS_REWRITE = 1 << 2,
};

/*
 * The following interface is used for reflog expiration. The caller
 * calls refs_reflog_expire(), supplying it with three callback functions,
 * of the following types. The callback functions define the
 * expiration policy that is desired.
 *
 * reflog_expiry_prepare_fn -- Called once after the reference is
 *     locked. Called with the OID of the locked reference.
 *
 * reflog_expiry_should_prune_fn -- Called once for each entry in the
 *     existing reflog. It should return true iff that entry should be
 *     pruned.
 *
 * reflog_expiry_cleanup_fn -- Called once before the reference is
 *     unlocked again.
 */
typedef void reflog_expiry_prepare_fn(const char *refname,
				      const struct object_id *oid,
				      void *cb_data);
typedef int reflog_expiry_should_prune_fn(struct object_id *ooid,
					  struct object_id *noid,
					  const char *email,
					  timestamp_t timestamp, int tz,
					  const char *message, void *cb_data);
typedef void reflog_expiry_cleanup_fn(void *cb_data);

/*
 * Expire reflog entries for the specified reference.
 * flags is a combination of the constants in
 * enum expire_reflog_flags. The three function pointers are described
 * above. On success, return zero.
 */
int refs_reflog_expire(struct ref_store *refs,
		       const char *refname,
		       unsigned int flags,
		       reflog_expiry_prepare_fn prepare_fn,
		       reflog_expiry_should_prune_fn should_prune_fn,
		       reflog_expiry_cleanup_fn cleanup_fn,
		       void *policy_cb_data);

struct ref_store *get_main_ref_store(struct repository *r);

/**
 * Submodules
 * ----------
 *
 * If you want to iterate the refs of a submodule you first need to add the
 * submodules object database. You can do this by a code-snippet like
 * this:
 *
 * 	const char *path = "path/to/submodule"
 * 	if (add_submodule_odb(path))
 * 		die("Error submodule '%s' not populated.", path);
 *
 * `add_submodule_odb()` will return zero on success. If you
 * do not do this you will get an error for each ref that it does not point
 * to a valid object.
 *
 * Note: As a side-effect of this you cannot safely assume that all
 * objects you lookup are available in superproject. All submodule objects
 * will be available the same way as the superprojects objects.
 *
 * Example:
 * --------
 *
 * ----
 * static int handle_remote_ref(const char *refname,
 * 		const unsigned char *sha1, int flags, void *cb_data)
 * {
 * 	struct strbuf *output = cb_data;
 * 	strbuf_addf(output, "%s\n", refname);
 * 	return 0;
 * }
 *
 */

/*
 * Return the ref_store instance for the specified submodule. For the
 * main repository, use submodule==NULL; such a call cannot fail. For
 * a submodule, the submodule must exist and be a nonbare repository,
 * otherwise return NULL. If the requested reference store has not yet
 * been initialized, initialize it first.
 *
 * For backwards compatibility, submodule=="" is treated the same as
 * submodule==NULL.
 */
struct ref_store *repo_get_submodule_ref_store(struct repository *repo,
					       const char *submodule);
struct ref_store *get_worktree_ref_store(const struct worktree *wt);

/*
 * Some of the names specified by refs have special meaning to Git.
 * Organize these namespaces in a common 'ref_namespace' array for
 * reference from multiple places in the codebase.
 */

struct ref_namespace_info {
	const char *ref;
	enum decoration_type decoration;

	/*
	 * If 'exact' is true, then we must match the 'ref' exactly.
	 * Otherwise, use a prefix match.
	 *
	 * 'ref_updated' is for internal use. It represents whether the
	 * 'ref' value was replaced from its original literal version.
	 */
	unsigned exact:1,
		 ref_updated:1;
};

enum ref_namespace {
	NAMESPACE_HEAD,
	NAMESPACE_BRANCHES,
	NAMESPACE_TAGS,
	NAMESPACE_REMOTE_REFS,
	NAMESPACE_STASH,
	NAMESPACE_REPLACE,
	NAMESPACE_NOTES,
	NAMESPACE_PREFETCH,
	NAMESPACE_REWRITTEN,

	/* Must be last */
	NAMESPACE__COUNT
};

/* See refs.c for the contents of this array. */
extern struct ref_namespace_info ref_namespace[NAMESPACE__COUNT];

/*
 * Some ref namespaces can be modified by config values or environment
 * variables. Modify a namespace as specified by its ref_namespace key.
 */
void update_ref_namespace(enum ref_namespace namespace, char *ref);

/*
 * Check whether the provided name names a root reference. This function only
 * performs a syntactic check.
 *
 * A root ref is a reference that lives in the root of the reference hierarchy.
 * These references must conform to special syntax:
 *
 *   - Their name must be all-uppercase or underscores ("_").
 *
 *   - Their name must end with "_HEAD". As a special rule, "HEAD" is a root
 *     ref, as well.
 *
 *   - Their name may not contain a slash.
 *
 * There is a special set of irregular root refs that exist due to historic
 * reasons, only. This list shall not be expanded in the future:
 *
 *   - AUTO_MERGE
 *
 *   - BISECT_EXPECTED_REV
 *
 *   - NOTES_MERGE_PARTIAL
 *
 *   - NOTES_MERGE_REF
 *
 *   - MERGE_AUTOSTASH
 */
int is_root_ref(const char *refname);

/*
 * Pseudorefs are refs that have different semantics compared to
 * "normal" refs. These refs can thus not be stored in the ref backend,
 * but must always be accessed via the filesystem. The following refs
 * are pseudorefs:
 *
 * - FETCH_HEAD may contain multiple object IDs, and each one of them
 *   carries additional metadata like where it came from.
 *
 * - MERGE_HEAD may contain multiple object IDs when merging multiple
 *   heads.
 *
 * Reading, writing or deleting references must consistently go either
 * through the filesystem (pseudorefs) or through the reference
 * backend (normal ones).
 */
int is_pseudo_ref(const char *refname);

/*
 * The following flags can be passed to `repo_migrate_ref_storage_format()`:
 *
 *   - REPO_MIGRATE_REF_STORAGE_FORMAT_DRYRUN: perform a dry-run migration
 *     without touching the main repository. The result will be written into a
 *     temporary ref storage directory.
 *
 *   - REPO_MIGRATE_REF_STORAGE_FORMAT_SKIP_REFLOG: skip migration of reflogs.
 */
#define REPO_MIGRATE_REF_STORAGE_FORMAT_DRYRUN      (1 << 0)
#define REPO_MIGRATE_REF_STORAGE_FORMAT_SKIP_REFLOG (1 << 1)

/*
 * Migrate the ref storage format used by the repository to the
 * specified one.
 */
int repo_migrate_ref_storage_format(struct repository *repo,
				    enum ref_storage_format format,
				    unsigned int flags,
				    struct strbuf *err);

/*
 * Reference iterators
 *
 * A reference iterator encapsulates the state of an in-progress
 * iteration over references. Create an instance of `struct
 * ref_iterator` via one of the functions in this module.
 *
 * A freshly-created ref_iterator doesn't yet point at a reference. To
 * advance the iterator, call ref_iterator_advance(). If successful,
 * this sets the iterator's refname, oid, and flags fields to describe
 * the next reference and returns ITER_OK. The data pointed at by
 * refname and oid belong to the iterator; if you want to retain them
 * after calling ref_iterator_advance() again or calling
 * ref_iterator_free(), you must make a copy. When the iteration has
 * been exhausted, ref_iterator_advance() releases any resources
 * associated with the iteration, frees the ref_iterator object, and
 * returns ITER_DONE. If you want to abort the iteration early, call
 * ref_iterator_free(), which also frees the ref_iterator object and
 * any associated resources. If there was an internal error advancing
 * to the next entry, ref_iterator_advance() aborts the iteration,
 * frees the ref_iterator, and returns ITER_ERROR.
 *
 * Putting it all together, a typical iteration looks like this:
 *
 *     int ok;
 *     struct ref_iterator *iter = ...;
 *
 *     while ((ok = ref_iterator_advance(iter)) == ITER_OK) {
 *             if (want_to_stop_iteration()) {
 *                     ok = ITER_DONE;
 *                     break;
 *             }
 *
 *             // Access information about the current reference:
 *             if (!(iter->flags & REF_ISSYMREF))
 *                     printf("%s is %s\n", iter->refname, oid_to_hex(iter->oid));
 *     }
 *
 *     if (ok != ITER_DONE)
 *             handle_error();
 *     ref_iterator_free(iter);
 */
struct ref_iterator;

/*
 * These flags are passed to refs_ref_iterator_begin() (and do_for_each_ref(),
 * which feeds it).
 */
enum do_for_each_ref_flags {
	/*
	 * Include broken references in a do_for_each_ref*() iteration, which
	 * would normally be omitted. This includes both refs that point to
	 * missing objects (a true repository corruption), ones with illegal
	 * names (which we prefer not to expose to callers), as well as
	 * dangling symbolic refs (i.e., those that point to a non-existent
	 * ref; this is not a corruption, but as they have no valid oid, we
	 * omit them from normal iteration results).
	 */
	DO_FOR_EACH_INCLUDE_BROKEN = (1 << 0),

	/*
	 * Only include per-worktree refs in a do_for_each_ref*() iteration.
	 * Normally this will be used with a files ref_store, since that's
	 * where all reference backends will presumably store their
	 * per-worktree refs.
	 */
	DO_FOR_EACH_PER_WORKTREE_ONLY = (1 << 1),

	/*
	 * Omit dangling symrefs from output; this only has an effect with
	 * INCLUDE_BROKEN, since they are otherwise not included at all.
	 */
	DO_FOR_EACH_OMIT_DANGLING_SYMREFS = (1 << 2),

	/*
	 * Include root refs i.e. HEAD and pseudorefs along with the regular
	 * refs.
	 */
	DO_FOR_EACH_INCLUDE_ROOT_REFS = (1 << 3),
};

/*
 * Return an iterator that goes over each reference in `refs` for
 * which the refname begins with prefix. If trim is non-zero, then
 * trim that many characters off the beginning of each refname.
 * The output is ordered by refname.
 */
struct ref_iterator *refs_ref_iterator_begin(
	struct ref_store *refs,
	const char *prefix, const char **exclude_patterns,
	int trim, enum do_for_each_ref_flags flags);

/*
 * Advance the iterator to the first or next item and return ITER_OK.
 * If the iteration is exhausted, free the resources associated with
 * the ref_iterator and return ITER_DONE. On errors, free the iterator
 * resources and return ITER_ERROR. It is a bug to use ref_iterator or
 * call this function again after it has returned ITER_DONE or
 * ITER_ERROR.
 */
int ref_iterator_advance(struct ref_iterator *ref_iterator);

enum ref_iterator_seek_flag {
	/*
	 * When the REF_ITERATOR_SEEK_SET_PREFIX flag is set, the iterator's prefix is
	 * updated to match the provided string, affecting all subsequent iterations. If
	 * not, the iterator seeks to the specified reference and clears any previously
	 * set prefix.
	 */
	REF_ITERATOR_SEEK_SET_PREFIX = (1 << 0),
};

/*
 * Seek the iterator to the first reference matching the given seek string.
 * The seek string is matched as a literal string, without regard for path
 * separators. If seek is NULL or the empty string, seek the iterator to the
 * first reference again.
 *
 * This function is expected to behave as if a new ref iterator has been
 * created, but allows reuse of existing iterators for optimization.
 *
 * Returns 0 on success, a negative error code otherwise.
 */
int ref_iterator_seek(struct ref_iterator *ref_iterator, const char *refname,
		      unsigned int flags);

/* Free the reference iterator and any associated resources. */
void ref_iterator_free(struct ref_iterator *ref_iterator);

/*
 * The common backend for the for_each_*ref* functions. Call fn for
 * each reference in iter. If the iterator itself ever returns
 * ITER_ERROR, return -1. If fn ever returns a non-zero value, stop
 * the iteration and return that value. Otherwise, return 0. In any
 * case, free the iterator when done. This function is basically an
 * adapter between the callback style of reference iteration and the
 * iterator style.
 */
int do_for_each_ref_iterator(struct ref_iterator *iter,
			     each_ref_fn fn, void *cb_data);

#endif /* REFS_H */
